// Parse node for primitive steps. 

package abl.compiler;

import java.util.*;
import jd.*;
import abl.runtime.Step;
// import debug.Assert;

public class ASTPrimitiveStep extends ArgumentStep implements AblDebuggerConstants {
	
	// The primitive act name. 
	String actName = null; 
	
	ASTPrimitiveStep(int id) {
		super(id);
	}
	
	ASTPrimitiveStep(AblParser p, int id) {
		super(p, id);
	}

    // fixme: Will probably eventually need to be moved to a publically acessible utility class.
    private boolean assignable(String assignedFrom, String assignedTo) {
    	try {
    		if ((assignedFrom == null) &&
    				(!primitiveType(assignedTo)))
    			return true;
    		if (primitiveType(assignedFrom) && primitiveType(assignedTo) &&
    				assignedFrom.equals(assignedTo)) 
    			// This could be made more complex to take into account all the casting conversions.
    			return true;
    		if (!primitiveType(assignedFrom) && !primitiveType(assignedTo)) {
    			Class assignedFromClass = getClass(assignedFrom);
    			Class assignedToClass = getClass(assignedTo);
    			if (assignedToClass.isAssignableFrom(assignedFromClass))
    				return true;
    		}
    		return false;
    	}
    	catch (ClassNotFoundException e) {
    		throw new CompileError("Error compiling primitive step", e);
    	}
    }
	   
    /* Returns the name of the primitive action class registered on
       the primitive step. Throws an exception if no primitive action
       class is found or the primitive action class is ambiguous. */
    private String determineActionClass(List<AblArgument> stepArguments) throws CompileException {
    	List<RegisteredAction> registeredActions = ASTBehaviorUnit.getBehaviorUnit().lookupRegisteredAction(actName);
    	
    	/* The action which should be called as determined by the act
    	 name and arguments. Exactly one registered action should be
    	 found. */
    	List<String> actionToCall = new ArrayList<String>();
    	if (registeredActions != null) {
    		// The act name has been registered
    		Iterator<RegisteredAction> iter = registeredActions.iterator();
    		while (iter.hasNext()) {
    			RegisteredAction action = iter.next();
    			List<String> argTypes = action.argTypes;
    			if (argTypes.size() == stepArguments.size()) {
    				/* If the number of arguments of this primitive
    				 step is equal to the number of arguments of
    				 some registered action, the registered action
    				 potentially applies to this step. */
    				Iterator<String> registeredActionArgTypesIter = argTypes.iterator();
    				Iterator<AblArgument> primitiveStepArgIter = stepArguments.iterator();
    				boolean argumentsMatch = true;
    				while(primitiveStepArgIter.hasNext()) {
    					// Compare the types, argument by argument
    					String registeredActionArgType = registeredActionArgTypesIter.next().intern();
    					String primitiveStepArgType = primitiveStepArgIter.next().getType().intern();
    					if (!assignable(primitiveStepArgType, registeredActionArgType)) {
    						/* If an argument of the registered action
    						 is not assignable from argument type of
    						 the primitive step, set argumentsMatch
    						 to false and stop comparing the
    						 argument lists. */
    						argumentsMatch = false;
    						break;
    					}
    				}
    				if (argumentsMatch)
    					// If the arguments do match, add the registered action to the list. 
    					actionToCall.add(action.actionClass);
    			}
    		}
    	}
    	else 
    		// The act name hasn't been registered. Throw a compiler exception.
    		throw new CompileException("Line: " + getFirstLineNumber() + ". Attempt to call unregistered action " + actName);
    	if (actionToCall.size() == 0) 
    		throw new CompileException("Line: " + getFirstLineNumber() + ". No registered action found for primitive step " + actName);
    	else if (actionToCall.size() > 1)
    		throw new CompileException("Line: " + getFirstLineNumber() + ". Ambiguous registered actions found for primitive step " + actName); 
    	else
    		return actionToCall.get(0);
    	
    }

    private CodeBlockDescriptor compilePrimitiveStepExecute(List<AblArgument> processedArgs) 
    {
    	assert (processedArgs != null);
    	
    	if (processedArgs.size() == 0)
    		return null; // no arguments, so create no execute case
    	else {
    		// Add a line of the form args[i] = new <Type>(<argVal|variable ref>) or
    		// args[i] = <variable ref>; for each step argument. 
    		final CodeBlockDescriptor block = new CodeBlockDescriptor("case " + stepID + ": {", "}");
    		block.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));
    		
    		final int argCount = processedArgs.size();
    		block.addToBlockBody(new CodeStringDescriptor("final Object[] args = new Object[" + argCount + "];"));
    		final Iterator<AblArgument> iter = processedArgs.iterator();
    		int i = 0;
    		while(iter.hasNext()) {
    			final AblArgument arg = iter.next();
    			block.addToBlockBody(new CodeStringDescriptor("args[" + i++ + "] = " + arg + ";"));
    		}
    		block.addToBlockBody(new CodeStringDescriptor("return args;"));
    		return block;
    	}
    }

    private CodeBlockDescriptor compileStepFactory(List<AblArgument> processedArgs) throws CompileException
    {
    	CodeBlockDescriptor stepFactory = new CodeBlockDescriptor("case " + stepID + ": {", "}");
    	
    	stepFactory.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));
    	
    	final CodeSequenceDescriptor propertyTableInit = compilePropertyTable();
    	if (propertyTableInit != null)
    		stepFactory.addToBlockBody(propertyTableInit);
    	
    	final String conflictArrayName = compileConflictArray(actName);
    	String fullConflictArrayName;
    	if (conflictArrayName != null)
    		fullConflictArrayName = getBehavingEntityField(conflictArrayName);
    	else
    		fullConflictArrayName = conflictArrayName;
    	// fixme: look up syntax to make this work
    	// conflictArrayName != null ? conflictArrayName : "null"
    	
    	StringBuffer factoryBuf = new StringBuffer(1024);
    	
    	if (Abl.debugLevel == GUI_DEBUGGER)
    		// construct debug versions of steps
    		factoryBuf.append("return new PrimitiveStepDebug(");
    	else
    		factoryBuf.append("return new PrimitiveStep(");
    	
    	factoryBuf.append(standardConstructorArguments() + ", new " + determineActionClass(processedArgs) + "(), " + 
    			fullConflictArrayName + ", \"" + actName + "\"");
    	
    	factoryBuf.append(");");
    	
    	stepFactory.addToBlockBody(new CodeStringDescriptor(factoryBuf.toString()));
    	
    	return stepFactory;
    }

    final void compileToJava() throws CompileException 
    {
    	final ASTBehaviorUnit behaviorUnitNode = ASTBehaviorUnit.getBehaviorUnit();
    	
    	initStep(Step.PRIMITIVE);
    	
    	// Process the string arguments (currently stored as Tokens). 
    	final List<AblArgument> processedArgs = processArgs();
    	
    	CodeBlockDescriptor actExecute = compilePrimitiveStepExecute(processedArgs);
    	if (actExecute != null)
    		behaviorUnitNode.writeArgumentStepExecute(actExecute, this);
    	
    	if (hasSuccessTest)
    		compileSuccessTest();
    	
    	// fixme: when step sharing is implemented, writeStepFactory won't even be implemented. Will need to call
    	// some new method on ASTBehaviorDefinition to get the name of the factory reflection field. 
    	
    	// Compile the step factory at the end since the factory references the class and method names of the 
    	// other step parts (e.g. success test, sensor factories, execute).
    	behaviorUnitNode.writeStepFactory(compileStepFactory(processedArgs), this);
    	
    	// Add the step id and factory method to the parent's list of steps.
    	((ASTBehaviorDefinition)jjtGetParent()).addStep(stepID, getStepFactoryMethod_rField());
    }

}	
	

    
