/* Parse node for mental steps. */

package abl.compiler;

import jd.*;
import abl.runtime.*;

public class ASTMentalStep extends GenericStep implements AblDebuggerConstants {

    /* Since Java doesn't support multiple inheritance, the scope
       maintainer for mental steps is a member ratehr than a
       parent. */
    private AblScopeMaintainer stepScope;

    ASTMentalStep(int id) {
        super(id);
    }
    
    ASTMentalStep(AblParser p, int id) {
        super(p, id);
    }
    
    private CodeBlockDescriptor compileMentalStepExecute() throws ScopeException
    {
        // handle variable scope setup first
        stepScope = new AblScopeMaintainer(id);
        stepScope.setScopeParent((AblScopeMaintainer)jjtGetParent());
        stepScope.setScopeName(stepClassName);
        stepScope.setScopeType(AblScopeMaintainer.METHOD_SCOPE);
        
        // handle code execution setup next
        CodeBlockDescriptor execute = new CodeBlockDescriptor("case " + stepID + ": {", "}");
        execute.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));
        
        
        AblParseNode n;
        for(int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through child nodes, looking for variable declarations and JavaStatements
            // Note: Only variable declarations at this top-level will be seen...
            
            n = (AblParseNode)jjtGetChild(i);
            if (n.id == JJTABLVARIABLEDECL) {
                // If the child node is a variable declaration, process it
                
                FieldDescriptor fields[]; 
                fields = ((ASTAblVariableDecl)n).getFieldDescriptors(stepScope);
				for(int x = 0; x < fields.length; x++) {
					stepScope.addVariableDeclaration(fields[x]);
					// Add the declared variables to the execute() method.
					execute.addToBlockBody(fields[x]);
				}
            }
            if (n.id == JJTJAVASTATEMENT) {
                // If the child node is a JavaStatement, add it to the method.
                execute.addToBlockBody(new CodeStringDescriptor(((ASTJavaStatement)n).processVariableReferences(stepScope)));
            }
        }
        
        // FIXME: This shouldn't be added if the last thing in a mental step is
        // also a non-local flow-control statement (like throwing an exception).
        execute.addToBlockBody(new CodeStringDescriptor("break;"));
        
        return execute;
    }
    
    private CodeBlockDescriptor compileStepFactory() throws CompileException
    {
        CodeBlockDescriptor stepFactory = new CodeBlockDescriptor("case " + stepID + ": {", "}");
        
        stepFactory.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));
        
        final CodeSequenceDescriptor propertyTableInit = compilePropertyTable();
        if (propertyTableInit != null)
            stepFactory.addToBlockBody(propertyTableInit);
        
        if (Abl.debugLevel != NO_DEBUG)
            // construct debug versions of steps
            stepFactory.addToBlockBody(new CodeStringDescriptor("return new MentalStepDebug(" + 
								    standardConstructorArguments() + ", (byte)" + 
                                    Abl.debugLevel + ", \"" + getUniqueName() + "\");"));
        else
            stepFactory.addToBlockBody(new CodeStringDescriptor("return new MentalStep(" + 
								    standardConstructorArguments() + ");"));
        
        return stepFactory;
    }
    
    @Override
    void compileToJava() throws CompileException 
    {
        final ASTBehaviorUnit behaviorUnitNode = ASTBehaviorUnit.getBehaviorUnit();
        final ASTBehaviorDefinition behaviorParent = (ASTBehaviorDefinition)jjtGetParent();
        
        initStep(Step.MENTAL);
        
        behaviorUnitNode.writeMentalStepExecute(compileMentalStepExecute(), this);
        
        if (hasSuccessTest) {
            compileSuccessTest();
        }
        
        // Compile the step factory at the end since the factory references the class and method names of the 
        // other step parts (e.g. success test, sensor factories, execute).
        behaviorUnitNode.writeStepFactory(compileStepFactory(), this);
        
        // Add the step id and factory method to the parent's list of steps.
        behaviorParent.addStep(stepID, getStepFactoryMethod_rField());
    }
    
}
