/* Parse node for Java names. Java names can include "." (unlike
   AblNames). This node has been defined to allow the retrieval of
   java names via dumpTokens() (defined on AblParseNode). */

package abl.compiler;
// import debug.*;
import java.lang.reflect.*;

public class ASTJavaName extends AblParseNode {

    final static String getReflectionWME = "getReflectionWME";

    ASTJavaName(int id) {
        super(id);
    }

    ASTJavaName(AblParser p, int id) {
        super(p, id);
    }

    // Given a AblScopeMaintainer, returns the scope ID for the variable represented by this JavaName.
    int lookupVariableScope(AblScopeMaintainer scope) {
        String variableName = dumpTokens(0, 1);
        return scope.lookupVariableScope(variableName);
    }

    // Given a AblScopeMaintainer, returns the type of the variable represented by this JavaName. 
    String lookupVariableType(AblScopeMaintainer scope) {
        String variableName = dumpTokens(0, 1);
        return scope.lookupVariableType(variableName);
    }

    private int getBehaviorFrameIndex(AblScopeMaintainer scope) {
        assert(lookupVariableScope(scope) == AblScopeMaintainer.BEHAVIOR_SCOPE);
        return scope.getBehaviorFrameIndex(dumpTokens(0, 1));
    }

    // fixme: remove - never used locally - why do we need this?
    /* private int getBehaviorFrameDepth(AblScopeMaintainer scope) {
        assert(lookupVariableScope(scope) == AblScopeMaintainer.BEHAVIOR_SCOPE);
        return scope.getBehaviorFrameDepth(dumpTokens(0, 1), 0);
    } */
    
    String getName() {
        return dumpTokens(0, 1);
    }

    String getFullName() {
        return dumpTokens();
    }

    // Given a non-simple ref (e.g. x.y.z) returns the rest of the ref (eg. .y.z)
    private String getRestOfName() {
        return dumpTokens(1, numberOfTokens());
    }

    // Returns true if this java name is a simple references (an identifier), false otherwise
    boolean isSimpleRef() {
        return getName().equals(getFullName());
    }

    // If the variable reference represented by this JavaName contains
    // a field reference (e.g. x.y), returns the type of the last
    // field reference.
    String lookupFieldType(AblScopeMaintainer scope) throws CompileException {
        String type = lookupVariableType(scope);

        if (numberOfTokens() == 1)
            // No field reference
            return type;
        else {
            // There is a field reference

            int numTok = numberOfTokens();
            assert((numTok % 2) == 1); // number of tokens should be odd
            int tokenCounter = 2;
            while (tokenCounter < numTok) {
                // fixme: check if (tokenCounter, numTok) is correct. Seems like it should be 
                // (tokenCounter, tokenCounter + 1)
                String currentField = dumpTokens(tokenCounter, numTok);
                try {
                    type = getFieldType(type, currentField).getName();
                    tokenCounter = tokenCounter + 2;
                } catch (ClassNotFoundException e) {
                    throw new CompileException("Class not found in variable reference. " + e.getMessage());
                } catch (NoSuchFieldException e) {
                    throw new CompileException("No such field found in variable reference. " + e.getMessage());
                }
            }
            return type;
        }
    }

    // Returns true if the JavaName is a declared variable. 
    boolean isVariableReference(AblScopeMaintainer scope) {
        int scopeType = lookupVariableScope(scope);
        if (scopeType == AblScopeMaintainer.NO_SCOPE)
            return false;
        else
            return true;
    }

    // Returns a Method object if the JavaName is the name of a static method, null otherwise. 
    Method getMethod() throws CompileException {
        // If the JavaName contains only 1 token (not a name of the form x.y.z) return null
        if (numberOfTokens() == 1)
            return null;
        else {
            try {
                int numTok = numberOfTokens();
                assert((numTok % 2) == 1); // number of tokens should be odd
                Class type = getClass(dumpTokens(0, 1));
                String typeName = type.getName();
                int tokenCounter = 2;
                // iterate through field references through the next-to-last reference 
                // (last reference should be the static method reference)
                while (tokenCounter < numTok - 1) {
                    type = getFieldType(typeName, dumpTokens(tokenCounter, tokenCounter + 1));
                    typeName = type.getName();
                    tokenCounter = tokenCounter + 2;
                }
                // the last reference should be the static method
                return type.getMethod(dumpTokens(numTok - 1, numTok), (Class[])null);
            } catch (ClassNotFoundException e) {
                throw new CompileException(getFirstLineNumber(), "Class not found in method reference. " + e.getMessage());
            } catch (NoSuchFieldException e) {
                throw new CompileException(getFirstLineNumber(), "Field not found in method reference. " + e.getMessage());
            } catch (NoSuchMethodException e) {
                return null; // The last reference in the name is not a method reference
            }
        }
    }

    // Returns the scope reference for the variable represented by this JavaName. 
    String getVariableReference(AblScopeMaintainer scope) throws ScopeException {
        int scopeType = lookupVariableScope(scope);

        switch (scopeType) {
            case AblScopeMaintainer.GLOBAL_SCOPE :
                // Return a global scope variable reference
                String behavingEntityClassName = ASTBehaviorUnit.getBehaviorUnit().getBehavingEntityClass();
                return "((" + behavingEntityClassName + ")__$thisEntity)." + getFullName();
            case AblScopeMaintainer.BEHAVIOR_SCOPE :
                // Return a behavior scope reference

                int behaviorFrameIndex = getBehaviorFrameIndex(scope);

                // fixme: remove
            	// We flatten nested behavior frames instead of nesting them in the first position. 
            	/* 
                // the depth refers to the nesting capability of anonymous blocks
                // which can refer to variables in the calling behavior's scope
                // we resolve that scope by chasing a chain of behaviorFrame entries
                int behaviorFrameDepth = getBehaviorFrameDepth(scope);
                String frameRef = new String("__$behaviorFrame");
                for (int n = 0; n < behaviorFrameDepth; ++n)
                    frameRef = "((Object[])" + frameRef + "[0])";
                */

            	final String frameRef = new String("__$behaviorFrame");
            	
                String varType = lookupVariableType(scope);
                if (primitiveType(varType)) {
                    if (varType.equals("int"))
                        return "((__ValueTypes.IntVar)"+frameRef+"[" + behaviorFrameIndex + "]).i";
                    else if (varType.equals("float"))
                        return "((__ValueTypes.FloatVar)"+frameRef+"[" + behaviorFrameIndex + "]).f";
                    else if (varType.equals("char"))
                        return "((__ValueTypes.CharVar)"+frameRef+"[" + behaviorFrameIndex + "]).c";
                    else if (varType.equals("boolean"))
                        return "((__ValueTypes.BooleanVar)"+frameRef+"[" + behaviorFrameIndex + "]).b";
                    else if (varType.equals("long"))
                        return "((__ValueTypes.LongVar)"+frameRef+"[" + behaviorFrameIndex + "]).l";
                    else if (varType.equals("short"))
                        return "((__ValueTypes.ShortVar)"+frameRef+"[" + behaviorFrameIndex + "]).s";
                    else if (varType.equals("byte"))
                        return "((__ValueTypes.ByteVar)"+frameRef+"[" + behaviorFrameIndex + "]).b";
                    else if (varType.equals("double"))
                        return "((__ValueTypes.DoubleVar)"+frameRef+"[" + behaviorFrameIndex + "]).d";
                    else
                        throw new CompileError("Unexpected primitive type " + varType);
                } else {
                    if (isSimpleRef()) {
                        if (isLValue())
                            return frameRef+"[" + behaviorFrameIndex + "]";
                        else
                            return "((" + varType + ")"+frameRef+"[" + behaviorFrameIndex + "])";
                    } else {
                        return "((" + varType + ")"+frameRef+"[" + behaviorFrameIndex + "])" + getRestOfName();
                    }
                }
            case AblScopeMaintainer.METHOD_SCOPE :
                // Return a local (method) scope reference
                return getFullName();
            default :
                // The argument is in neither the global, behavior, or local scope
                throw new ScopeException(getFullName());
        }
    }

    // Returns a string representing the value of the variable as an Object. If the variable is of non-primitive type, then this just returns 
    // getVariableReference. But if the variable is of primitive type, it returns a string representation of the wrapped value
    // (e.g. new Integer(((Foo)parent).bar) ).
    String getVariableReferenceAsObject(AblScopeMaintainer scope) throws ScopeException {
        String variableRefString = getVariableReference(scope);
        String variableType = lookupVariableType(scope);

        // fixme: use macros
        if (variableType.equals("int"))
            return "new Integer(" + variableRefString + ")";
        if (variableType.equals("long"))
            return "new Long(" + variableRefString + ")";
        if (variableType.equals("short"))
            return "new Short(" + variableRefString + ")";
        else if (variableType.equals("float"))
            return "new Float(" + variableRefString + ")";
        else if (variableType.equals("double"))
            return "new Double(" + variableRefString + ")";
        else if (variableType.equals("char"))
            return "new Character(" + variableRefString + ")";
        else if (variableType.equals("boolean"))
            return "new Boolean(" + variableRefString + ")";
        else
            // Not a primitive type, just return the string representation of the variable reference
            return variableRefString;
    }

    // If the variable represented by this JavaName is in scope, returns the scope reference, otherwise treats 
    // JavaName as a constant reference and returns the constant value (as a string). If the JavaName is neither
    // in scope nor a constant, throws a compile exception. 
    String getVariableReferenceOrConstant(AblScopeMaintainer scope) throws CompileException {
        try {
            return getVariableReference(scope);
        } catch (ScopeException e1) {
            String variableName = dumpTokens();
            Field constant = ASTBehaviorUnit.getBehaviorUnit().getDeclaredConstant(variableName);
            if (constant != null) {
                String constantType = constant.getType().getName();
                Object value;
                try {
                    value = constant.get(null);
                } catch (IllegalAccessException e2) {
                    throw new CompileException(getFirstLineNumber(), "Illegal access exception while processing constant " + variableName);
                }
                if (!(constantType.equals("int") || constantType.equals("float") || constantType.equals("double") || constantType.equals("char") || constantType.equals("String") || constantType.equals("boolean")))
                    throw new CompileException(getFirstLineNumber(), "Constant " + variableName + " of type " + constantType + " has an unrecognized type.");
                if (constantType.equals("String"))
                    return "\"" + value.toString() + "\"";
                else
                    return value.toString();
            } else
                throw new CompileException(getFirstLineNumber(), "The variable " + variableName + " is not defined in any scope nor is it a constant.");
        }
    }

    // Returns a string representing the value of the variable or
    // constant as an Object. If the variable is of non-primitive
    // type, then this just returns getVariableReference(). But if the
    // variable is of primitive type, or this ASTJavaName refers to a
    // constant, it returns a string representation of the wrapped
    // value (e.g. new Integer(((Foo)parent).bar) ).
    String getVariableReferenceOrConstantAsObject(AblScopeMaintainer scope) throws CompileException {
        try {
            return getVariableReferenceAsObject(scope);
        } catch (ScopeException e1) {
            String variableName = dumpTokens();
            Field constant = ASTBehaviorUnit.getBehaviorUnit().getDeclaredConstant(variableName);
            if (constant != null) {
                String constantType = constant.getType().getName();
                Object value;
                try {
                    value = constant.get(null);
                } catch (IllegalAccessException e2) {
                    throw new CompileException(getFirstLineNumber(), "Illegal access exception while processing constant " + variableName);
                }
                if (constantType.equals("int"))
                    return "new Integer(" + value + ")";
                else if (constantType.equals("float"))
                    return "new Float(" + value + ")";
                else if (constantType.equals("double"))
                	return "new Double(" + value + ")";
                else if (constantType.equals("char"))
                    return "new Character(" + value + ")";
                else if (constantType.equals("boolean"))
                    return "new Boolean(" + value + ")";
                else if (constantType.equals("String"))
                    return "new String(\"" + value + "\")";
                else
                    throw new CompileException(getFirstLineNumber(), "Constant " + variableName + " of type " + constantType + " has an unrecognized type.");
            } else
                throw new CompileException(getFirstLineNumber(), "The variable " + variableName + " is not defined in any scope nor is it a constant.");
        }
    }

    boolean isConstant(AblScopeMaintainer scope) throws CompileException {
        if (isVariableReference(scope))
            return false;
        else {
            String constantName = dumpTokens();
            if (ASTBehaviorUnit.getBehaviorUnit().getDeclaredConstant(constantName) != null)
                return true;
            else
                return false;
        }
    }

    // If the JavaName is a constant, returns the constant's value as a wrapper object (e.g. Integer, Float, ...). 
    // If the JavaName is not a constant, throws a CompileException.
    Object getConstantValue() throws CompileException {
        String variableName = dumpTokens();
        Field constant = ASTBehaviorUnit.getBehaviorUnit().getDeclaredConstant(variableName);
        if (constant != null) {
            String constantType = constant.getType().getName();
            Object value;
            try {
                value = constant.get(null);
            } catch (IllegalAccessException e) {
                throw new CompileException(getFirstLineNumber(), "Illegal access exception while processing constant " + variableName);
            }
            if (!(constantType.equals("int") || constantType.equals("float") || constantType.equals("double") || constantType.equals("char") || constantType.equals("String") || constantType.equals("boolean")))
                throw new CompileException(getFirstLineNumber(), "Constant " + variableName + " of type " + constantType + " has an unrecognized type.");
            return value;
        } else
            throw new CompileException(getFirstLineNumber(), "The variable " + variableName + " is not defined in any scope nor is it a constant.");
    }

    // if JavaName is a constant, returns the type name of the constant
    String getConstantType() throws CompileException {
        String constantName = dumpTokens();
        Field constant = ASTBehaviorUnit.getBehaviorUnit().getDeclaredConstant(constantName);
        if (constant != null)
            return constant.getType().getName();
        else
            throw new CompileException(getFirstLineNumber(), "The constant " + constantName + " is not defined");
    }

    String getConstantString() throws CompileException {
        return getConstantValue().toString();
    }

    /* Replaces the list of tokens between firstToken and lastToken
       with a single token containing the image. */
    void setJavaNameImage(String image) {
        firstToken.image = image;
        firstToken.kind = IDENTIFIER;
        firstToken.next = lastToken.next;
        lastToken = firstToken;
    }

    // returns the type name of the ASTJavaName
    String getType(AblScopeMaintainer scope) throws CompileException {
        if (isVariableReference(scope))
            return lookupFieldType(scope);
        else if (isConstant(scope))
            return getConstantType();
        else
            throw new CompileException(getFirstLineNumber(), "The expression " + dumpTokens(0, 1) + " is neither a declared variable nor constant");
    }

    // Returns true if this java name is an lvalue. Examines the token immediately following the last token of the
    // java name to see if it is an assignment operator. 
    boolean isLValue() {
        final int nt = lastToken.next.kind;
        final AblParseNode parent2 = (AblParseNode) jjtGetParent().jjtGetParent();
        if (nt == ASSIGN || nt == PLUSASSIGN || nt == MINUSASSIGN || nt == STARASSIGN || nt == SLASHASSIGN || nt == ANDASSIGN || nt == ORASSIGN || nt == XORASSIGN || nt == REMASSIGN || nt == LSHIFTASSIGN || nt == RSIGNEDSHIFTASSIGN || nt == RUNSIGNEDSHIFTASSIGN || ((parent2.id == JJTWMEFIELDTEST) && (((ASTWMEFieldTest) parent2).testOp == V_BIND)))
            return true;
        else
            return false;
    }

    // Rewrites references to special abl methods (e.g. getReflectionWME()).
    void rewriteSpecialMethods() {
        if ((firstToken == lastToken) && (lastToken.next.kind == LPAREN))
            // Method invocation - could be a special method
            if (firstToken.image.equals(getReflectionWME)) {
                firstToken.image = "__$thisStep." + getReflectionWME;
            }
    }
}
