package abl.compiler;

import jd.*;
import java.util.*;
import abl.runtime.Step;
import abl.runtime.GoalStep;

/**
 * Parse node for a goal step.
 *
 */
public class ASTGoalStep extends ArgumentStep implements AblDebuggerConstants {

    /** The name of the goal referenced in this subgoal. */
    String goalName = null;

    /** True if all participating members in a joint goal are needed for success, false otherwise. */
    private boolean teamNeededForSuccess = false;
    
    /** Set to true teamNeededForSuccess is explicitly set by the compiler (meaning that a success annotation was found). */
    private boolean teamSuccessModifierExplicitlySet = false;

    /**
     * List of AblArguments computed by {@link ArgumentStep#processArgs()}.
     * Used by getSignature() to get the signature string and by
     * compileGoalStepExecute() to compile the argument initializers.
     */
    private List<AblArgument> processedArgs; 

    /** Flag indicating whether this goal step is a subgoal or spawngoal step - defaults to subgoal. */
    private short goalExecutionType = GoalStep.SUBGOAL;

    /** True if this goal step is a joint goal, false otherwise */
    private boolean isJointGoal = false;

    /** True if this goal step is to an anonymous-block behavior */
    private boolean isNestedScope = false;

    /** The parent at which to reroot at spawngoal  */
    private ASTJavaName rerootParent = null; 

    ASTGoalStep(int id) {
		super(id);
    }
    
    ASTGoalStep(AblParser p, int id) {
		super(p, id);
    }

    /** Constructs a new goal step with step modifiers that are a copy of the step modifiers in GenericStep g. */
    ASTGoalStep(int id, GenericStep g) {
    	super(id, g);
    }
    
    // accessors for goalType.
    void setGoalExecutionType(short goalExecutionType) { this.goalExecutionType = goalExecutionType; }
    int getGoalExecutionType() { return goalExecutionType; }

    // accessors for isJointGoal.
    void setJointGoal() { isJointGoal = true; }
    boolean isJointGoal() { return isJointGoal; }

    void setNestedScope(boolean isNested) { isNestedScope = isNested; }
    boolean isNestedScope() { return isNestedScope; }

    // accessors for teamNeededForSuccess.
    void setTeamNeededForSuccess(boolean b) 
    { 
		teamNeededForSuccess = b; 
		teamSuccessModifierExplicitlySet = true;
    }

    boolean getTeamNeededForSuccess() { return teamNeededForSuccess; }

    // Returns the goal signature.
    String getSignature() 
    {
		assert (processedArgs != null);

		StringBuffer signatureBuf = new StringBuffer();
		signatureBuf.append(goalName + "("); 

		final Iterator<AblArgument> iter = processedArgs.iterator();
		while(iter.hasNext()) {
			// fixme: returning only the short name of the type - eventually all types should internally be 
			// represented by long name.
			String type = iter.next().getType();
			signatureBuf.append(type.substring(type.lastIndexOf('.') + 1) + ", ");
		}
		
		// Delete the final ", ".
		if (processedArgs.size() != 0)
			signatureBuf.delete(signatureBuf.length() - 2, signatureBuf.length());
		
		signatureBuf.append(")");
		
		return signatureBuf.toString().intern();
		
    }

    // accessors for rerootParent 
    void setRerootParent(ASTJavaName p) { rerootParent = p; }

    private CodeBlockDescriptor compileGoalStepExecute() throws CompileException
    {
        // create block, it will get thrown away if not used
        final CodeBlockDescriptor block = new CodeBlockDescriptor("case " + stepID + ": {", "}");
        block.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));

		assert (processedArgs != null); 
		if (processedArgs.size() == 0 && rerootParent == null)
        {
            if (!isNestedScope)
                return null; // no arguments, so create no execute case
            else {
            	// Thread the enclosing behaviorFrame as the arguments of this step
            	block.addToBlockBody(new CodeStringDescriptor("final Object[] args = __$behaviorFrame;"));
            }
        }
		else {
			// Add a line of the form args[i] = new <Type>(<argVal|variable ref>) or
			// args[i] = <variable ref>; for each step argument. 
			// If rerootParent != null, add the reroot parent as the first arg


			int argCount;
			if (rerootParent == null)
                argCount = processedArgs.size();
            else
                argCount = processedArgs.size() + 1;
			
			block.addToBlockBody(new CodeStringDescriptor("final Object[] args = new Object[" + argCount + "];"));

			if (rerootParent != null) {
			    AblScopeMaintainer scope = (ASTBehaviorDefinition)jjtGetParent();
			    String rerootParentTypeName = rerootParent.lookupVariableType(scope);
			    if (rerootParent.isVariableReference(scope)) {
				    if (rerootParentTypeName.equals("MultiStepBehaviorWME") ||
				        rerootParentTypeName.equals("CollectionBehaviorWME") ||
				        rerootParentTypeName.equals("ParallelBehaviorWME") ||
						rerootParentTypeName.equals("abl.runtime.MultiStepBehaviorWME") ||
				        rerootParentTypeName.equals("abl.runtime.CollectionBehaviorWME") ||
				        rerootParentTypeName.equals("abl.runtime.ParallelBehaviorWME"))
                    {
				        // check if the type of rerootParent is MultiStepBehaviorWME
				        block.addToBlockBody(new CodeStringDescriptor("args[0] = " + 
										          rerootParent.getVariableReference(scope) + ";"));
				    }
				    else
				    // Illegal type
				    throw new CompileException(getFirstLineNumber(), "Attempt to spawn goal at parent " + rerootParent.dumpTokens() + " with type " 
							       + rerootParentTypeName + 
							       ". Reroot parent must be a MultiStepBehaviorWME (or subtype).");
			    }
			    else
				    // Not a variable reference
				    throw new CompileException(getFirstLineNumber(), "Spawngoal reroot parent " + rerootParent.dumpTokens() + " is not a declared variable");
			}
			
			final Iterator<AblArgument> iter = processedArgs.iterator();
			int i;
            if (rerootParent == null)
                i = 0;
            else
                i = 1;

            while(iter.hasNext()) {
                final AblArgument arg = iter.next();
                block.addToBlockBody(new CodeStringDescriptor("args[" + i++ + "] = " + arg + ";"));
            }
		}
        block.addToBlockBody(new CodeStringDescriptor("return args;"));
        return block;
    }

    private CodeBlockDescriptor compileStepFactory() throws CompileException
    {
		CodeBlockDescriptor stepFactory = new CodeBlockDescriptor("case " + stepID + ": {", "}");
		
		stepFactory.addToBlockBody(new CodeStringDescriptor("// " + getUniqueName()));

		final CodeSequenceDescriptor propertyTableInit = compilePropertyTable();
		if (propertyTableInit != null)
			stepFactory.addToBlockBody(propertyTableInit);

		final String conflictArrayName = compileConflictArray(goalName);
		String fullConflictArrayName;
		if (conflictArrayName != null)
			fullConflictArrayName = getBehavingEntityField(conflictArrayName);
		else
			fullConflictArrayName = conflictArrayName;
		// fixme: look up syntax to make this work
		// conflictArrayName != null ? conflictArrayName : "null"

		StringBuffer factoryBuf = new StringBuffer(1024);
		factoryBuf.append("return ");
		
		if (isJointGoal && Abl.debugLevel != NO_DEBUG)
			// JointGoalStepDebugs are created for any non-zero debug level
			factoryBuf.append("new JointGoalStepDebug(");
		else if (isJointGoal && Abl.debugLevel == NO_DEBUG)
			factoryBuf.append("new JointGoalStep(");
		else if (!isJointGoal && Abl.debugLevel == GUI_DEBUGGER)
			// GoalStepDebugs are created only for debug level 2
			factoryBuf.append("new GoalStepDebug(");
		else
			factoryBuf.append("new GoalStep(");
		
		factoryBuf.append(standardConstructorArguments() + ", " + "\"" + getSignature() + "\"" + ", " + 
				  fullConflictArrayName + ", (short)" + goalExecutionType);

        if (isJointGoal) {
            // add team-needed-for-success 
            if (teamSuccessModifierExplicitlySet)
                factoryBuf.append(", " + teamNeededForSuccess);
            else
                factoryBuf.append(", " + ASTBehaviorUnit.getBehaviorUnit().getTeamNeededForSuccess());
        }

		// Add the debug level argument to the JointGoalStepDebug
		if (isJointGoal && Abl.debugLevel != NO_DEBUG)
			factoryBuf.append(", (byte)" + Abl.debugLevel);

		factoryBuf.append(");");

		stepFactory.addToBlockBody(new CodeStringDescriptor(factoryBuf.toString()));

		return stepFactory;
    }

    // fixme: consolidate this code with the similar code in ASTPrimitiveStep
    final void compileToJava() throws CompileException 
    {
		final ASTBehaviorUnit behaviorUnitNode = ASTBehaviorUnit.getBehaviorUnit();
		final ASTBehaviorDefinition behaviorParent = (ASTBehaviorDefinition)jjtGetParent();

		initStep(Step.GOAL);
			
		// It's an error to specify a team success modifier on a non-joint goal
		if ( teamSuccessModifierExplicitlySet && !isJointGoal ) {
			if (teamNeededForSuccess)
			throw new CompileException(getFirstLineNumber(), 
						   "team_needed_for_success modifier declared on a non-joint goal");
			else
			throw new CompileException(getFirstLineNumber(), 
						   "one_needed_for_success modifier declared on a non-joint goal");
		}

		// Process the string arguments (currently stored as Tokens). 
		processedArgs = processArgs();

		CodeBlockDescriptor goalExecute = compileGoalStepExecute();
		// Only write the execute case if the goal step has arguments. 
		if (goalExecute != null)
			behaviorUnitNode.writeArgumentStepExecute(goalExecute, this);


		if (hasSuccessTest) {
			compileSuccessTest();
		}
			
		// fixme: when step sharing is implemented, writeStepFactory won't even be implemented. Will need to call
		// some new method on ASTBehaviorDefinition to get the name of the factory reflection field. 

		// Compile the step factory at the end since the factory references the class and method names of the 
		// other step parts (e.g. success test, sensor factories, execute).
		behaviorUnitNode.writeStepFactory(compileStepFactory(), this);

		// Add the step id and factory method to the parent's list of steps.
		behaviorParent.addStep(stepID, getStepFactoryMethod_rField());
    }

    public String getGoalName() {
        return goalName;
    }

    public List getProcessedArgs() {
        return processedArgs;
    }

}




