/* Parse node for behavior units. */

package abl.compiler;

import java.util.*;
import java.io.*;
import java.lang.reflect.*;
import jd.*;
import macro.*;

public class ASTBehaviorUnit
	extends AblScopeMaintainer
	implements MacroDefinitions, AblParserTreeConstants, AblDebuggerConstants {

	private final int BEHAVIOR_FACTORY_LIMIT = 225;
	private final int PRECONDITION_SENSOR_FACTORY_LIMIT = 1000;
	private final int CONTEXT_CONDITION_SENSOR_FACTORY_LIMIT = 1000;
	private final int SUCCESS_CONDITION_SENSOR_FACTORY_LIMIT = 1000;
	private final int SUCCESS_TEST_SENSOR_FACTORY_LIMIT = 1000;
	private final int REINFORCEMENT_TEST_SENSOR_FACTORY_LIMIT = 1000;
	private final int STATE_WME_SENSOR_FACTORY_LIMIT = 300;
	private final int CONTEXT_CONDITION_LIMIT = 300;
	private final int SUCCESS_CONDITION_LIMIT = 300;
	private final int SUCCESS_TEST_LIMIT = 300;
	private final int REINFORCEMENT_TEST_LIMIT = 300;
	private final int STATE_WME_LIMIT = 300;
	private final int PRECONDITION_LIMIT = 300;
	private final int STEP_FACTORY_LIMIT = 1000;
	private final int STEP_EXECUTE_LIMIT = 250;

	// Number of characters in a single conflict set init method 
	private final int INIT_CONFLICT_SET_METHOD_SIZE = 40000;

	final static int DEFAULT_SUCCEED_STEP_ID = -1;
	final static int DEFAULT_FAIL_STEP_ID = -2;
	final static int DEFAULT_WAIT_STEP_ID = -3;
	// INITIATED_JOINT_GOAL_STEP_ID = -4; this id is defined in runtime/InitiatedJointGoalStep.java - shouldn't appear 
	// in any generated method. InitiatedJointGoalStep.java has self-contained logic for execution. 

	// pseudo step type for default steps (no annotations).
	final static short DEFAULT_SUCCEED_STEP = -1;
	final static short DEFAULT_FAIL_STEP = -2;
	final static short DEFAULT_WAIT_STEP = -3;

	private final static CodeBlockDescriptor DEFAULT_WAIT_STEP_FACTORY;
	private final static CodeBlockDescriptor DEFAULT_FAIL_STEP_FACTORY;
	private final static CodeBlockDescriptor DEFAULT_SUCCEED_STEP_FACTORY;

	private final static CodeBlockDescriptor DEFAULT_WAIT_STEP_FACTORY_DEBUG;
	private final static CodeBlockDescriptor DEFAULT_FAIL_STEP_FACTORY_DEBUG;
	private final static CodeBlockDescriptor DEFAULT_SUCCEED_STEP_FACTORY_DEBUG;

	static {
		// Factories for default steps.
		DEFAULT_WAIT_STEP_FACTORY = new CodeBlockDescriptor("case " + DEFAULT_WAIT_STEP_ID + ": {", "}");
		DEFAULT_WAIT_STEP_FACTORY.addToBlockBody(new CodeStringDescriptor("// default wait step"));
		DEFAULT_WAIT_STEP_FACTORY.addToBlockBody(
			new CodeStringDescriptor(
				"return new WaitStep("
					+ DEFAULT_WAIT_STEP_ID
					+ ", __$behaviorParent, false, false, false, false, false, false, (short)-32768, (short)0, false, null, null, null, null, null);"));

		DEFAULT_WAIT_STEP_FACTORY_DEBUG = new CodeBlockDescriptor("case " + DEFAULT_WAIT_STEP_ID + ": {", "}");
		DEFAULT_WAIT_STEP_FACTORY_DEBUG.addToBlockBody(new CodeStringDescriptor("// default wait step"));
		DEFAULT_WAIT_STEP_FACTORY_DEBUG.addToBlockBody(
			new CodeStringDescriptor(
				"return new WaitStepDebug("
					+ DEFAULT_WAIT_STEP_ID
					+ ", __$behaviorParent, false, false, false, false, false, false, (short)-32768, (short)0, false, null, null, null, null, null);"));

		DEFAULT_FAIL_STEP_FACTORY = new CodeBlockDescriptor("case " + DEFAULT_FAIL_STEP_ID + ": {", "}");
		DEFAULT_FAIL_STEP_FACTORY.addToBlockBody(new CodeStringDescriptor("// default fail step"));
		DEFAULT_FAIL_STEP_FACTORY.addToBlockBody(
			new CodeStringDescriptor(
				"return new FailStep("
					+ DEFAULT_FAIL_STEP_ID
					+ ", __$behaviorParent, false, false, false, (short)-32768, (short)0, false, null, null);"));

		DEFAULT_FAIL_STEP_FACTORY_DEBUG = new CodeBlockDescriptor("case " + DEFAULT_FAIL_STEP_ID + ": {", "}");
		DEFAULT_FAIL_STEP_FACTORY_DEBUG.addToBlockBody(new CodeStringDescriptor("// default fail step"));
		DEFAULT_FAIL_STEP_FACTORY_DEBUG.addToBlockBody(
			new CodeStringDescriptor(
				"return new FailStepDebug("
					+ DEFAULT_FAIL_STEP_ID
					+ ", __$behaviorParent, false, false, false, (short)-32768, (short)0, false, null, null);"));

		DEFAULT_SUCCEED_STEP_FACTORY = new CodeBlockDescriptor("case " + DEFAULT_SUCCEED_STEP_ID + ": {", "}");
		DEFAULT_SUCCEED_STEP_FACTORY.addToBlockBody(new CodeStringDescriptor("// default succeed step"));
		DEFAULT_SUCCEED_STEP_FACTORY.addToBlockBody(
			new CodeStringDescriptor(
				"return new SucceedStep("
					+ DEFAULT_SUCCEED_STEP_ID
					+ ", __$behaviorParent, false, false, (short)-32768, (short)0, false, null, null);"));

		DEFAULT_SUCCEED_STEP_FACTORY_DEBUG = new CodeBlockDescriptor("case " + DEFAULT_SUCCEED_STEP_ID + ": {", "}");
		DEFAULT_SUCCEED_STEP_FACTORY_DEBUG.addToBlockBody(new CodeStringDescriptor("// default succeed step"));
		DEFAULT_SUCCEED_STEP_FACTORY_DEBUG.addToBlockBody(
			new CodeStringDescriptor(
				"return new SucceedStepDebug("
					+ DEFAULT_SUCCEED_STEP_ID
					+ ", __$behaviorParent, false, false, (short)-32768, (short)0, false, null, null);"));
	}

	protected static final String[] packageMods = { "final", "static" };
	protected static final String[] privateMods = { "private", "final", "static" };

	private class CodeStream {

		static final int CODE_TYPE_BEHAVIOR_FACTORY = 0;
		static final int CODE_TYPE_PRECONDITION_SENSOR_FACTORY = CODE_TYPE_BEHAVIOR_FACTORY + 1;
		static final int CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY = CODE_TYPE_PRECONDITION_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY = CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY = CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY = CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_STATE_WME_SENSOR_FACTORY = CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_CONTEXT_CONDITION = CODE_TYPE_STATE_WME_SENSOR_FACTORY + 1;
		static final int CODE_TYPE_SUCCESS_CONDITION = CODE_TYPE_CONTEXT_CONDITION + 1;
		static final int CODE_TYPE_SUCCESS_TEST = CODE_TYPE_SUCCESS_CONDITION + 1;
		static final int CODE_TYPE_REINFORCEMENT_TEST = CODE_TYPE_SUCCESS_TEST + 1;
		static final int CODE_TYPE_RL_POLICY = CODE_TYPE_REINFORCEMENT_TEST + 1;
		static final int CODE_TYPE_STATE_WME = CODE_TYPE_RL_POLICY + 1;
		static final int CODE_TYPE_PRECONDITION = CODE_TYPE_STATE_WME + 1;
		static final int CODE_TYPE_STEP_FACTORY = CODE_TYPE_PRECONDITION + 1;
		static final int CODE_TYPE_ARGUMENT_STEP_EXECUTE = CODE_TYPE_STEP_FACTORY + 1;
		static final int CODE_TYPE_MENTAL_STEP_EXECUTE = CODE_TYPE_ARGUMENT_STEP_EXECUTE + 1;

		private final PrintStream codeStream;
		private final int codeStreamType;
		private int methodCounter; // static code method counter
		private int caseCounter;
		// counts code block cases in the switch statement of the static code method
		private final int caseLimit;
		// the case limit at which a new static code method should be started
		// List of methods appearing in various code classes. Used to construct the static Method initializers in the code
		// classes.
		// Format of List: <code class DEFANGED_name> <method name> (length of list should be even)
		private final List<String> methods;

		CodeStream(String arg_codeStreamName, int arg_codeStreamType, int arg_caseLimit) {
			codeStreamType = arg_codeStreamType;
			methodCounter = -1;
			caseCounter = 0;
			caseLimit = arg_caseLimit;
			methods = new ArrayList<String>(1000);
			try {
			    File packageDir = getFullPackagePath(Abl.objectDirectory,behavingEntityPackage);
				codeStream = new PrintStream(new BufferedOutputStream(new FileOutputStream(
                                     new File(packageDir, arg_codeStreamName + ".java")),100000));
				codeStream.println("package " + behavingEntityPackage + ";");
				codeStream.println("");
				for (int i = 0; i < importPackages.length; i++)
					codeStream.println("import " + importPackages[i] + ";");
				printUserImports(codeStream);
				codeStream.print("public class " + arg_codeStreamName + " ");
				printUserConstantDeclarations(codeStream);
				codeStream.println("{");
				startCodeMethod();
			} catch (IOException e) {
				throw new CompileError("Error initing stream " + arg_codeStreamName, e);
			}
		}

		int getMethodCounter() {
			return methodCounter;
		}

		List getMethods() {
			return methods;
		}

		// Given a PrintStream and a codeStreamType, opens a new method.
		private void startCodeMethod() {
			methodCounter++;
			methods.add(getCodeClassName(codeStreamType));
			methods.add(getCodeMethodName(codeStreamType));
			switch (codeStreamType) {
				case CODE_TYPE_BEHAVIOR_FACTORY :
					if (Abl.debugLevel == GUI_DEBUGGER)
						codeStream.println(
							"   static public Behavior "
								+ getCodeMethodName(codeStreamType)
								+ "(int __$behaviorID, Object[] __$args, Hashtable __$boundVars, "
								+ "GoalStep __$parentGoal, String __$signature, __BehaviorDesc __$behaviorDesc) {");
					else
						codeStream.println(
							"   static public Behavior "
								+ getCodeMethodName(codeStreamType)
								+ "(int __$behaviorID, Object[] __$args, Hashtable __$boundVars, "
								+ "GoalStep __$parentGoal, String __$signature) {");
					break;
				case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$behaviorID) {");
					break;
				case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$behaviorID) {");
					break;
				case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$behaviorID) {");
					break;
				case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$stepID) {");
					break;
				case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID) {");
					break;
				case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
					codeStream.println(
						"   static public SensorActivation[] "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID) {");
					break;
				case CODE_TYPE_CONTEXT_CONDITION :
					codeStream.println(
						"   static public boolean "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$behaviorID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_SUCCESS_CONDITION :
					codeStream.println(
						"   static public boolean "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_SUCCESS_TEST :
					codeStream.println(
						"   static public boolean "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$stepID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_REINFORCEMENT_TEST :
					codeStream.println(
						"   static public double "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_RL_POLICY :
					codeStream.println(
						"   static public int "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID, Object prevLocation, int stepIndex, Object currLocation, double rewardValue, boolean isRewarding) {");
					break;
				case CODE_TYPE_STATE_WME :
					codeStream.println(
						"   static public Object "
						+ getCodeMethodName(codeStreamType)
						+ "(int __$behaviorID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_PRECONDITION :
					codeStream.println(
						"   static public boolean "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$behaviorID, Object[] __$args, Hashtable __$variableTable, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_STEP_FACTORY :
					codeStream.println(
						"   static public Step "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$stepID, Behavior __$behaviorParent, final Object[] __$behaviorFrame) {");
					break;
				case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
					codeStream.println(
						"   static public Object[] "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$stepID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity) {");
					break;
				case CODE_TYPE_MENTAL_STEP_EXECUTE :
					codeStream.println(
						"   static public void "
							+ getCodeMethodName(codeStreamType)
							+ "(int __$stepID, final Object[] __$behaviorFrame, final BehavingEntity __$thisEntity, MentalStep __$thisStep) {");
					break;
				default :
					throw new CompileError("Unexpected code type " + codeStreamType);
			}
			switch (codeStreamType) {
				case CODE_TYPE_BEHAVIOR_FACTORY :
				case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
				case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
				case CODE_TYPE_CONTEXT_CONDITION :
				case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
				case CODE_TYPE_SUCCESS_CONDITION :
				case CODE_TYPE_REINFORCEMENT_TEST :
				case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
				case CODE_TYPE_STATE_WME :
				case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
				case CODE_TYPE_PRECONDITION :
					codeStream.println("      switch (__$behaviorID) {");
					break;
				case CODE_TYPE_STEP_FACTORY :
				case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
				case CODE_TYPE_MENTAL_STEP_EXECUTE :
				case CODE_TYPE_SUCCESS_TEST :
				case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					codeStream.println("      switch (__$stepID) {");
					break;
				case CODE_TYPE_RL_POLICY:
					codeStream.println("      switch (__$behaviorID) {");
					break;
				default :
					throw new CompileError("Unexpected code type " + codeStreamType);
			}
		}

		// fixme: may need to additionally create new clases in addition to new methods
		void writeCode(CodeBlockDescriptor code, AblParseNode node) {
			assert(code != null);

			if (node != null) {
				switch (codeStreamType) {
					case CODE_TYPE_BEHAVIOR_FACTORY :
						((ASTBehaviorDefinition) node).setBehaviorFactoryClass(
							getCodeClassName(CODE_TYPE_BEHAVIOR_FACTORY));
						((ASTBehaviorDefinition) node).setBehaviorFactoryMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_BEHAVIOR_FACTORY)));
						break;
					case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
						((ASTBehaviorDefinition) node).setPreconditionSensorFactoryClass(
							getCodeClassName(CODE_TYPE_PRECONDITION_SENSOR_FACTORY));
						((ASTBehaviorDefinition) node).setPreconditionSensorFactoryMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_PRECONDITION_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
						((ASTBehaviorDefinition) node).setContextConditionSensorFactoryClass(
							getCodeClassName(CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY));
						((ASTBehaviorDefinition) node).setContextConditionSensorFactoryMethod_rField(
							getCodeMethodReflectionField(
								getCodeMethodName(CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
						((ASTBehaviorDefinition) node).setSuccessConditionSensorFactoryClass(
							getCodeClassName(CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY));
						((ASTBehaviorDefinition) node).setSuccessConditionSensorFactoryMethod_rField(
							getCodeMethodReflectionField(
							getCodeMethodName(CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
						((GenericStep) node).setSuccessTestSensorFactoryMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
						((ASTBehaviorDefinition) node).setReinforcementSignalsSensorFactoryClass(
							getCodeClassName(CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY));
						((ASTBehaviorDefinition) node).setReinforcementSignalsSensorFactoryMethod_rField(
							getCodeMethodReflectionField(
							getCodeMethodName(CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
						((ASTBehaviorDefinition) node).setStateWMESensorFactoryClass(
							getCodeClassName(CODE_TYPE_STATE_WME_SENSOR_FACTORY));
						((ASTBehaviorDefinition) node).setStateWMESensorFactoryMethod_rField(
							getCodeMethodReflectionField(
							getCodeMethodName(CODE_TYPE_STATE_WME_SENSOR_FACTORY)));
						break;
					case CODE_TYPE_CONTEXT_CONDITION :
						((ASTBehaviorDefinition) node).setContextConditionClass(
							getCodeClassName(CODE_TYPE_CONTEXT_CONDITION));
						((ASTBehaviorDefinition) node).setContextConditionMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_CONTEXT_CONDITION)));
						break;
					case CODE_TYPE_SUCCESS_CONDITION :
						((ASTBehaviorDefinition) node).setSuccessConditionClass(
							getCodeClassName(CODE_TYPE_SUCCESS_CONDITION));
						((ASTBehaviorDefinition) node).setSuccessConditionMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_SUCCESS_CONDITION)));
						break;
					case CODE_TYPE_SUCCESS_TEST :
						((GenericStep) node).setSuccessTestMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_SUCCESS_TEST)));
						break;
					case CODE_TYPE_REINFORCEMENT_TEST :
						((ASTBehaviorDefinition) node).setReinforcementSignalsClass(
							getCodeClassName(CODE_TYPE_REINFORCEMENT_TEST));
						((ASTBehaviorDefinition) node).setReinforcementSignalsMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_REINFORCEMENT_TEST)));
						break;
					case CODE_TYPE_RL_POLICY :
						((ASTBehaviorDefinition) node).setRLPolicyClass(
							getCodeClassName(CODE_TYPE_RL_POLICY));
						((ASTBehaviorDefinition) node).setRLPolicyMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_RL_POLICY)));
						break;
					case CODE_TYPE_STATE_WME :
						// fixme: remove - the private field stateWMEClass is never read in ASTBeahviorDefinition - why do we set it?
						((ASTBehaviorDefinition) node).setStateWMEClass(
							getCodeClassName(CODE_TYPE_STATE_WME));
						((ASTBehaviorDefinition) node).setStateWMEMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_STATE_WME)));
						break;
					case CODE_TYPE_PRECONDITION :
						 ((ASTBehaviorDefinition) node).setPreconditionClass(getCodeClassName(CODE_TYPE_PRECONDITION));
						((ASTBehaviorDefinition) node).setPreconditionMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(CODE_TYPE_PRECONDITION)));
						break;
					case CODE_TYPE_STEP_FACTORY :
						((GenericStep) node).setStepFactoryMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(codeStreamType)));
						break;
					case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
					case CODE_TYPE_MENTAL_STEP_EXECUTE :
						((GenericStep) node).setExecuteMethod_rField(
							getCodeMethodReflectionField(getCodeMethodName(codeStreamType)));
						break;
					default :
						throw new CompileError("Unexpected code type " + codeStreamType);
				}
			}

			codeStream.print(code.toString(3));
			caseCounter++;
			if (caseCounter > caseLimit) {
				closeSwitch();
				caseCounter = 0;
				startCodeMethod();
			}
		}

		private void closeSwitch() {
			// When closing the stream, close the currently open method.
			codeStream.println("      default:");

			switch (codeStreamType) {
				case CODE_TYPE_BEHAVIOR_FACTORY :
				case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
				case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
				case CODE_TYPE_CONTEXT_CONDITION :
				case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
				case CODE_TYPE_SUCCESS_CONDITION :
				case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
				case CODE_TYPE_REINFORCEMENT_TEST :
				case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
				case CODE_TYPE_STATE_WME :
				case CODE_TYPE_PRECONDITION :
					codeStream.println(
						"         throw new AblRuntimeError(\"Unexpected behaviorID \" + __$behaviorID);");
					break;
				case CODE_TYPE_STEP_FACTORY :
				case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
				case CODE_TYPE_MENTAL_STEP_EXECUTE :
				case CODE_TYPE_SUCCESS_TEST :
				case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					codeStream.println("         throw new AblRuntimeError(\"Unexpected stepID \" + __$stepID);");
					break;
				case CODE_TYPE_RL_POLICY:
					codeStream.println(
						"         throw new AblRuntimeError(\"Unexpected behaviorID \" + __$behaviorID);");
					break;
				default :
					throw new CompileError("Unexpected code type " + codeStreamType);
			}

			codeStream.println("      }"); // close the switch
			codeStream.println("   }"); // close the method
		}

		void closeCode() {
			closeSwitch();
			codeStream.println("}"); // close the close
			codeStream.close(); // close the file
		}

		void closeCode(CodeBlockDescriptor suffix) {
			closeSwitch();
			codeStream.print(suffix.toString(0));
			codeStream.println("}"); // close the close
			codeStream.close(); // close the file
		}
		
		private String getCodeClassName(int codeType) {
			switch (codeType) {
				case CODE_TYPE_BEHAVIOR_FACTORY :
					return behavingEntityClass + "_BehaviorFactories";
				case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
					return behavingEntityClass + "_PreconditionSensorFactories";
				case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
					return behavingEntityClass + "_ContextConditionSensorFactories";
				case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
					return behavingEntityClass + "_SuccessConditionSensorFactories";
				case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					return behavingEntityClass + "_SuccessTestSensorFactories";
				case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
					return behavingEntityClass + "_ReinforcementSignalSensorFactories";
				case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
					return behavingEntityClass + "_StateWMESensorFactories";
				case CODE_TYPE_CONTEXT_CONDITION :
					return behavingEntityClass + "_ContextConditions";
				case CODE_TYPE_SUCCESS_CONDITION :
					return behavingEntityClass + "_SuccessConditions";
				case CODE_TYPE_SUCCESS_TEST :
					return behavingEntityClass + "_SuccessTests";
				case CODE_TYPE_REINFORCEMENT_TEST :
					return behavingEntityClass + "_ReinforcementSignals";
				case CODE_TYPE_RL_POLICY:
					return behavingEntityClass + "_RLPolicy";
				case CODE_TYPE_STATE_WME :
					return behavingEntityClass + "_StateWMEs";
				case CODE_TYPE_PRECONDITION :
					return behavingEntityClass + "_Preconditions";
				case CODE_TYPE_STEP_FACTORY :
					return behavingEntityClass + "_StepFactories";
				case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
					return behavingEntityClass + "_ArgumentStepExecute";
				case CODE_TYPE_MENTAL_STEP_EXECUTE :
					return behavingEntityClass + "_MentalStepExecute";
				default :
					throw new CompileError("Unexpected code type " + codeType);
			}
		}

		private String getCodeMethodName(int codeType) {
			switch (codeType) {
				case CODE_TYPE_BEHAVIOR_FACTORY :
					return "behaviorFactory" + methodCounter;
				case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
					return "preconditionSensorFactory" + methodCounter;
				case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
					return "contextConditionSensorFactory" + methodCounter;
				case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
					return "successConditionSensorFactory" + methodCounter;
				case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					return "successTestSensorFactory" + methodCounter;
				case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
					return "reinforcementSignalSensorFactory" + methodCounter;
				case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
					return "stateWMESensorFactory" + methodCounter;
				case CODE_TYPE_CONTEXT_CONDITION :
					return "contextCondition" + methodCounter;
				case CODE_TYPE_SUCCESS_CONDITION :
					return "successCondition" + methodCounter;
				case CODE_TYPE_SUCCESS_TEST :
					return "successTest" + methodCounter;
				case CODE_TYPE_REINFORCEMENT_TEST :
					return "reinforcementSignal" + methodCounter;
				case CODE_TYPE_RL_POLICY :
					return "rlPolicy" + methodCounter;
				case CODE_TYPE_STATE_WME :
					return "stateWME" + methodCounter;
				case CODE_TYPE_PRECONDITION :
					return "precondition" + methodCounter;
				case CODE_TYPE_STEP_FACTORY :
					return "stepFactory" + methodCounter;
				case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
					return "argumentExecute" + methodCounter;
				case CODE_TYPE_MENTAL_STEP_EXECUTE :
					return "mentalExecute" + methodCounter;
				default :
					throw new CompileError("Unexpected code type " + codeType);
			}
		}

		CodeSequenceDescriptor compileReflectionFields(ClassDescriptor behavingEntity) {
			assert(methods.size() % 2 == 0); // Methods list is even

			final CodeSequenceDescriptor reflectionFieldInit = new CodeSequenceDescriptor();

			final Set<String> initedClasses = new HashSet<String>();
			final Iterator<String> methodsIter = methods.iterator();
			while (methodsIter.hasNext()) {
				String className = methodsIter.next();

				if (!initedClasses.contains(className)) {
					initedClasses.add(className);
					final FieldDescriptor classField = new FieldDescriptor();
					classField.fieldType = "Class";
					classField.addFieldModifiers(privateMods);
					final String classFieldName = getCodeMethodReflectionField(className);
					classField.addFieldName(classFieldName);
					behavingEntity.addField(classField);
					final String[] classFieldInitMacroArgs = new String[3];
					classFieldInitMacroArgs[0] = classFieldName;
					classFieldInitMacroArgs[1] = className;
					classFieldInitMacroArgs[2] = behavingEntityPackage;
					try {
						reflectionFieldInit.addToSequence(
							new CodeStringDescriptor(initCodeClassReflectionField.expand(classFieldInitMacroArgs)));
					} catch (SimpleMacroException e) {
						throw new CompileError("Error compiling static initializer block", e);
					}

				}

				String methodName = (String) methodsIter.next();
				String methodFieldName = getCodeMethodReflectionField(methodName);
				FieldDescriptor methodField = new FieldDescriptor();
				methodField.fieldType = "Method";
				methodField.addFieldModifiers(packageMods);
				methodField.addFieldName(methodFieldName);
				behavingEntity.addField(methodField);
				String[] macroArgs1 = new String[4];
				macroArgs1[0] = methodFieldName;
				macroArgs1[1] = getCodeMethodReflectionField(className);
				macroArgs1[2] = methodName;
				switch (codeStreamType) {
					case CODE_TYPE_BEHAVIOR_FACTORY :
						macroArgs1[3] = "__$behFactoryArgArray";
						break;
					case CODE_TYPE_PRECONDITION_SENSOR_FACTORY :
					case CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY :
					case CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY :
					case CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY :
					case CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY :
					case CODE_TYPE_STATE_WME_SENSOR_FACTORY :
						macroArgs1[3] = "__$sensorFactoryArgArray";
						break;
					case CODE_TYPE_CONTEXT_CONDITION :
					case CODE_TYPE_SUCCESS_TEST :
						macroArgs1[3] = "__$continuousConditionArgArray";
						break;
					case CODE_TYPE_SUCCESS_CONDITION :
						macroArgs1[3] = "__$successConditionArgArray";
						break;
					case CODE_TYPE_REINFORCEMENT_TEST :
						macroArgs1[3] = "__$reinforcementSignalArgArray";
						break;
					case CODE_TYPE_RL_POLICY :
						macroArgs1[3] = "__$rlPolicyArgArray";
						break;
					case CODE_TYPE_STATE_WME :
						macroArgs1[3] = "__$stateWMEArgArray";
						break;
					case CODE_TYPE_PRECONDITION :
						macroArgs1[3] = "__$preconditionArgArray";
						break;
					case CODE_TYPE_STEP_FACTORY :
						macroArgs1[3] = "__$stepFactoryArgArray";
						break;
					case CODE_TYPE_ARGUMENT_STEP_EXECUTE :
						macroArgs1[3] = "__$argumentStepExecuteArgArray";
						break;
					case CODE_TYPE_MENTAL_STEP_EXECUTE :
						macroArgs1[3] = "__$mentalStepExecuteArgArray";
						break;
					default :
						throw new CompileError("Unexpected code type " + codeStreamType);
				}
				String[] macroArgs2 = { methodFieldName };
				try {
					reflectionFieldInit.addToSequence(
						new CodeStringDescriptor(initCodeMethodReflectionField.expand(macroArgs1)));
					reflectionFieldInit.addToSequence(
						new CodeStringDescriptor(setAssignableCodeMethodReflectionField.expand(macroArgs2)));
				} catch (SimpleMacroException e) {
					throw new CompileError("Error compiling static initializer block", e);
				}
			}
			return reflectionFieldInit;
		}
	}

	// The name of the behaving entity defined by this behavior unit. 
	protected String behavingEntityClass;

	/* Name of the package within which all concrete class for this
	   behaving enity should be defined. */
	String behavingEntityPackage = null;

	/* Names of import packages. */
	protected final static String[] importPackages =
		{ "abl.runtime.*", "wm.WME", "java.util.*", "java.lang.reflect.Method" , "abl.learning.*"};

	/* Names of user declared import package. */
	private List<String> userImportPackages = new ArrayList<String>();

	/* Names of user declared imported classes. */
	private List<String> userImportClasses = new ArrayList<String>();

	// Hashtable mapping constant interfaces to Classes
	private final Hashtable<String, Class> userConstantDeclarations = new Hashtable<String, Class>();
	private final List<String> userConstantDeclarationsList = new ArrayList<String>();

	// Concrete behavior class registration macro. Used in the constructor of the concrete behaving entity. 
	private final static SimpleMacro behaviorRegistration =
		new SimpleMacro(behaviorUnitBehaviorRegistrationMacroString);

	// Registration method call. Appears in the constructor.
	private final static SimpleMacro behaviorRegistrationCall =
		new SimpleMacro(behaviorUnitBehaviorRegistrationCallMacroString);

	// fixme: remove?
	// private final static SimpleMacro ablEventSupportFieldInitializer = 
	// new SimpleMacro(MacroDefinitions.ablEventSupportFieldInitializerMacroString); 

	// fixme: remove?
	// private final static SimpleMacro addAblListenerBody = 
	// new SimpleMacro(MacroDefinitions.addAblListenerBodyMacroString);

	// fixme: remove?
	// private final static SimpleMacro removeAblListenerBody = 
	// 	new SimpleMacro(MacroDefinitions.removeAblListenerBodyMacroString);

	private final static SimpleMacro codeMethodReflectionFieldName =
		new SimpleMacro(MacroDefinitions.behaviorUnitCodeMethodReflectionFieldNameMacroString);

	protected final static SimpleMacro initCodeClassReflectionField =
		new SimpleMacro(MacroDefinitions.behaviorUnitInitCodeClassReflectionFieldMacroString);

	protected final static SimpleMacro initCodeMethodReflectionField =
		new SimpleMacro(MacroDefinitions.behaviorUnitInitCodeMethodReflectionFieldMacroString);

	protected final static SimpleMacro setAssignableCodeMethodReflectionField =
		new SimpleMacro(MacroDefinitions.behaviorUnitSetAssignableCodeMethodReflectionFieldMacroString);

	/* Hashtable mapping behavior names to Integer counts. Used by
	   behavior definition nodes to determine the name of the concrete
	   behavior class. */
	private Hashtable<String, Integer> behaviorCount = new Hashtable<String, Integer>();

	/* List of ASTBehaviorDefinitions defined in this behavior
	   unit. This list is modified by ASTBehaviorDefinition (by
	   calling addBehaviorNode). */
	private List<ASTBehaviorDefinition> individualBehaviorDefinitionNodes = new ArrayList<ASTBehaviorDefinition>();

	// List of joint ASTBehaviorDefinitions defined in this behavior.
	// This list is modified by ASTBehaviorDefinition (by calling addBehaviorNode).
	private List<ASTBehaviorDefinition> jointBehaviorDefinitionNodes = new ArrayList<ASTBehaviorDefinition>();

	/* Hash table of registered WMEs. WME class names are the keys and
	   sensors are the values. */
	private Hashtable<String, String> wmeRegistrations = new Hashtable<String, String>();

	/* Any ABL program only has one behavior unit node; this is the
	   root node of the parse tree. This field stores a static
	   reference to the unique instantiated parse node. */
	private static ASTBehaviorUnit behaviorUnit;

	/* Hash table of registered actions. ABL action names are the
	   keys; lists of RegisteredActions are the values. */
	private Hashtable<String, List<RegisteredAction>> actionRegistrations = new Hashtable<String, List<RegisteredAction>>();

	/* Hash table of WMEs declared in the behavior unit. Keys are
	   class names, values are hash tables mapping field names to
	   types. */
	private Hashtable<String, Hashtable<String, String>> declaredWMEs = new Hashtable<String, Hashtable<String, String>>();

	// Hash table of registered conflicts. Keys are step names (goals
	// or primitive acts). Values are hash sets of all the
	// conflicting steps.
	private Hashtable<String, Set<String>> registeredConflicts = new Hashtable<String, Set<String>>();

	// String representation of the decision cycle callback.
	ASTJavaName decisionCycleSMCallNode = null;

	// True if default success negotiation requires the whole team to succeed.
	private boolean teamNeededForSuccess = false;
	private boolean teamSuccessModifierExplicitlySet = false;

	// Hash table of declared properties. Keys are property names (strings).
	// Values are type names (strings).
	private Hashtable<String, String> declaredProperties = new Hashtable<String, String>();

	// The number of behavior registrations to include within each registration call
	private final static int REGISTRATION_SIZE = 1000;

	private CodeStream behaviorFactoryStream;
	private CodeStream contextConditionSensorFactoryStream;
	private CodeStream successConditionSensorFactoryStream;
	private CodeStream preconditionSensorFactoryStream;
	private CodeStream contextConditionStream;
	private CodeStream successConditionStream;
	private CodeStream preconditionStream;
	private CodeStream stepFactoryStream;
	private CodeStream argumentStepExecuteStream;
	private CodeStream mentalStepExecuteStream;
	private CodeStream successTestStream;
	private CodeStream successTestSensorFactoryStream;
	private CodeStream reinforcementTestStream;
	private CodeStream reinforcementTestSensorFactoryStream;
	private CodeStream rlPolicyStream;
	private CodeBlockDescriptor rlPolicySuffix;
	private CodeStream stateWMEStream;
	private CodeStream stateWMESensorFactoryStream;

	// fixme: create new counters for context conditions. 
	// By creating separate ids for context conditions, I can make these cases dense.
	// Counter used to assign unique behavior IDs
	private int behaviorIDCounter = 0;

	// fixme: create new counters for argument execute, mental execute and success tests.
	// By creating separate ids for these code entities, I can make these cases dense.
	// Counter used to assign unique step IDs
	private int stepIDCounter = 0;

	// behavior id for the root collection behavior - set by ASTBehaviorDefinition
	int rootBehaviorID;

	// A set of conflict sets. Used to create a collection of conflict set arrays to define on <entity>.java.
	private final Set<List<String>> conflictSets = new HashSet<List<String>>(100);
	private final Hashtable<List<String>, String> conflictSetsToFieldNames = new Hashtable<List<String>, String>(100);

	ASTBehaviorUnit(int id) {
		super(id);
		behaviorUnit = this;
	}

	ASTBehaviorUnit(AblParser p, int id) {
		super(p, id);
		behaviorUnit = this;
	}

	void setBehavingEntityClass(String behaviorUnitName) {
		/* Uppercase the first character of the behaviorUnitName so
		       that it becomes a proper class name for the concrete
		       behaving entity. */
		behavingEntityClass = uppercaseFirstCharacter(behaviorUnitName);
	}

	public String getBehavingEntityClass() {
		return behavingEntityClass;
	}

	/* Set accessor for user imports. */
	void addUserImport(String importName) {
		if (importName.endsWith("*")) {
			/* The import name ends with a "*"; trim off the ".*"
			       suffix and add the resulting package name to
			       userImportPackages. */
			userImportPackages.add(importName.substring(0, importName.length() - 2));
		} else {
			/* The import name doesn't end in "*"; it is an import of
			       a specific class. */
			userImportClasses.add(importName);
		}
	}

	/* Adds all user imports (packages and classes) to the
	   ClassDescriptor. */
	void addUserImports(ClassDescriptor c) {
		final Iterator<String> packageIter = userImportPackages.listIterator();
		while (packageIter.hasNext()) {
			c.addPackageImport(packageIter.next() + ".*");
		}

		final Iterator<String> classIter = userImportClasses.listIterator();
		while (classIter.hasNext()) {
			c.addPackageImport(classIter.next());
		}
	}

	void printUserImports(PrintStream p) {
		final Iterator<String> packageIter = userImportPackages.iterator();
		while (packageIter.hasNext()) {
			p.println("import " + packageIter.next() + ".*;");
		}

		final Iterator<String> classIter = userImportClasses.iterator();
		while (classIter.hasNext()) {
			p.println("import " + classIter.next() + ";");
		}
	}

	/* Get accessor for user package imports. Returns a ListIterator
	   for the package imports. */
	Iterator<String> getUserImportPackages() {
		return userImportPackages.iterator();
	}

	Iterator<String> getUserImportClasses() {
		return userImportClasses.iterator();
	}

	// Set accessor for user constant declarations. 
	void addUserConstants(String constantsName) throws CompileException {
		Class constantInterface;
		try {
			constantInterface = Class.forName(constantsName);
			if (!Modifier.isInterface(constantInterface.getModifiers()))
				throw new CompileException("Expected " + constantsName + " to be an interface declaring constants.");
		} catch (ClassNotFoundException e) {
			throw new CompileException("Unable to find interface " + constantsName);
		}
		Field[] fields = constantInterface.getFields();
		for (int i = 0; i < fields.length; i++) {
			final String fieldName = fields[i].getName();
			if (userConstantDeclarations.get(fieldName) != null)
				throw new CompileException(
					"Duplicate declaration of constant " + fieldName + " in file " + constantsName);
			userConstantDeclarations.put(fieldName, constantInterface);
		}
		userConstantDeclarationsList.add(constantsName);

	}

	/* Adds all user constant declarations to the list of interfaces
	   the class descriptor implements. */
	void addUserConstantDeclarations(ClassDescriptor c) {
		final Iterator<String> constantsIter = userConstantDeclarationsList.iterator();
		while (constantsIter.hasNext()) {
			c.addInterface(constantsIter.next());
		}
	}

	/* Adds all user constant declarations to the list of interfaces
	   the class descriptor implements. */
	void printUserConstantDeclarations(PrintStream p) {
		final Iterator<String> constantsIter = userConstantDeclarationsList.iterator();
		if (constantsIter.hasNext()) {
			p.print("implements ");
			while (constantsIter.hasNext()) {
				p.print(constantsIter.next());
				if (constantsIter.hasNext())
					p.print(", ");
			}
			p.println("");
		}
	}

	Field getDeclaredConstant(String name) throws CompileException {
		Class constantInterface = (Class) userConstantDeclarations.get(name);
		if (constantInterface == null)
			return null; // No constant name was found
		else {
			try {
				return constantInterface.getDeclaredField(name);
				// Found a final declaration; return it.
			} catch (SecurityException e) {
				throw new CompileError(
					"Security exception accessing " + name + " in interface " + constantInterface.getName());
			} catch (NoSuchFieldException e) {
				throw new CompileError("Constant " + name + " not found in interface " + constantInterface.getName());
			}
		}
	}

	// set accessor for teamNeededForSuccess.
	// Returns false if teamNeededForSuccess has been previously set, true otherwise.  
	boolean setTeamNeededForSuccess(boolean teamNeededForSuccess) {
		this.teamNeededForSuccess = teamNeededForSuccess;
		if (teamSuccessModifierExplicitlySet)
			return false;
		else {
			teamSuccessModifierExplicitlySet = true;
			return true;
		}
	}

	// get accessor for teamNeededForSuccess.
	boolean getTeamNeededForSuccess() {
		return teamNeededForSuccess;
	}

	/* get accessor for the unique ASTBehaviorUnit reference. */
	static ASTBehaviorUnit getBehaviorUnit() {
		return behaviorUnit;
	}

	void addBehaviorNode(ASTBehaviorDefinition behaviorNode) {
		if (behaviorNode.isJoint())
			jointBehaviorDefinitionNodes.add(behaviorNode);
		else
			individualBehaviorDefinitionNodes.add(behaviorNode);
	}

	/* Returns the current count of the number of behaviors sharing a
	   behavior name. The first time this is called it returns 1, not
	   0. */
	Integer getBehaviorCount(String behaviorName) {
		Integer count = (Integer) behaviorCount.get(behaviorName);
		if (count == null)
			return new Integer(1);
		else
			return count;
	}

	/* Increment the current count of the number of behaviors sharing
	   a behavior name. */
	void incrementBehaviorCount(String behaviorName) {
		Integer count = (Integer) behaviorCount.get(behaviorName);
		if (count == null)
			behaviorCount.put(behaviorName, new Integer(2));
		else {
			int newCount = count.intValue() + 1;
			behaviorCount.put(behaviorName, new Integer(newCount));
		}
	}

	private void compileChildren() throws CompileException {
		AblParseNode n;
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through the children nodes, looking compileable nodes.

			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTWMEDECL) {
				// Store the field names and types in the declaredWMEs hash table

				ASTWMEDecl wmeDecl = (ASTWMEDecl) n;
				declaredWMEs.put(wmeDecl.wmeClassName, wmeDecl.getFieldTypeTable());

				ClassDescriptor wmeClass = (ClassDescriptor) wmeDecl.compileToJava();
				if (!Abl.noCodeGeneration)
					wmeClass.writeToFile(getFullPackagePath(Abl.objectDirectory,behavingEntityPackage));
			} else if (n.id == JJTBEHAVIORDEFINITION) {
				// If the child node is compileable, compile it.
				 ((ASTBehaviorDefinition) n).compileToJava();
			}
		}
	}

	// Returns true if the wme class wmeClassName has been declared in
	// the behavior unit. 
	boolean isWMEDeclared(String wmeClassName) {
		if (declaredWMEs.get(wmeClassName) != null)
			return true;
		else
			return false;
	}

	// Given a wme class name and a field name, returns the name of
	// the type of the field. If either the class name has not been
	// declared in the behavior unit or the class doesn't contain the
	// requested field, returns null.
	String lookupWMEFieldType(String wmeClassName, String wmeFieldName) {
		Hashtable typeTable = (Hashtable) declaredWMEs.get(wmeClassName);
		if (typeTable != null)
			return (String) typeTable.get(wmeFieldName);
		else
			return null;
	}

	private MethodDescriptor compileDecisionCycleSMCall() throws CompileException {
		if (decisionCycleSMCallNode != null) {
			Method decisionCycleSMCallMethod = decisionCycleSMCallNode.getMethod();
			if (decisionCycleSMCallMethod == null)
				throw new CompileException(
					"The decision cycle callback " + decisionCycleSMCallNode.dumpTokens() + " is not a method");
			int methodMods = decisionCycleSMCallMethod.getModifiers();
			if (!Modifier.isStatic(methodMods))
				throw new CompileException("The decision cycle callback method must be static");
			if (!Modifier.isPublic(methodMods))
				throw new CompileException("The decision cycle callback method must be public");
			if (decisionCycleSMCallMethod.getParameterTypes().length != 0)
				throw new CompileException("The decision cycle callback method must take 0 parameters");

			// if we get this far, define the decision cycle callback
			MethodDescriptor decisionCycleSMCallDescriptor = new MethodDescriptor();
			decisionCycleSMCallDescriptor.addModifier("protected");
			decisionCycleSMCallDescriptor.addModifier("void");
			decisionCycleSMCallDescriptor.methodName = "decisionCycleSMCall";
			decisionCycleSMCallDescriptor.addToBlockBody(
				new CodeStringDescriptor(decisionCycleSMCallNode.dumpTokens() + "();"));
			return decisionCycleSMCallDescriptor;
		} else
			return null;
	}

	// Compile individual registration method
	private MethodDescriptor compileRegistrationMethod(int methodCounter, Hashtable<Integer, HashSet<String>> signatures, final Iterator<ASTBehaviorDefinition> behaviorIter) {
		MethodDescriptor regMethod = new MethodDescriptor();
		regMethod.addModifier("private");
		regMethod.addModifier("static");
		regMethod.addModifier("void");

		regMethod.methodName = "registerBehaviors_" + methodCounter;

		regMethod.addArgument(new MethodArgDescriptor("BehaviorLibrary", "behaviorLibrary"));

		try {
			// Add registration calls
			for (int i = 0; i < REGISTRATION_SIZE && behaviorIter.hasNext(); i++) {
				final ASTBehaviorDefinition beh = behaviorIter.next();
				final int behaviorID = beh.getID();
				final String behaviorFactoryMethod_rField = beh.getBehaviorFactoryMethod_rField();
				final String preconditionMethod_rField = beh.getPreconditionMethod_rField();
				final String preconditionSensorFactoryMethod_rField = beh.getPreconditionSensorFactoryMethod_rField();
				final StringBuffer teamBuf = new StringBuffer(100);
				if (beh.isJoint()) {
					final List teamMembers = beh.getTeamMembers();
					final Iterator<String> iter = teamMembers.iterator();
					teamBuf.append("new String[] {");
					while (iter.hasNext()) {
						final String roleName = iter.next();
						teamBuf.append("\"" + roleName + "\"");
						if (iter.hasNext())
							teamBuf.append(", ");
					}
					teamBuf.append("}");
				} else
					teamBuf.append("null");

                final StringBuffer signaturesBuf = new StringBuffer(100);
                //put the main signature into the string first
                signaturesBuf.append("new String[] {\"" + beh.getSignature() + "\"" + ", ");
                if (signatures.get(behaviorID) == null || signatures.get(behaviorID).size() == 0)
                {
                    signaturesBuf.append("null");
                }
                else {
                    for (String sig : signatures.get(behaviorID)) {
                        if (!sig.equals(beh.getSignature()))
                            signaturesBuf.append("\"" + sig + "\"" + ", ");
                    }
                }
                //remove the trailing ", "
                signaturesBuf.delete(signaturesBuf.lastIndexOf(","), signaturesBuf.length());
                signaturesBuf.append("}");



				String[] behaviorRegistrationArgs =
					{
						beh.getSignature(),
                        signaturesBuf.toString(),
						Integer.toString(behaviorID),
						behaviorFactoryMethod_rField,
						preconditionMethod_rField,
						preconditionSensorFactoryMethod_rField,
						teamBuf.toString(),
						Integer.toString(beh.specificity)};
				regMethod.addToBlockBody(
					new CodeStringDescriptor(behaviorRegistration.expand(behaviorRegistrationArgs)));
			}
		} catch (SimpleMacroException e) {
			throw new CompileError("Error compiling behavior registration method", e);
		}

		return regMethod;
	}

    /*
      This adds a very naive convenience entrypoint/main method: 
      public static void main(String args[]) { new EntityName().startBehaving(); }
     */
    private MethodDescriptor compileEntryPoint() {
        MethodDescriptor entryPoint = new MethodDescriptor();
        entryPoint.addModifier("public");
        entryPoint.addModifier("static");
        entryPoint.addModifier("void");
        entryPoint.methodName = "main";
        entryPoint.addArgument(new MethodArgDescriptor("String[]", "args"));
        entryPoint.addToBlockBody(new CodeStringDescriptor("new "+behavingEntityClass+"().startBehaving();"));
        return entryPoint;
    }

	// Compile the constructor for the behaving entity. 
	private MethodDescriptor compileConstructor() {
		MethodDescriptor constructor = new MethodDescriptor();
		constructor.addModifier("public");
		constructor.methodName = behavingEntityClass;

		// First initialize the BehaviorLibraries
		constructor.addToBlockBody(
			new CodeStringDescriptor(
				"individualBehaviorLibrary = new BehaviorLibrary("
					+ individualBehaviorDefinitionNodes.size() * 2
					+ ");"));
		constructor.addToBlockBody(
			new CodeStringDescriptor(
				"jointBehaviorLibrary = new BehaviorLibrary(" + jointBehaviorDefinitionNodes.size() * 2 + ");"));

		if (Abl.debugLevel != NO_DEBUG) {
			// If a debug level has been specified, set it in the BehavingEntity
			constructor.addToBlockBody(new CodeStringDescriptor("debugLevel = " + Abl.debugLevel + ";"));
		}

		if (Abl.debugLevel == GUI_DEBUGGER) {
			// If the GUI debugger has been activated, create it in the BehavingEntity
			constructor.addToBlockBody(new CodeStringDescriptor("debuggerGUI = new Debugger(this);"));
		}

		try {
			// Register the behaviors with the behaving entity.

			// Add calls to the constructor for the methods that register the behaviors
			int registrationMethodCounter = 0;

			// Add calls for individual behaviors
			for (int i = 0;
				i < individualBehaviorDefinitionNodes.size() / REGISTRATION_SIZE + 1;
				i++, registrationMethodCounter++) {
				String[] args = { Integer.toString(registrationMethodCounter), "individualBehaviorLibrary" };
				constructor.addToBlockBody(new CodeStringDescriptor(behaviorRegistrationCall.expand(args)));
			}

			// Add calls for joint behaviors
			for (int i = 0;
				i < jointBehaviorDefinitionNodes.size() / REGISTRATION_SIZE + 1;
				i++, registrationMethodCounter++) {
				String[] args = { Integer.toString(registrationMethodCounter), "jointBehaviorLibrary" };
				constructor.addToBlockBody(new CodeStringDescriptor(behaviorRegistrationCall.expand(args)));
			}

			// Next initialize the conflict sets
			for (int i = 0; i < initConflictSetMethodCounter; i++)
				constructor.addToBlockBody(new CodeStringDescriptor("__$initConflictSet" + i + "();"));

			// Initialize the behavior tree.
			// Assume that the root collection behahavior is always defined in the last behaviorFactory method.
			// fixme: if we move to multiple BehaviorFactory classes, will need to indicate which behavior factory class.
			// fixme: doing this now wihtout macros. SimpleMacro doesn't handle numbers in the macro string (problem with 
			// StreamTokenizer)
			/* String[] abtInitializationArgs = {behavingEntityClass, Integer.toString(rootBehaviorID), 
						      Integer.toString(behaviorFactoryStream.getMethodCounter()) };
			if (Abl.bMetaEnabled) {
			if (Abl.bDebugBehavingEntity)
			    constructor.addToBlockBody(new CodeStringDescriptor(abtInitializationMetaDebug.expand(abtInitializationArgs)));		
			else
			    constructor.addToBlockBody(new CodeStringDescriptor(abtInitializationMeta.expand(abtInitializationArgs)));
			}
			else {
			if (Abl.bDebugBehavingEntity)
			    constructor.addToBlockBody(new CodeStringDescriptor(abtInitializationNoMetaDebug.expand(abtInitializationArgs)));
			else
			    constructor.addToBlockBody(new CodeStringDescriptor(abtInitializationNoMeta.expand(abtInitializationArgs)));
			}
			*/
			if (Abl.debugLevel == GUI_DEBUGGER)
				constructor.addToBlockBody(
					new CodeStringDescriptor(
						"ABT = (CollectionBehavior)"
							+ behavingEntityClass
							+ "_BehaviorFactories.behaviorFactory"
							+ behaviorFactoryStream.getMethodCounter()
							+ "("
							+ rootBehaviorID
							+ ", null, null, null, \""
							+ behavingEntityClass
							+ "_RootCollectionBehavior()\", (__BehaviorDesc)((List)individualBehaviorLibrary.lookupBehavior(\""
							+ behavingEntityClass
							+ "_RootCollectionBehavior()\")).get(0));"));
			else
				constructor.addToBlockBody(
					new CodeStringDescriptor(
						"ABT = (CollectionBehavior)"
							+ behavingEntityClass
							+ "_BehaviorFactories.behaviorFactory"
							+ behaviorFactoryStream.getMethodCounter()
							+ "("
							+ rootBehaviorID
							+ ", null, null, null, \""
							+ behavingEntityClass
							+ "_RootCollectionBehavior()\");"));
			if (Abl.bMetaEnabled)
				constructor.addToBlockBody(new CodeStringDescriptor("startWMEReflection(ABT);"));

			constructor.addToBlockBody(new CodeStringDescriptor("ABT.initRootBehavior();"));

			/* If the current line of expansion heuristic has been
			       turned off at the command line, set the current line of
			       expansion flag to false. */
			if (!Abl.currentLineOfExpansion)
				constructor.addToBlockBody(new CodeStringDescriptor("bCurrentLineOfExpansion = false;"));

			// If a decision cycle sm call has been specified, set the flag.
			if (decisionCycleSMCallNode != null)
				constructor.addToBlockBody(new CodeStringDescriptor("bDecisionCycleSMCall = true;"));

			// Register this entity with its name
			constructor.addToBlockBody(
				new CodeStringDescriptor("registerEntity(" + "\"" + behavingEntityClass + "\", this);"));

		} catch (SimpleMacroException e) {
			throw new CompileError("Error compiling behaving entity constructor", e);
		}

		return constructor;
	}

	/* Process all children action registration parse nodes. */
	private void processActionRegistration() {
		AblParseNode node;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through all the child nodes looking for action registrations. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTACTIONREGISTRATION) {
				/* An action registration node found. Hash the registration. */

				final RegisteredAction regAction =
					new RegisteredAction(
						((ASTActionRegistration) node).getPrimitiveAction(),
						((ASTActionRegistration) node).getArgumentTypes());

				final String actionName = ((ASTActionRegistration) node).actionName;

				/* Eventually add some error checking to make sure
				           that the actionClass exists. */
				final List<RegisteredAction> regActions = actionRegistrations.get(actionName);
				if (regActions != null)
					/* Actions with this name have been registered
					           before. Add this particular registration to the
					           list. */
					// Eventually add some error checking for duplicate registrations. 
					regActions.add(regAction);
				else {
					/* No action with this name has been registered
					           before. Create a new linked list and add the
					           action. */
					final List<RegisteredAction> newActionList = new ArrayList<RegisteredAction>();
					newActionList.add(regAction);
					actionRegistrations.put(actionName, newActionList);
				}
			}
		}
	}

	/* get accessor for action registrations. Given an action
	   name, returns a linked list of RegisteredAction objects
	   representing all the registered actions sharing that
	   name. Returns null if there are no registered actions with the
	   requested action name. */
	List<RegisteredAction> lookupRegisteredAction(String actionName) {
		return actionRegistrations.get(actionName);
	}

	/* Process all children WME registration parse nodes. */
	private void processWMERegistration() {
		String registeredWME;
		String sensorWMERegisteredOn;
		AblParseNode node;
		for (int i = 0; i < jjtGetNumChildren(); i++) {
			/* Loop through all the child nodes, looking for WME
			       registrations. */

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTWMEREGISTRATION) {
				/* A WME registration node found. Hash the
				           registration. */

				registeredWME = ((ASTWMERegistration) node).getRegisteredWME();
				sensorWMERegisteredOn = ((ASTWMERegistration) node).getSensorWMERegisteredOn();

				/* Eventually add some error checking to make sure
				           that any registered wme has been defined. */
				// System.out.println("Processed registration: " + registeredWME + " " + sensorWMERegisteredOn);
				wmeRegistrations.put(registeredWME, sensorWMERegisteredOn);
			}
		}
	}

	/* Given the string representation of a wme class name, returns
	   the string representation of the sensor the WME is registered
	   on. Returns null if the WME class has not been registered. */
	String lookupRegisteredWME(String wmeClassName) {
		return wmeRegistrations.get(wmeClassName);
	}

	// process the property declarations
	private void processPropertyDeclarations() throws CompileException {
		AblParseNode node;

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// loop through all the child nodes looking for property declarations. 

			node = (AblParseNode) jjtGetChild(i);
			if (node.id == JJTPROPERTYDECLARATION) {
				// A property declaration node found. Declare the property.

				String propertyName = ((ASTPropertyDeclaration) node).getPropertyName();
				String propertyType = ((ASTPropertyDeclaration) node).getPropertyType();
				declareProperty(propertyName, propertyType);

			}
		}
	}

	// Registers stepName2 as conflicting with stepName1
	private void registerConflictHelper(String stepName1, String stepName2) {
		Set<String> conflictingSteps = registeredConflicts.get(stepName1);
		if (conflictingSteps != null) {
			if (!conflictingSteps.contains(stepName2)) {
				// This conflicting step has not been seen before - add it.
				// Silently ignore redundant declarations
				conflictingSteps.add(stepName2);
			}
		} else {
			conflictingSteps = new HashSet<String>();
			conflictingSteps.add(stepName2);
			registeredConflicts.put(stepName1, conflictingSteps);
		}
	}

	// Registers a pair of step names as conflicting with each other
	void registerConflictPair(String stepName1, String stepName2) {
		registerConflictHelper(stepName1, stepName2);
		if (!stepName1.equals(stepName2))
			registerConflictHelper(stepName2, stepName1);
	}

	// Registers a List of step names as conflicting with each other
	void registerConflictList(List conflictList) {
		for (int i = 0; i < conflictList.size() - 1; i++) {
			String stepName1 = (String) conflictList.get(i);
			for (int j = i + 1; j < conflictList.size(); j++) {
				String stepName2 = (String) conflictList.get(j);
				registerConflictPair(stepName1, stepName2);
			}
		}
	}

	// Returns any array of step names the argument step conflicts
	// with. Returns null if there are no conflicts.
	String[] lookupConflict(String stepName) {
		Set<String> conflictingSteps = registeredConflicts.get(stepName);
		if (conflictingSteps == null)
			return null;
		else
			return conflictingSteps.toArray(new String[conflictingSteps.size()]);
	}

	// Registers a property name. Checks for duplicate property names.
	private void declareProperty(String name, String type) throws CompileException {
		// first check for duplicate registration
		if (declaredProperties.get(name) != null)
			throw new CompileException("Duplicate property declaration: " + name);
		declaredProperties.put(name, type);
	}

	// Lookup a declared property by name. If it has been declared, returns the type name (as string), otherwise
	// returns null.
	String lookupDeclaredProperty(String name) {
		return (String) declaredProperties.get(name);
	}


    /*
     * Added by Gillian Smith, 7-17-10.
     *
     * Given the short name of a class (i.e. tokens that the parser finds), searches the imported
     * classes and packages to find the matching long name of the class, and then returns the Class
     * object associated with it.
     */
    public Class findClassFromShortName(String shortName) {
        //handle the case for them being primitive types
        if (shortName.equals("boolean")) return Boolean.TYPE;
        else if (shortName.equals("byte")) return Byte.TYPE;
        else if (shortName.equals("char")) return Character.TYPE;
        else if (shortName.equals("short")) return Short.TYPE;
        else if (shortName.equals("int")) return Integer.TYPE;
        else if (shortName.equals("long")) return Long.TYPE;
        else if (shortName.equals("float")) return Float.TYPE;
        else if (shortName.equals("double")) return Double.TYPE;
        else if (shortName.equals("void")) return Void.TYPE;

        //First check to see if it was directly imported
        Iterator<String> classnameItr = getUserImportClasses();
        while (classnameItr.hasNext()) {
            String classname = classnameItr.next();
            if (classname.substring(classname.lastIndexOf('.') + 1, classname.length()).equals(shortName)) {
                try {
                    return Class.forName(classname);
                }
                catch (ClassNotFoundException e) {
                }
            }
        }

        //If not, then check the package that ABL source compiles to
        try {
            return Class.forName(behavingEntityPackage + "." + shortName);
        }
        catch (ClassNotFoundException e) {    
        }

        //If not, then we need to search the imported packages
        Iterator<String> packagenameItr = getUserImportPackages();
        while (packagenameItr.hasNext()) {
            String packagename = packagenameItr.next();
            try {
                Class c = Class.forName(packagename + "." + shortName);
                return c;
            }
            catch (ClassNotFoundException e) {
            }
        }

        //If it's not in the user declared packages OR classes, then it doesn't exist? Return null.
        return null;
    }

    /*
     * Added by Gillian Smith, 7-17-10
     *
     * Goes through all behaviors defined in the source, collecting signatures as lists of Classes.
     * Key to hashtable is a String describing the behavior's name and # arguments ( e.g. Foo(2) ).
     * Value is a list of lists, where the nested lists are lists of classes prepended by their behaviorID.
     *
     * Uses hashtables (one for joint, one for individual behaviors) to create a hashtable that is keyed by
     * behavior ID and maps to a list of valid signatures for that hashtable.
     *
     * @return two hashtables mapping behavior IDs to valid signatures used in the program (matching for inheritance)
     * the first item in the array is for individual behaviors, the second is for joint behaviors
     */
    private ArrayList<Hashtable<Integer, HashSet<String>>> buildBehaviorSignatureIndices() throws CompileException {
        AblParseNode n;

        //Loop over the behaviors, collecting signatures as lists of Classes
        //Class lists are hashed using the name plus the number of arguments
        //as the key. The values are lists of lists, where the nested lists
        //are lists of Classes. e.g.
        //
        //behavior Foo(DogWME, CatWME){}
        //behavior Foo(DogWME, FoodWME) {}
        //this would create the entry
        //"Foo(2)" -> [ [behaviorID, DogWME, CatWME] , [ [behaviorID, DogWME, FoodWME] ]
        

        Hashtable<String, ArrayList<ArrayList<Object>>> individualBehaviorSigTable =
                new Hashtable<String, ArrayList<ArrayList<Object>>>();

        Hashtable<String, ArrayList<ArrayList<Object>>> jointBehaviorSigTable =
                new Hashtable<String, ArrayList<ArrayList<Object>>>();

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            n = (AblParseNode)jjtGetChild(i);
            if (n.id == JJTBEHAVIORDEFINITION) {
                ASTBehaviorDefinition behDef = (ASTBehaviorDefinition) n;
                //store the signature of this behavior
                String behaviorSig = behDef.getSignature();
                //get the arguments - this is the short name
                Object[] args = behDef.getFormalArguments();
                //figure out the key for the table
                String key = behaviorSig.substring(0, behaviorSig.indexOf('(')) + "(" + args.length + ")";
                //Build an array of Objects: [int id, Class... args]
                ArrayList<Object> value = new ArrayList<Object>();
                value.add(behDef.getID());
                for (Object o : args) {
                    //find the Class object corresponding to this short name from the classpath
                    Class c = findClassFromShortName(((MethodArgDescriptor)o).argType);
                    if (c != null)
                    {
                        value.add(c);
                    }
                    else
                    {
                        throw new CompileException("Unable to find class " + ((MethodArgDescriptor)o).argType +
                                                   " in the imported packages.");
                    }
                }

                //check to see if the behavior is joint or individual
                if (behDef.isJoint())
                {
                    //if the key is already in the behavior signature table, add this new signature to the list
                    if (jointBehaviorSigTable.containsKey(key))
                    {
                        jointBehaviorSigTable.get(key).add(value);
                    }
                    //otherwise, create the initial array and add that under the key
                    else
                    {
                        ArrayList<ArrayList<Object>> tempList = new ArrayList<ArrayList<Object>>();
                        tempList.add(value);
                        jointBehaviorSigTable.put(key, tempList);
                    }
                }
                else
                {
                    //if the key is already in the behavior signature table, add this new signature to the list
                    if (individualBehaviorSigTable.containsKey(key))
                    {
                        individualBehaviorSigTable.get(key).add(value);
                    }
                    //otherwise, create the initial array and add that under the key
                    else
                    {
                        ArrayList<ArrayList<Object>> tempList = new ArrayList<ArrayList<Object>>();
                        tempList.add(value);
                        individualBehaviorSigTable.put(key, tempList);
                    }
                }
            }
        }

        //now that we have the individual and joint behavior signature tables, use them to determine a mapping
        //of behavior IDs -> signatures that are actually used in the program.

        ArrayList<Hashtable<Integer, HashSet<String>>> signatureIndices =
                                                            new ArrayList<Hashtable<Integer, HashSet<String>>>();

        Hashtable<Integer, HashSet<String>> individualSignatureIndex = new Hashtable<Integer, HashSet<String>>();
        Hashtable<Integer, HashSet<String>> jointSignatureIndex = new Hashtable<Integer, HashSet<String>>();

        for (int i = 0; i < jjtGetNumChildren(); i++)
        {
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTBEHAVIORDEFINITION)
            {
                //get the signatures that should be added to each table
                ArrayList<Hashtable<Integer, HashSet<String>>> signatures =
                        ((ASTBehaviorDefinition)n).typeCheckSubgoalsWithInheritance(individualBehaviorSigTable,
                                                                                   jointBehaviorSigTable,
                                                                                   this);

                //go through the queue of individual signatures to be added and put them into the individual signature index
                Enumeration<Integer> indKeys = signatures.get(0).keys();
                while (indKeys.hasMoreElements())
                {
                    Integer key = indKeys.nextElement();
                    if (individualSignatureIndex.containsKey(key))
                    {
                        HashSet<String> sigs = signatures.get(0).get(key);
                        for (String s : sigs)
                        {
                            individualSignatureIndex.get(key).add(s);
                        }
                    }
                    else
                    {
                        individualSignatureIndex.put(key, signatures.get(0).get(key));
                    }
                }

                //go through the queue of joint signatures to be added and put them into the joint signature index
                Enumeration<Integer> jointKeys = signatures.get(1).keys();
                while (jointKeys.hasMoreElements())
                {
                    Integer key = jointKeys.nextElement();
                    if (jointSignatureIndex.containsKey(key))
                    {
                        HashSet<String> sigs = signatures.get(0).get(key);
                        for (String s : sigs)
                        {
                            jointSignatureIndex.get(key).add(s);
                        }
                    }
                    else
                    {
                        jointSignatureIndex.put(key, signatures.get(0).get(key));
                    }
                }
            }
        }


        //DEBUG: Print the individual behavior signature table
        /*System.out.println("\n=========Individual Behavior Signature Table=========");
        Set<String> keys = individualBehaviorSigTable.keySet();
        for (String s : keys)
        {
            System.out.print(s + ":--" + "\t");
            for (int i = 0; i < individualBehaviorSigTable.get(s).size(); i++)
            {
                ArrayList<Object> value = individualBehaviorSigTable.get(s).get(i);
                System.out.print(value.get(0));
                for (int j = 1; j < value.size(); j++)
                {
                    System.out.print(", " + ((Class)value.get(j)).getCanonicalName());
                }
                System.out.print("\t");
            }
            System.out.println();
        }*/

        //DEBUG: Print the individual signature index
       /* System.out.println("\n=========Individual Signature Index=========");
        Set<Integer> indexKeys = individualSignatureIndex.keySet();
        for (Integer i : indexKeys) {
            System.out.print(i + ":--" + "\t");
            for (String s : individualSignatureIndex.get(i)) {
                System.out.print(s + " ; ");
            }
            System.out.println();
        }*/

        signatureIndices.add(individualSignatureIndex);
        signatureIndices.add(jointSignatureIndex);

        return signatureIndices;
    }


	// Looks for subgoal steps with signatures not matching the signatures of any behavior. 
	private void checkForSubgoalsWithNoMatchingBehaviors() throws CompileException {
		AblParseNode n;

		// First get a hash set of the behavior signatures.

		final Set<String> individualSigSet = new HashSet<String>();
		// set of individual behavior signatures
		final Set<String> jointSigSet = new HashSet<String>(); // set of joint behavior signatures

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through the children nodes looking for behavior definitions

			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTBEHAVIORDEFINITION) {
				ASTBehaviorDefinition behDef = (ASTBehaviorDefinition) n;
				// If the child node is a behavior definition, add its signature to the appropriate hashset
				if (behDef.isJoint())
					jointSigSet.add(behDef.getSignature());
				else
					individualSigSet.add(behDef.getSignature());
			}
		}

		// Then check the subgoals of every step looking for unsatisfied subgoals

		for (int i = 0; i < jjtGetNumChildren(); i++) {
			// Loop through the children nodes, looking for behavior definitions

			n = (AblParseNode) jjtGetChild(i);
			if (n.id == JJTBEHAVIORDEFINITION) {
				// If the child node is a behavior definition, check its steps for unsatisfied subgoals

				 ((ASTBehaviorDefinition) n).checkForSubgoalsWithNoMatchingBehaviors(individualSigSet, jointSigSet);
			}
		}
	}

	// fixme: consolidate with near identical code in GenericStep and ASTBehaviorDefinition.
	// fixme: remove?
	/* private FieldDescriptor getAblEventSupportField() 
	{
	FieldDescriptor ablEventSupportField = new FieldDescriptor();
	String[] modifiers = {"private", "static"};
	
	ablEventSupportField.addFieldModifiers(modifiers);
	ablEventSupportField.fieldType = "AblEventSupport";
	ablEventSupportField.addFieldName("_eventSupport");
	return ablEventSupportField;
	}*/

	// fixme: consolidate with near identical code in GenericStep and ASTBehaviorDefinition.
	// fixme: remove?
	/* private CodeStringDescriptor getAblEventSupportFieldInit(String packageName, String className) 
	{
	String[] ablEventSupportFieldInitArgs = { packageName + "." + className };
	try {
	    return new CodeStringDescriptor(ablEventSupportFieldInitializer.expand(ablEventSupportFieldInitArgs));
	} catch (SimpleMacroException e) { 
	    throw new CompileError("Error compiling abl event support field initializer", e); 
	}
	}*/

	// fixme: consolidate with idential methods in ASTBehaviorDefinition and GenericStep
	// fixme: remove?
	/* private MethodDescriptor getAddAblListener()
	{
	MethodDescriptor addAblListener = new MethodDescriptor();
	addAblListener.methodName = "addAblListener";
	addAblListener.addModifier("public");
	addAblListener.addModifier("void");
	addAblListener.addArgument("AblListener", "lis");
	try {
	    addAblListener.addToBlockBody(new CodeStringDescriptor(addAblListenerBody.expand(null)));
	} catch (SimpleMacroException e) { throw new CompileError("Error compiling addAblListener ", e); }
	return addAblListener;
	}*/

	// fixme: consolidate with idential methods in ASTBehaviorDefinition and GenericStep
	// fixme: remove?
	/* private MethodDescriptor getRemoveAblListener()
	{
	MethodDescriptor removeAblListener = new MethodDescriptor();
	removeAblListener.methodName = "removeAblListener";
	removeAblListener.addModifier("public");
	removeAblListener.addModifier("void");
	removeAblListener.addArgument("AblListener", "lis");
	try {
	    removeAblListener.addToBlockBody(new CodeStringDescriptor(removeAblListenerBody.expand(null)));
	} catch (SimpleMacroException e) { throw new CompileError("Error compiling removeAblListener ", e); }
	return removeAblListener;
	}*/

	void writeBehaviorFactory(CodeBlockDescriptor behaviorFactory, ASTBehaviorDefinition node) {
		behaviorFactoryStream.writeCode(behaviorFactory, node);
	}

	void writeContextConditionSensorActivation(
		CodeBlockDescriptor contextConditionSensorActivation,
		ASTBehaviorDefinition node) {
		contextConditionSensorFactoryStream.writeCode(contextConditionSensorActivation, node);
	}

	void writeContextCondition(CodeBlockDescriptor contextCondition, ASTBehaviorDefinition node) {
		contextConditionStream.writeCode(contextCondition, node);
	}

	void writeSuccessConditionSensorActivation(
		CodeBlockDescriptor successConditionSensorActivation,
		ASTBehaviorDefinition node) {
		successConditionSensorFactoryStream.writeCode(successConditionSensorActivation, node);
	}
	
	void writeSuccessCondition(CodeBlockDescriptor successCondition, ASTBehaviorDefinition node) {
		successConditionStream.writeCode(successCondition, node);
	}
	
	void writeSuccessTestSensorActivation(CodeBlockDescriptor successTestSensorActivation, GenericStep node) {
		successTestSensorFactoryStream.writeCode(successTestSensorActivation, node);
	}

	void writeSuccessTest(CodeBlockDescriptor successTest, GenericStep node) {
		successTestStream.writeCode(successTest, node);
	}

	void writeReinforcementTestSensorActivation(CodeBlockDescriptor reinforcementTestSensorActivation, ASTBehaviorDefinition node) {
		reinforcementTestSensorFactoryStream.writeCode(reinforcementTestSensorActivation, node);
	}
	
	void writeReinforcementTest(CodeBlockDescriptor reinforcementTest, ASTBehaviorDefinition node) {
		reinforcementTestStream.writeCode(reinforcementTest, node);
	}

	void writeRLPolicy(CodeBlockDescriptor rlPolicy, ASTBehaviorDefinition node) {
		rlPolicyStream.writeCode(rlPolicy, node);
	}

	void writeRLPolicySuffix(CodeStringDescriptor suffix) {
		rlPolicySuffix.addToBlockBody(suffix);
	}

	void writeStateWMESensorActivation(CodeBlockDescriptor stateWMESensorActivation, ASTBehaviorDefinition node) {
		stateWMESensorFactoryStream.writeCode(stateWMESensorActivation, node);
	}
	
	void writeStateWME(CodeBlockDescriptor stateWME, ASTBehaviorDefinition node) {
		stateWMEStream.writeCode(stateWME, node);
	}
	
	void writePreconditionSensorActivation(
		CodeBlockDescriptor preconditionSensorActivation,
		ASTBehaviorDefinition node) {
		preconditionSensorFactoryStream.writeCode(preconditionSensorActivation, node);
	}

	void writePrecondition(CodeBlockDescriptor precondition, ASTBehaviorDefinition node) {
		preconditionStream.writeCode(precondition, node);
	}

	void writeStepFactory(CodeBlockDescriptor stepFactory, GenericStep node) {
		stepFactoryStream.writeCode(stepFactory, node);
	}

	void writeArgumentStepExecute(CodeBlockDescriptor execute, ArgumentStep node) {
		argumentStepExecuteStream.writeCode(execute, node);
	}

	void writeMentalStepExecute(CodeBlockDescriptor execute, ASTMentalStep node) {
		mentalStepExecuteStream.writeCode(execute, node);
	}

	void writeRawClass(CodeBlockDescriptor classCode, String className) {
		final PrintStream codeStream;
		try {
		    File packageDir = getFullPackagePath(Abl.objectDirectory,behavingEntityPackage);
			codeStream = new PrintStream(new BufferedOutputStream(new FileOutputStream(
                             new File(packageDir, className + ".java")),100000));
			codeStream.println("package " + behavingEntityPackage + ";");
			codeStream.println("");
			for (int i = 0; i < importPackages.length; i++)
				codeStream.println("import " + importPackages[i] + ";");
			codeStream.print(classCode.toString(3));

			codeStream.close();
		} catch (IOException e) {
			throw new CompileError("Error initing stream " + className, e);
		}
	}

	int getUniqueBehaviorID() {
		return behaviorIDCounter++;
	}
	int getUniqueStepID() {
		return stepIDCounter++;
	}

	int getSpecialStepID(int stepType) {
		switch (stepType) {
			case DEFAULT_WAIT_STEP :
				return DEFAULT_WAIT_STEP_ID;
			case DEFAULT_FAIL_STEP :
				return DEFAULT_FAIL_STEP_ID;
			case DEFAULT_SUCCEED_STEP :
				return DEFAULT_SUCCEED_STEP_ID;
			default :
				throw new CompileError("Unexpected step type " + stepType);
		}
	}

	protected String getCodeMethodReflectionField(String codeMethodName) {
		try {
			String[] macroArgs = { codeMethodName };
			return codeMethodReflectionFieldName.expand(macroArgs);
		} catch (SimpleMacroException e) {
			throw new CompileError("Error generating code method reflection field name", e);
		}
	}

	private void initializeArgumentFields(ClassDescriptor behavingEntity) {

		CodeBlockDescriptor staticInit =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");

		FieldDescriptor tempObjArray = new FieldDescriptor();
		tempObjArray.fieldType = "Object[]";
		tempObjArray.addFieldModifiers(privateMods);
		tempObjArray.addFieldName("__$tempObjArray");
		tempObjArray.initializer = "new Object[1]";
		behavingEntity.addField(tempObjArray);

		FieldDescriptor sensorFactoryArgArray = new FieldDescriptor();
		sensorFactoryArgArray.fieldType = "Class[]";
		sensorFactoryArgArray.addFieldModifiers(privateMods);
		sensorFactoryArgArray.addFieldName("__$sensorFactoryArgArray");
		sensorFactoryArgArray.initializer = "new Class[1]";
		behavingEntity.addField(sensorFactoryArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$sensorFactoryArgArray[0] = Integer.TYPE;"));

		FieldDescriptor behFactoryArgArray = new FieldDescriptor();
		behFactoryArgArray.fieldType = "Class[]";
		behFactoryArgArray.addFieldModifiers(privateMods);
		behFactoryArgArray.addFieldName("__$behFactoryArgArray");
		if (Abl.debugLevel == GUI_DEBUGGER)
			behFactoryArgArray.initializer = "new Class[6]";
		else
			behFactoryArgArray.initializer = "new Class[5]";
		behavingEntity.addField(behFactoryArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$behFactoryArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(new CodeStringDescriptor("__$behFactoryArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$behFactoryArgArray[2] = Class.forName(\"java.util.Hashtable\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$behFactoryArgArray[3] = Class.forName(\"abl.runtime.GoalStep\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$behFactoryArgArray[4] = Class.forName(\"java.lang.String\");"));
		if (Abl.debugLevel == GUI_DEBUGGER)
			staticInit.addToBlockBody(
				new CodeStringDescriptor("__$behFactoryArgArray[5] = Class.forName(\"abl.runtime.__BehaviorDesc\");"));

		FieldDescriptor preconditionArgArray = new FieldDescriptor();
		preconditionArgArray.fieldType = "Class[]";
		preconditionArgArray.addFieldModifiers(privateMods);
		preconditionArgArray.addFieldName("__$preconditionArgArray");
		preconditionArgArray.initializer = "new Class[4]";
		behavingEntity.addField(preconditionArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$preconditionArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(new CodeStringDescriptor("__$preconditionArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$preconditionArgArray[2] = Class.forName(\"java.util.Hashtable\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$preconditionArgArray[3] = Class.forName(\"abl.runtime.BehavingEntity\");"));

		FieldDescriptor continuousConditionArgArray = new FieldDescriptor();
		continuousConditionArgArray.fieldType = "Class[]";
		continuousConditionArgArray.addFieldModifiers(privateMods);
		continuousConditionArgArray.addFieldName("__$continuousConditionArgArray");
		continuousConditionArgArray.initializer = "new Class[3]";
		behavingEntity.addField(continuousConditionArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$continuousConditionArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$continuousConditionArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$continuousConditionArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));

		FieldDescriptor stepFactoryArgArray = new FieldDescriptor();
		stepFactoryArgArray.fieldType = "Class[]";
		stepFactoryArgArray.addFieldModifiers(privateMods);
		stepFactoryArgArray.addFieldName("__$stepFactoryArgArray");
		stepFactoryArgArray.initializer = "new Class[3]";
		behavingEntity.addField(stepFactoryArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$stepFactoryArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$stepFactoryArgArray[1] = Class.forName(\"abl.runtime.Behavior\");"));
		staticInit.addToBlockBody(new CodeStringDescriptor("__$stepFactoryArgArray[2] = __$tempObjArray.getClass();"));

		FieldDescriptor argumentStepExecuteArgArray = new FieldDescriptor();
		argumentStepExecuteArgArray.fieldType = "Class[]";
		argumentStepExecuteArgArray.addFieldModifiers(privateMods);
		argumentStepExecuteArgArray.addFieldName("__$argumentStepExecuteArgArray");
		argumentStepExecuteArgArray.initializer = "new Class[3]";
		behavingEntity.addField(argumentStepExecuteArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$argumentStepExecuteArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$argumentStepExecuteArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$argumentStepExecuteArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));

		FieldDescriptor mentalStepExecuteArgArray = new FieldDescriptor();
		mentalStepExecuteArgArray.fieldType = "Class[]";
		mentalStepExecuteArgArray.addFieldModifiers(privateMods);
		mentalStepExecuteArgArray.addFieldName("__$mentalStepExecuteArgArray");
		mentalStepExecuteArgArray.initializer = "new Class[4]";
		behavingEntity.addField(mentalStepExecuteArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$mentalStepExecuteArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$mentalStepExecuteArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$mentalStepExecuteArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$mentalStepExecuteArgArray[3] = Class.forName(\"abl.runtime.MentalStep\");"));

		FieldDescriptor reinforcementSignalArgArray = new FieldDescriptor();
		reinforcementSignalArgArray.fieldType = "Class[]";
		reinforcementSignalArgArray.addFieldModifiers(privateMods);
		reinforcementSignalArgArray.addFieldName("__$reinforcementSignalArgArray");
		reinforcementSignalArgArray.initializer = "new Class[3]";
		behavingEntity.addField(reinforcementSignalArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$reinforcementSignalArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$reinforcementSignalArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$reinforcementSignalArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));
		
		FieldDescriptor rlPolicyArgArray = new FieldDescriptor();
		rlPolicyArgArray.fieldType = "Class[]";
		rlPolicyArgArray.addFieldModifiers(privateMods);
		rlPolicyArgArray.addFieldName("__$rlPolicyArgArray");
		rlPolicyArgArray.initializer = "new Class[6]";
		behavingEntity.addField(rlPolicyArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$rlPolicyArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$rlPolicyArgArray[1] = Class.forName(\"java.lang.Object\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$rlPolicyArgArray[2] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$rlPolicyArgArray[3] = Class.forName(\"java.lang.Object\");"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$rlPolicyArgArray[4] = Double.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$rlPolicyArgArray[5] = Boolean.TYPE;"));
		
		FieldDescriptor stateWMEArgArray = new FieldDescriptor();
		stateWMEArgArray.fieldType = "Class[]";
		stateWMEArgArray.addFieldModifiers(privateMods);
		stateWMEArgArray.addFieldName("__$stateWMEArgArray");
		stateWMEArgArray.initializer = "new Class[3]";
		behavingEntity.addField(stateWMEArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$stateWMEArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$stateWMEArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$stateWMEArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));
		
		FieldDescriptor successConditionArgArray = new FieldDescriptor();
		successConditionArgArray.fieldType = "Class[]";
		successConditionArgArray.addFieldModifiers(privateMods);
		successConditionArgArray.addFieldName("__$successConditionArgArray");
		successConditionArgArray.initializer = "new Class[3]";
		behavingEntity.addField(successConditionArgArray);
		staticInit.addToBlockBody(new CodeStringDescriptor("__$successConditionArgArray[0] = Integer.TYPE;"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$successConditionArgArray[1] = __$tempObjArray.getClass();"));
		staticInit.addToBlockBody(
			new CodeStringDescriptor("__$successConditionArgArray[2] = Class.forName(\"abl.runtime.BehavingEntity\");"));
		
		behavingEntity.addStaticBlock(new CodeStringDescriptor(staticInit.toString()));
	}

	// Constructs the static initializer. Initializes reflection fields.
	private void compileEntityStaticInitializer(ClassDescriptor behavingEntity) {
		initializeArgumentFields(behavingEntity);

		CodeBlockDescriptor tryBlock;

		// Initialize reflection fields
		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(behaviorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(preconditionStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(preconditionSensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(contextConditionStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(contextConditionSensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(stepFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(argumentStepExecuteStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(mentalStepExecuteStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(successTestStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
				"try {",
				"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(successTestSensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));

		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(reinforcementTestStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(reinforcementTestSensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(rlPolicyStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(stateWMEStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(stateWMESensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(successConditionStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
		tryBlock =
			new CodeBlockDescriptor(
			"try {",
			"} catch (Exception e) { throw new AblRuntimeError(\"Error in static initializer\", e); }");
		tryBlock.addToBlockBody(successConditionSensorFactoryStream.compileReflectionFields(behavingEntity));
		behavingEntity.addStaticBlock(new CodeStringDescriptor(tryBlock.toString()));
		
	}

	private int initConflictSetMethodCounter = 0;
	private MethodDescriptor defineInitConflictSet() {
		MethodDescriptor initConflictSet = new MethodDescriptor();
		initConflictSet.addModifier("private");
		initConflictSet.addModifier("static");
		initConflictSet.addModifier("void");
		initConflictSet.methodName = "__$initConflictSet" + initConflictSetMethodCounter++;
		return initConflictSet;
	}

	// Defines the conflict fields on behaving entity. 
	private void compileConflictSets(ClassDescriptor behavingEntity) {
		if (conflictSets.size() > 0) {
			MethodDescriptor initConflictSet = defineInitConflictSet();
			final Iterator<List<String>> sets = conflictSets.iterator();
			while (sets.hasNext()) {
				final List<String> conflictSet = sets.next();
				final String fieldName = conflictSetsToFieldNames.get(conflictSet);
				final FieldDescriptor f = new FieldDescriptor();
				f.fieldType = "String[]";
				// Field should be final, but can't initialize it in non-constructor methods
				// f.addFieldModifier("final");
				f.addFieldModifier("static");
				f.addFieldName(fieldName);
				behavingEntity.addField(f);

				final StringBuffer buf = new StringBuffer(4096);
				buf.append("{");
				for (int i = 0; i < conflictSet.size() - 1; i++)
					buf.append("\"" + conflictSet.get(i) + "\"" + ", ");
				buf.append("\"" + conflictSet.get(conflictSet.size() - 1) + "\"" + "};");

				initConflictSet.addToBlockBody(
					new CodeStringDescriptor(fieldName + " = new String[]" + buf.toString()));
				if (initConflictSet.toString().length() > INIT_CONFLICT_SET_METHOD_SIZE) {
					behavingEntity.addMethod(initConflictSet);
					initConflictSet = defineInitConflictSet();
				}
			}
			behavingEntity.addMethod(initConflictSet);
		}
	}

	public void compileToJava() throws CompileException {
		// Initialize the global variable scope.
		scopeName = behavingEntityClass;
		// Set the name for this variable scope.
		setScopeType(AblScopeMaintainer.GLOBAL_SCOPE); // Set the scope type.
		initializeScope();
		// Process all the children ASTAblVariableDecl nodes.
		addChildVariableDeclarations();

		// Process WME registrations. 
		processWMERegistration();

		// Process the action registrations.
		processActionRegistration();

		// process the property declarations
		processPropertyDeclarations();

		// create the output directory so the streams can be initialized
		getFullPackagePath(Abl.objectDirectory,behavingEntityPackage).mkdirs();
		
		// init the various streams
		behaviorFactoryStream =
			new CodeStream(
				behavingEntityClass + "_BehaviorFactories",
				CodeStream.CODE_TYPE_BEHAVIOR_FACTORY,
				BEHAVIOR_FACTORY_LIMIT);
		preconditionSensorFactoryStream =
			new CodeStream(
				behavingEntityClass + "_PreconditionSensorFactories",
				CodeStream.CODE_TYPE_PRECONDITION_SENSOR_FACTORY,
				PRECONDITION_SENSOR_FACTORY_LIMIT);
		contextConditionSensorFactoryStream =
			new CodeStream(
				behavingEntityClass + "_ContextConditionSensorFactories",
				CodeStream.CODE_TYPE_CONTEXT_CONDITION_SENSOR_FACTORY,
				CONTEXT_CONDITION_SENSOR_FACTORY_LIMIT);
		contextConditionStream =
			new CodeStream(
				behavingEntityClass + "_ContextConditions",
				CodeStream.CODE_TYPE_CONTEXT_CONDITION,
				CONTEXT_CONDITION_LIMIT);
		preconditionStream =
			new CodeStream(
				behavingEntityClass + "_Preconditions",
				CodeStream.CODE_TYPE_PRECONDITION,
				PRECONDITION_LIMIT);

		stepFactoryStream =
			new CodeStream(
				behavingEntityClass + "_StepFactories",
				CodeStream.CODE_TYPE_STEP_FACTORY,
				STEP_FACTORY_LIMIT);
		// Write factory cases for the default steps. 
		if (Abl.debugLevel == GUI_DEBUGGER) {
			stepFactoryStream.writeCode(DEFAULT_WAIT_STEP_FACTORY_DEBUG, null);
			stepFactoryStream.writeCode(DEFAULT_FAIL_STEP_FACTORY_DEBUG, null);
			stepFactoryStream.writeCode(DEFAULT_SUCCEED_STEP_FACTORY_DEBUG, null);
		} else {
			stepFactoryStream.writeCode(DEFAULT_WAIT_STEP_FACTORY, null);
			stepFactoryStream.writeCode(DEFAULT_FAIL_STEP_FACTORY, null);
			stepFactoryStream.writeCode(DEFAULT_SUCCEED_STEP_FACTORY, null);
		}

		argumentStepExecuteStream =
			new CodeStream(
				behavingEntityClass + "_ArgumentStepExecute",
				CodeStream.CODE_TYPE_ARGUMENT_STEP_EXECUTE,
				STEP_EXECUTE_LIMIT);

		mentalStepExecuteStream =
			new CodeStream(
				behavingEntityClass + "_MentalStepExecute",
				CodeStream.CODE_TYPE_MENTAL_STEP_EXECUTE,
				STEP_EXECUTE_LIMIT);

		successTestStream =
			new CodeStream(
				behavingEntityClass + "_SuccessTests",
				CodeStream.CODE_TYPE_SUCCESS_TEST,
				SUCCESS_TEST_LIMIT);
		successTestSensorFactoryStream =
			new CodeStream(
				behavingEntityClass + "_SuccessTestSensorFactories",
				CodeStream.CODE_TYPE_SUCCESS_TEST_SENSOR_FACTORY,
				SUCCESS_TEST_SENSOR_FACTORY_LIMIT);

		reinforcementTestStream =
			new CodeStream(
			behavingEntityClass + "_ReinforcementSignals",
			CodeStream.CODE_TYPE_REINFORCEMENT_TEST,
			REINFORCEMENT_TEST_LIMIT);
		reinforcementTestSensorFactoryStream =
			new CodeStream(
			behavingEntityClass + "_ReinforcementSignalSensorFactories",
			CodeStream.CODE_TYPE_REINFORCEMENT_TEST_SENSOR_FACTORY,
			REINFORCEMENT_TEST_SENSOR_FACTORY_LIMIT);

		rlPolicyStream = 
			new CodeStream(
			behavingEntityClass + "_RLPolicy",
			CodeStream.CODE_TYPE_RL_POLICY,
			1);
		rlPolicySuffix =
			new CodeBlockDescriptor("","");
		
		stateWMEStream =
			new CodeStream(
			behavingEntityClass + "_StateWMEs",
			CodeStream.CODE_TYPE_STATE_WME,
			STATE_WME_LIMIT);
		stateWMESensorFactoryStream =
			new CodeStream(
			behavingEntityClass + "_StateWMESensorFactories",
			CodeStream.CODE_TYPE_STATE_WME_SENSOR_FACTORY,
			STATE_WME_SENSOR_FACTORY_LIMIT);
		
		successConditionSensorFactoryStream =
			new CodeStream(
			behavingEntityClass + "_SuccessConditionSensorFactories",
			CodeStream.CODE_TYPE_SUCCESS_CONDITION_SENSOR_FACTORY,
			SUCCESS_CONDITION_SENSOR_FACTORY_LIMIT);
		successConditionStream =
			new CodeStream(
			behavingEntityClass + "_SuccessConditions",
			CodeStream.CODE_TYPE_SUCCESS_CONDITION,
			SUCCESS_CONDITION_LIMIT);

		// Recursively compile children.
		compileChildren();

        // Added by Gillian Smith, 07-02-10
        // Build a behavior signature table, and then use it to typecheck all subgoals (incl. inheritance)
        ArrayList<Hashtable<Integer, HashSet<String>>> signatures = buildBehaviorSignatureIndices();

		// Check for subgoal steps which have goal signatures not matching any behavior signature.
		// Removed by Gillian Smith, 07-22-10; type checking now done with inheritance using behavior signature index
        //checkForSubgoalsWithNoMatchingBehaviors();

		// Finally, construct the class descriptor for the behaving entity. 
		ClassDescriptor behavingEntity = new ClassDescriptor();

		behavingEntity.className = behavingEntityClass;

		for (int i = 0; i < importPackages.length; i++)
			behavingEntity.addPackageImport(importPackages[i]);
		addUserImports(behavingEntity);
		addUserConstantDeclarations(behavingEntity);

		behavingEntity.packageName = behavingEntityPackage;
		behavingEntity.addClassModifier("public");
		behavingEntity.extendsClass = "BehavingEntity";

		// Add the fields stored in the AblScopeMaintainer. 
		final Iterator<FieldDescriptor> iter = fieldDescriptors.listIterator();
		while (iter.hasNext())
			behavingEntity.addField(iter.next());

		compileEntityStaticInitializer(behavingEntity);

		compileConflictSets(behavingEntity);

		// Define the behavior registration methods
		final Iterator<ASTBehaviorDefinition> individualBehaviorIter = individualBehaviorDefinitionNodes.iterator();
		int registrationMethodCounter = 0;
		for (int i = 0;
			i < individualBehaviorDefinitionNodes.size() / REGISTRATION_SIZE + 1;
			i++, registrationMethodCounter++) {
			behavingEntity.addMethod(compileRegistrationMethod(registrationMethodCounter, signatures.get(0), individualBehaviorIter));
		}

		final Iterator<ASTBehaviorDefinition> jointBehaviorIter = jointBehaviorDefinitionNodes.iterator();
		for (int i = 0;
			i < jointBehaviorDefinitionNodes.size() / REGISTRATION_SIZE + 1;
			i++, registrationMethodCounter++) {
			behavingEntity.addMethod(compileRegistrationMethod(registrationMethodCounter, signatures.get(1), jointBehaviorIter));
		}

		// Define the decision cycle sm for behaving entity. 
		MethodDescriptor decisionCycleSMCall = compileDecisionCycleSMCall();
		if (decisionCycleSMCall != null)
			behavingEntity.addMethod(decisionCycleSMCall);

		// fixme: whether an explicit entry point is produced
		// or not should be a command line parameter Define
		// the entry point (main) for the behaving entity.
                //
		// UPDATE: (sooraj 7/30/06) The only reason for not
		// allowing it as a default is so that agents that
		// need extra [sensorimotor] initiliazation don't get
		// launched inappropriately.  I think it is a handly
		// enough feature to be compiled in by default
		behavingEntity.addMethod(compileEntryPoint());

		// Compile the constructor for the behaving entity. 
		behavingEntity.addMethod(compileConstructor());

		behavingEntity.writeToFile(getFullPackagePath(Abl.objectDirectory,behavingEntityPackage));

		behaviorFactoryStream.closeCode();
		preconditionSensorFactoryStream.closeCode();
		contextConditionSensorFactoryStream.closeCode();
		contextConditionStream.closeCode();
		successConditionSensorFactoryStream.closeCode();
		successConditionStream.closeCode();
		preconditionStream.closeCode();
		stepFactoryStream.closeCode();
		argumentStepExecuteStream.closeCode();
		mentalStepExecuteStream.closeCode();
		successTestStream.closeCode();
		successTestSensorFactoryStream.closeCode();
		reinforcementTestStream.closeCode();
		reinforcementTestSensorFactoryStream.closeCode();
		rlPolicyStream.closeCode(rlPolicySuffix);
		stateWMEStream.closeCode();
		stateWMESensorFactoryStream.closeCode();
	}



    // If the conflict set is new, add it to the table of conflict sets and return a new conflict set name.
	// Otherwise, return the existing conflict set name.
	String addConflictSet(String[] conflictArray) {
		List<String> conflictList = Arrays.asList(conflictArray);
		// Use Lists in conflictSets for List equality
		if (conflictSets.contains(conflictList)) {
			// this conflict set has already appeared - return the field name
			assert(conflictSetsToFieldNames.get(conflictList) != null);
			return conflictSetsToFieldNames.get(conflictList);
		} else {
			// conflict set has not appeared before
			assert(conflictSetsToFieldNames.get(conflictList) == null);
			conflictSets.add(conflictList);
			String fieldName = "__$conflictSet" + conflictSets.size();
			conflictSetsToFieldNames.put(conflictList, fieldName);
			return fieldName;
		}
	}

    File getFullPackagePath(File baseDir, String packageName) {
	return new File(baseDir,packageName.replace('.',File.separatorChar));
    }
}
