/* Parse node for ABL Behavior definitions. */

package abl.compiler;

import java.util.*;

import jd.CodeBlockDescriptor;
import jd.CodeStringDescriptor;
import jd.FieldDescriptor;
import jd.MethodArgDescriptor;
import jd.MethodArglistDescriptor;

public class ASTBehaviorDefinition
        extends AblScopeMaintainer
        implements AblDebuggerConstants {

    /* The behavior type: sequential, parallel, collection. */
    private String behaviorType = null;

    // The behavior name.
    String behaviorName;

    // Set to true if this behavior represents the initial_tree.
    private boolean initialTree = false;

    // behaviorName + _<num> where <num> counts the number of behviors sharing the behavior name
    private String uniqueBehaviorName;

    // a unique integer ID for this behavior; used in the switch cases of static methods.
    private int behaviorID;

    // Set to true by the parser if the behavior has a context condition.
    private boolean hasContextCondition = false;

    // Set to true by the parser if teh behavior has a precondition.
    private boolean hasPrecondition = false;

    // Set to true by the parser if the behavior has an entry condition.
    private boolean hasEntryCondition = false;

    // Set to true by the parser if the behavior has a success condition.
    private boolean hasSuccessCondition = false;
    private boolean hasReinforcementSignals = false;
    private boolean hasStateCondition = false;

    // Set to the behavior specificity by the ABL parser.
    int specificity = 0;

    // The number of steps needed for a collection or parallel behavior to succeed.
    int numberNeededForSuccess = -1;

    // Set to true if the behavior is declared to be an atomic behavior - default false.
    boolean isAtomic = false;

    // Set to true if the behavior is declared to be a joint behavior - default false.
    private boolean isJoint = false;

    // Set to true if the behavior is declared to be an adaptive behavior - default false.
    private boolean isAdaptive = false;

    private MethodArglistDescriptor formalArgs = new MethodArglistDescriptor();

    // List of role specifiers (Strings) for a joint behavior's team members
    private List<String> teamMembers = new ArrayList<String>();

    // List of stepIDs for steps of this behavior.
    // This list is in one-to-one correspondance with stepFactories.
    private List<Integer> stepIDs = new ArrayList<Integer>(100);

    // List of reflection fields of factory methods for the steps of this behavior.
    // This list is in one-to-one correspondance with stepIDs.
    private List<String> stepFactories = new ArrayList<String>(100);

    // Counter of non-effect-only steps - used to set the numberNeededForSuccess
    private int nonEffectOnlySteps = 0;

    private String behaviorFactoryClass;
    // Name of the code class containing the behavior factory for this behavior.
    private String behaviorFactoryMethod_rField;
    // Reflection field of behavior factory containing the case for this behavior.
    private String preconditionSensorFactoryClass;
    // Name of the code class containing the precondition sensor factory for this behavior.
    private String preconditionSensorFactoryMethod_rField;
    // Reflection field of precondition sensor factory containing the case for this behavior.
    private String preconditionClass;
    // Name of the code class containing the precondition for this behavior.
    private String preconditionMethod_rField;
    // Reflection field of precondition containing the case for this behavior.
    private String contextConditionSensorFactoryClass;
    // Name of the code class containing the CC sensor factory for this beh.
    private String contextConditionSensorFactoryMethod_rField;
    // Reflection field CC sensor factory containing the case for this beh.
    private String contextConditionClass;
    // Name of the code class containing the context condition for this beh.
    private String contextConditionMethod_rField;
    // Reflection field of context condition containing the case for this behavior.
    private String successConditionSensorFactoryClass;
    // Name of the code class containing the CC sensor factory for this beh.
    private String successConditionSensorFactoryMethod_rField;
    // Reflection field CC sensor factory containing the case for this beh.
    private String successConditionClass;
    // Name of the code class containing the success condition for this beh.
    private String successConditionMethod_rField;
    // Reflection field of success condition containing the case for this beh.
    private String successTestSensorFactoryClass;
    // Name of the code class containing the ST sensor factory for this beh.
    private String successTestSensorFactoryMethod_rField;
    // Reflection field ST sensor factory containing the case for this beh.
    private String successTestClass;
    // Name of the code class containing the success test for this beh.
    private String successTestMethod_rField;
    // Reflection field of success test containing the case for this beh.
    private String reinforcementSignalsSensorFactoryClass;
    // Name of the code class containing the CC sensor factory for this beh.
    private String reinforcementSignalsSensorFactoryMethod_rField;
    // Reflection field CC sensor factory containing the case for this beh.
    private String reinforcementSignalsClass;
    // Name of the code class containing the reinforcement signals for this beh.
    private String reinforcementSignalsMethod_rField;
    // Reflection field of reinforcement signals containing the case for this beh.
    private String rlPolicyClass;
    // Name of the code class containing the RL Policy for this beh.
    private String rlPolicyMethod_rField;
    // Reflection field of RL Policy containing the case for this beh.
    private String stateWMESensorFactoryClass;
    // Name of the code class containing the CC sensor factory for this beh.
    private String stateWMESensorFactoryMethod_rField;
    // Reflection field CC sensor factory containing the case for this beh.

    // fixme: remove - this field is never read locally - why do we need it?
    private String stateWMEClass;
    // Name of the code class containing the state wmes for this beh.

    private String stateWMEMethod_rField;
    // Reflection field of state wmes containing the case for this beh.

    /* Counter of the compiled steps of this behavior. Used during
        step compilation to construct the step name. */
    private int stepCounter = 0;

    ASTBehaviorDefinition(int id) {
        super(id);
    }

    ASTBehaviorDefinition(AblParser p, int id) {
        super(p, id);
    }

    public boolean isInitialTree() {
        return initialTree;
    }

    public String getBehaviorName() {
        return behaviorName;
    }

    // package accessors for isJoint

    boolean isJoint() {
        return isJoint;
    }

    void setIsJoint() {
        isJoint = true;
    }

    /**
     * Package accessor for setting initialTree to true.
     */
    void setInitialTree() {
        initialTree = true;
        setBehaviorType("collection");
        behaviorName = "__RootCollectionBehavior";
    }

    // package accessors for isAdaptive

    boolean isAdaptive() {
        return isAdaptive;
    }

    void setIsAdaptive() {
        isAdaptive = true;
    }

    void addRole(String role) {
        if (!isJoint)
            throw new CompileError(
                    "Attempt to add a role specifier to a non-joint behavior "
                            + getSignature());
        else
            teamMembers.add(role);
    }

    List getTeamMembers() {
        return teamMembers;
    }

    boolean teamMembersSpecified() {
        if (!teamMembers.isEmpty())
            return true;
        else
            return false;
    }

    void setBehaviorType(String behaviorTypeToSet) {
        if ((behaviorTypeToSet.equals("sequential"))
                || (behaviorTypeToSet.equals("parallel"))
                || (behaviorTypeToSet.equals("collection")))
            behaviorType = behaviorTypeToSet;
        else
            throw new CompileError(
                    "Attempt to set illegal behavior type: " + behaviorTypeToSet);
    }

    String getBehaviorType() {
        return behaviorType;
    }

    void addFormalArgument(MethodArgDescriptor arg) {
        formalArgs.addArgument(arg);
    }

    Object[] getFormalArguments() {
        return formalArgs.getArguments();
    }

    // Package accessor for getting the concrete class name.

    String getUniqueName() {
        if (uniqueBehaviorName != null)
            return uniqueBehaviorName;
        else
            return behaviorName;
    }

    AblScopeMaintainer getEnclosingBehaviorScope() {
        return this;
    }

    void incrementNonEffectOnlySteps() {
        nonEffectOnlySteps++;
    }

    // Add the stepID for a step of this behavior. Called by child step nodes.

    void addStep(int stepID, String factory) {
        if ((stepID != ASTBehaviorUnit.DEFAULT_SUCCEED_STEP_ID)
                && (stepID != ASTBehaviorUnit.DEFAULT_FAIL_STEP_ID)
                && (stepID != ASTBehaviorUnit.DEFAULT_WAIT_STEP_ID)) {
            // not one of the default steps - add the (non-null) factory

            assert (factory != null);
            stepFactories.add(factory);
        } else {
            // one of the default steps - they are in the first factory method
            stepFactories.add("__$stepFactory0_rfield");
        }
        stepIDs.add(new Integer(stepID));
    }

    int getID() {
        return behaviorID;
    }

    boolean getHasPrecondition() {
        return hasPrecondition;
    }

    void setHasPrecondition(boolean arg_hasPrecondition) {
        hasPrecondition = arg_hasPrecondition;
    }

    boolean getHasContextCondition() {
        return hasContextCondition;
    }

    void setHasContextCondition(boolean arg_hasContextCondition) {
        hasContextCondition = arg_hasContextCondition;
    }

    boolean getHasEntryCondition() {
        return hasEntryCondition;
    }

    void setHasEntryCondition(boolean arg_hasEntryCondition) {
        hasEntryCondition = arg_hasEntryCondition;
    }

    boolean getHasSuccessCondition() {
        return hasSuccessCondition;
    }

    void setHasSuccessCondition(boolean arg_hasSuccessCondition) {
        hasSuccessCondition = arg_hasSuccessCondition;
    }

    boolean getHasReinforcementSignals() {
        return hasReinforcementSignals;
    }

    void setHasReinforcementSignals(boolean arg_hasReinforcementSignals) {
        hasReinforcementSignals = arg_hasReinforcementSignals;
    }

    boolean getHasStateCondition() {
        return hasStateCondition;
    }

    void setHasStateCondition(boolean arg_hasStateCondition) {
        hasStateCondition = arg_hasStateCondition;
    }

    /* Increments the step counter and returns the incremented
        value. Called by AST step nodes. */

    public int incrementStepCount() {
        return ++stepCounter;
    }

    String getSignature() {
        Object[] args = formalArgs.getArguments();
        StringBuffer signatureBuf = new StringBuffer();
        signatureBuf.append(behaviorName + "(");
        for (int i = 0; i < args.length; i++)
            signatureBuf.append(((MethodArgDescriptor) args[i]).argType + ", ");

        // Delete the final ", ".
        if (args.length != 0)
            signatureBuf.delete(
                    signatureBuf.length() - 2,
                    signatureBuf.length());

        signatureBuf.append(")");
        return signatureBuf.toString().intern();
    }

    String getBehaviorFactoryClass() {
        return behaviorFactoryClass;
    }

    String getBehaviorFactoryMethod_rField() {
        return behaviorFactoryMethod_rField;
    }

    String getPreconditionSensorFactoryClass() {
        return preconditionSensorFactoryClass;
    }

    String getPreconditionSensorFactoryMethod_rField() {
        return preconditionSensorFactoryMethod_rField;
    }

    String getPreconditionClass() {
        return preconditionClass;
    }

    String getPreconditionMethod_rField() {
        return preconditionMethod_rField;
    }

    String getContextConditionSensorFactoryClass() {
        return contextConditionSensorFactoryClass;
    }

    String getContextConditionSensorFactoryMethod_rField() {
        return contextConditionSensorFactoryMethod_rField;
    }

    String getContextConditionClass() {
        return contextConditionClass;
    }

    String getContextConditionMethod_rField() {
        return contextConditionMethod_rField;
    }

    String getSuccessConditionSensorFactoryClass() {
        return successConditionSensorFactoryClass;
    }

    String getSuccessConditionSensorFactoryMethod_rField() {
        return successConditionSensorFactoryMethod_rField;
    }

    String getSuccessConditionClass() {
        return successConditionClass;
    }

    String getSuccessConditionMethod_rField() {
        return successConditionMethod_rField;
    }

    String getSuccessTestSensorFactoryClass() {
        return successTestSensorFactoryClass;
    }

    String getSuccessTestSensorFactoryMethod_rField() {
        return successTestSensorFactoryMethod_rField;
    }

    String getSuccessTestClass() {
        return successTestClass;
    }

    String getSuccessTestMethod_rField() {
        return successTestMethod_rField;
    }

    String getReinforcementSignalsSensorFactoryClass() {
        return reinforcementSignalsSensorFactoryClass;
    }

    String getReinforcementSignalsSensorFactoryMethod_rField() {
        return reinforcementSignalsSensorFactoryMethod_rField;
    }

    String getReinforcementSignalsClass() {
        return reinforcementSignalsClass;
    }

    String getReinforcementSignalsMethod_rField() {
        return reinforcementSignalsMethod_rField;
    }

    void setBehaviorFactoryClass(String arg_behaviorFactoryClass) {
        if (behaviorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set behaviorFactoryClass multiple times: "
                            + behaviorFactoryClass);

        behaviorFactoryClass = arg_behaviorFactoryClass;
    }

    void setBehaviorFactoryMethod_rField(String arg_behaviorFactoryMethod_rField) {
        if (behaviorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set behaviorFactoryMethod_rField multiple times: "
                            + behaviorFactoryMethod_rField);

        behaviorFactoryMethod_rField = arg_behaviorFactoryMethod_rField;
    }

    void setPreconditionSensorFactoryClass(String arg_preconditionSensorFactoryClass) {
        if (preconditionSensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set preconditionSensorFactoryClass multiple times: "
                            + preconditionSensorFactoryClass);

        preconditionSensorFactoryClass = arg_preconditionSensorFactoryClass;
    }

    void setPreconditionSensorFactoryMethod_rField(String arg_preconditionSensorFactoryMethod_rField) {
        if (preconditionSensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set preconditionSensorFactoryMethod_rField multiple times: "
                            + preconditionSensorFactoryMethod_rField);

        preconditionSensorFactoryMethod_rField =
                arg_preconditionSensorFactoryMethod_rField;
    }

    void setPreconditionClass(String arg_preconditionClass) {
        if (preconditionClass != null)
            throw new CompileError(
                    "Attempt to set preconditionClass multiple times: "
                            + preconditionClass);

        preconditionClass = arg_preconditionClass;
    }

    void setPreconditionMethod_rField(String arg_preconditionMethod_rField) {
        if (preconditionMethod_rField != null)
            throw new CompileError(
                    "Attempt to set preconditionMethod_rField multiple times: "
                            + preconditionMethod_rField);

        preconditionMethod_rField = arg_preconditionMethod_rField;
    }

    void setContextConditionSensorFactoryClass(String arg_contextConditionSensorFactoryClass) {
        if (contextConditionSensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set contextConditionSensorFactoryClass multiple times: "
                            + contextConditionSensorFactoryClass);

        contextConditionSensorFactoryClass =
                arg_contextConditionSensorFactoryClass;
    }

    void setContextConditionSensorFactoryMethod_rField(String arg_contextConditionSensorFactoryMethod_rField) {
        if (contextConditionSensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set contextConditionSensorFactoryMethod_rField multiple times: "
                            + contextConditionSensorFactoryMethod_rField);

        contextConditionSensorFactoryMethod_rField =
                arg_contextConditionSensorFactoryMethod_rField;
    }

    void setContextConditionClass(String arg_contextConditionClass) {
        if (contextConditionClass != null)
            throw new CompileError(
                    "Attempt to set contextConditionClass multiple times: "
                            + contextConditionClass);

        contextConditionClass = arg_contextConditionClass;
    }

    void setContextConditionMethod_rField(String arg_contextConditionMethod_rField) {
        if (contextConditionMethod_rField != null)
            throw new CompileError(
                    "Attempt to set contextConditionMethod_rField multiple times: "
                            + contextConditionMethod_rField);

        contextConditionMethod_rField = arg_contextConditionMethod_rField;
    }

    void setSuccessConditionSensorFactoryClass(String arg_successConditionSensorFactoryClass) {
        if (successConditionSensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set successConditionSensorFactoryClass multiple times: "
                            + successConditionSensorFactoryClass);

        successConditionSensorFactoryClass =
                arg_successConditionSensorFactoryClass;
    }

    void setSuccessConditionSensorFactoryMethod_rField(String arg_successConditionSensorFactoryMethod_rField) {
        if (successConditionSensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set successConditionSensorFactoryMethod_rField multiple times: "
                            + successConditionSensorFactoryMethod_rField);

        successConditionSensorFactoryMethod_rField =
                arg_successConditionSensorFactoryMethod_rField;
    }

    void setSuccessConditionClass(String arg_successConditionClass) {
        if (successConditionClass != null)
            throw new CompileError(
                    "Attempt to set successConditionClass multiple times: "
                            + successConditionClass);

        successConditionClass = arg_successConditionClass;
    }

    void setSuccessConditionMethod_rField(String arg_successConditionMethod_rField) {
        if (successConditionMethod_rField != null)
            throw new CompileError(
                    "Attempt to set successConditionMethod_rField multiple times: "
                            + successConditionMethod_rField);

        successConditionMethod_rField = arg_successConditionMethod_rField;
    }

    void setSuccessTestSensorFactoryClass(String arg_successTestSensorFactoryClass) {
        if (successTestSensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set successTestSensorFactoryClass multiple times: "
                            + successTestSensorFactoryClass);

        successTestSensorFactoryClass = arg_successTestSensorFactoryClass;
    }

    void setSuccessTestSensorFactoryMethod_rField(String arg_successTestSensorFactoryMethod_rField) {
        if (successTestSensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set successTestSensorFactoryMethod_rField multiple times: "
                            + successTestSensorFactoryMethod_rField);

        successTestSensorFactoryMethod_rField =
                arg_successTestSensorFactoryMethod_rField;
    }

    void setSuccessTestClass(String arg_successTestClass) {
        if (successTestClass != null)
            throw new CompileError(
                    "Attempt to set successTestClass multiple times: "
                            + successTestClass);

        successTestClass = arg_successTestClass;
    }

    void setSuccessTestMethod_rField(String arg_successTestMethod_rField) {
        if (successTestMethod_rField != null)
            throw new CompileError(
                    "Attempt to set successTestMethod_rField multiple times: "
                            + successTestMethod_rField);

        successTestMethod_rField = arg_successTestMethod_rField;
    }

    void setReinforcementSignalsSensorFactoryClass(String arg_reinforcementSignalsSensorFactoryClass) {
        if (reinforcementSignalsSensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set reinforcementSignalsSensorFactoryClass multiple times: "
                            + reinforcementSignalsSensorFactoryClass);

        reinforcementSignalsSensorFactoryClass =
                arg_reinforcementSignalsSensorFactoryClass;
    }

    void setReinforcementSignalsSensorFactoryMethod_rField(String arg_reinforcementSignalsSensorFactoryMethod_rField) {
        if (reinforcementSignalsSensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set reinforcementSignalsSensorFactoryMethod_rField multiple times: "
                            + reinforcementSignalsSensorFactoryMethod_rField);

        reinforcementSignalsSensorFactoryMethod_rField =
                arg_reinforcementSignalsSensorFactoryMethod_rField;
    }

    void setReinforcementSignalsClass(String arg_reinforcementSignalsClass) {
        //if (reinforcementSignalsClass != null)
        //	throw new CompileError(
        //	"Attempt to set reinforcementSignalsClass multiple times: "
        //	+ reinforcementSignalsClass);

        reinforcementSignalsClass = arg_reinforcementSignalsClass;
    }

    void setReinforcementSignalsMethod_rField(String arg_reinforcementSignalsMethod_rField) {
        //if (reinforcementSignalsMethod_rField != null)
        //	throw new CompileError(
        //	"Attempt to set reinforcementSignalsMethod_rField multiple times: "
        //	+ reinforcementSignalsMethod_rField);

        reinforcementSignalsMethod_rField = arg_reinforcementSignalsMethod_rField;
    }

    void setRLPolicyClass(String arg_rlPolicyClass) {
        if (rlPolicyClass != null)
            throw new CompileError(
                    "Attempt to set rlPolicyClass multiple times: "
                            + rlPolicyClass);

        rlPolicyClass = arg_rlPolicyClass;
    }

    void setRLPolicyMethod_rField(String arg_rlPolicyMethod_rField) {
        if (rlPolicyMethod_rField != null)
            throw new CompileError(
                    "Attempt to set rlPolicyMethod_rField multiple times: "
                            + rlPolicyMethod_rField);

        rlPolicyMethod_rField = arg_rlPolicyMethod_rField;
    }

    void setStateWMESensorFactoryClass(String arg_stateWMESensorFactoryClass) {
        if (stateWMESensorFactoryClass != null)
            throw new CompileError(
                    "Attempt to set stateWMESensorFactoryClass multiple times: "
                            + stateWMESensorFactoryClass);

        stateWMESensorFactoryClass =
                arg_stateWMESensorFactoryClass;
    }

    void setStateWMESensorFactoryMethod_rField(String arg_stateWMESensorFactoryMethod_rField) {
        if (stateWMESensorFactoryMethod_rField != null)
            throw new CompileError(
                    "Attempt to set stateWMESensorFactoryMethod_rField multiple times: "
                            + stateWMESensorFactoryMethod_rField);

        stateWMESensorFactoryMethod_rField =
                arg_stateWMESensorFactoryMethod_rField;
    }

    // fixme: remove - since stateWMEClass is never read locally (and it's private), why do we ever need to set it?

    void setStateWMEClass(String arg_stateWMEClass) {
        //if (stateWMEClass != null)
        //	throw new CompileError(
        //	"Attempt to set stateWMEClass multiple times: "
        //	+ stateWMEClass);

        stateWMEClass = arg_stateWMEClass;
    }

    void setStateWMEMethod_rField(String arg_stateWMEMethod_rField) {
        //if (stateWMEMethod_rField != null)
        //	throw new CompileError(
        //	"Attempt to set stateWMEMethod_rField multiple times: "
        //	+ stateWMEMethod_rField);

        stateWMEMethod_rField = arg_stateWMEMethod_rField;
    }

    // fixme: Consolidate this code with the nearly identical code in ASTTestExpression.

    private String compileBehaviorArgument(String name, String type) {
        int i = formalArgs.getArgIndex(name);
        if (primitiveType(type)) {
            if (type.equals("int"))
                return "(Integer)__$args[" + i + "]";
            else if (type.equals("float"))
                return "(Float)__$args[" + i + "]";
            else if (type.equals("char"))
                return "(Char)__$args[" + i + "]";
            else if (type.equals("boolean"))
                return "(Boolean)__$args[" + i + "]";
            else if (type.equals("long"))
                return "(Long)__$args[" + i + "]";
            else if (type.equals("short"))
                return "(Short)__$args[" + i + "]";
            else if (type.equals("byte"))
                return "(Byte)__$args[" + i + "]";
            else if (type.equals("double"))
                return "(Double)__$args[" + i + "]";
            else
                throw new CompileError("Unexpected primitive type " + type);
        } else
            return "(" + type + ")__$args[" + i + "]";

    }

    /* If the behavior has a precondition, returns the set of
        variables bound in the precondition, otherwise returns an empty HashSet. */

    private HashSet getBoundVariables() {
        AblParseNode n;
        ASTTestExpression precondition = null;
        ASTTestExpression entrycondition = null;
        HashSet<String> returnSet = new HashSet<String>();

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for a context condition. 
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTTESTEXPRESSION) {
                ASTTestExpression testexp = (ASTTestExpression) n;
                // The child is a test expression; it's either a precondition or context condition. 
                if (testexp.getTestType() == ASTTestExpression.PRECONDITION) {
                    // Found the precondition.
                    precondition = testexp;
                } else if (testexp.getTestType() == ASTTestExpression.ENTRY_CONDITION) {
                    // Found the entry condition.
                    entrycondition = testexp;
                }
                /*
                if (   (!hasPrecondition || precondition != null)
                    && (!hasEntryCondition || entrycondition != null))
                    break;
                    */
            }
        }
        // note: this could be optimized to just return one of them right away if it's the only one
        if (precondition != null)
            /* A precondition has been defined on this behavior. */
            returnSet.addAll(precondition.getBoundVariables());
        if (entrycondition != null)
            /* An entry condition has been defined on this behavior. */
            returnSet.addAll(entrycondition.getBoundVariables());
        return returnSet;
    }

    // fixme: remove - never used - why is it defined?
    /* private Set<String> getEscapeVariables() {
        AblParseNode n;
        ASTTestExpression entrycondition = null;
        
        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for a context condition. 
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTTESTEXPRESSION)
                // The child is a test expression; it's either a precondition or context condition. 
                if (((ASTTestExpression) n).getTestType()
                    == ASTTestExpression.ENTRY_CONDITION) {
                    // Found the condition.
                    entrycondition = (ASTTestExpression) n;
                    break;
                }
        }
        if (entrycondition != null)
        {
            // An entry condition has been defined on this behavior.
            Set<String> varSet = entrycondition.getBoundVariables();
//System.out.println("Found " + varSet.size() + " escaping variables");
            return varSet;
        }
        else
            return new HashSet<String>();
    } */

    private CodeBlockDescriptor compileSensorActivationFactory(int testType) {
        AblParseNode n;
        ASTTestExpression test = null;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for a test of type testType
            n = (AblParseNode) jjtGetChild(i);
            if ((n.id == JJTTESTEXPRESSION)
                    && ((ASTTestExpression) n).getTestType() == testType)
                // Found the test type we're looking for
                test = (ASTTestExpression) n;
        }
        if (test != null) {
            // A test of type testType has been defined on this behavior
            CodeBlockDescriptor sensorArray =
                    test.compileSensorActivationFactory();
            if (sensorArray != null) {
                CodeBlockDescriptor factorySwitchCase =
                        new CodeBlockDescriptor("case " + behaviorID + ": {", "}");
                factorySwitchCase.addToBlockBody(sensorArray);
                return factorySwitchCase;
            }
        }
        return null;
    }

    // Compile all the steps of the behavior.

    private void compileSteps() throws CompileException {
        AblParseNode n;
        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for steps.
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTGOALSTEP
                    || n.id == JJTPRIMITIVESTEP
                    || n.id == JJTMENTALSTEP
                    || n.id == JJTWAITSTEP
                    || n.id == JJTFAILSTEP
                    || n.id == JJTSUCCEEDSTEP
                    || n.id == JJTMODIFYSTEP
                    || n.id == JJTANONYMOUSSTEP) {
                // If the child node is compileable, compile it.
                if (n.id == JJTANONYMOUSSTEP) {
                    String behType = ((ASTAnonymousStep) n).getBehaviorType();
                    if (behType.equals(this.behaviorType))
                        throw new CompileException(n.getFirstLineNumber(), "Anonymous Block has same behavior type as parent");
                }

                ((GenericStep) n).compileToJava();
            }
        }
    }

    // Compiles the precondition and context condition (if they exist)
    // Any variables referenced in the precondition or context condition that are not in scope are added to the behavior
    // scope.

    private void compileTests() throws CompileException {
        AblParseNode n;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for a precondition or context condition.
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTTESTEXPRESSION) {
                // The child is a test expression; it's either a precondition or context condition.
                // Or an entry condition, which can be implemented as a WaitStep
                ASTTestExpression testexp = (ASTTestExpression) n;
                testexp.compileToJava();
                if (testexp.getTestType() == ASTTestExpression.ENTRY_CONDITION) {
//System.out.println("EntryCondition compiling");
                    ASTWaitStep waitstep = new ASTWaitStep(JJTWAITSTEP);
                    waitstep.jjtAddChild(n, 0);
                    waitstep.hasSuccessTest = true;
                    waitstep.jjtSetParent(this);
                    testexp.jjtSetParent(waitstep);
                    testexp.setTestType(ASTTestExpression.SUCCESS_TEST);
                    waitstep.compileToJava();

                    // set the tree back to normal
                    testexp.jjtSetParent(this);
                    testexp.setTestType(ASTTestExpression.ENTRY_CONDITION);
                }
            }
        }
    }

    private void compileReinforcementStateAndSignals() throws CompileException {
        AblParseNode n;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for a precondition or context condition.
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTREINFORCEMENTSTATE) {
                ASTReinforcementState state = (ASTReinforcementState) n;
                state.compileToJava();
            } else if (n.id == JJTREINFORCEMENTSIGNALS) {
                ASTReinforcementSignals signals = (ASTReinforcementSignals) n;
                signals.compileToJava();
            }
        }
    }

    private void compileReinforcementPolicyAndAction() throws CompileException {
        CodeBlockDescriptor codeBlock;
        ASTBehaviorUnit behUnit = ASTBehaviorUnit.getBehaviorUnit();
        String actionClassName = behUnit.getBehavingEntityClass() + "_Action" + behaviorID;

        int numSteps = 0;
        String actionSteps = "";
        for (int i = 0; i < jjtGetNumChildren(); ++i) {
            AblParseNode n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTGOALSTEP) {
                if (numSteps > 0)
                    actionSteps += ",";
                numSteps += 1;
                ASTGoalStep goalStep = (ASTGoalStep) n;
                actionSteps += "\"" + goalStep.getSignature() + "\"";
            }
        }
        String actionNames = "";
        for (int i = 0; i < numSteps; ++i) {
            if (i > 0)
                actionNames += ", ";
            actionNames += "new " + actionClassName + "(" + i + ")";
        }
        String actionsVarName = "actions" + behaviorID;
        String learnerVarName = "learner" + behaviorID;

        codeBlock = new CodeBlockDescriptor("public class " + actionClassName + " implements Action {", "}");
        codeBlock.addToBlockBody(new CodeStringDescriptor("Integer stepIndex;"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("static String[] names = new String[] {" + actionSteps + "};"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("public " + actionClassName + "(int stepIndex) {"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("	this.stepIndex = new Integer(stepIndex);}"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("public int hashCode() {"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("	return stepIndex.hashCode();}"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("public boolean equals(Object o) {"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("	return (o instanceof " + actionClassName + ") ? stepIndex.equals( ((" + actionClassName + ")o).stepIndex ) : false;}"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("public String toString() {"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("	return names[stepIndex.intValue()];}"));
        behUnit.writeRawClass(codeBlock, actionClassName);

        codeBlock = new CodeBlockDescriptor("case " + behaviorID + ": {", "}");
        codeBlock.addToBlockBody(new CodeStringDescriptor("State prevState = (State)prevLocation;"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("if (isRewarding) { "));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    State currState = (State)currLocation;"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    Action prevAction = new " + actionClassName + "(stepIndex);"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    " + learnerVarName + ".update(prevState,prevAction,rewardValue,currState); "));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    return 0; "));
        codeBlock.addToBlockBody(new CodeStringDescriptor("}"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("else {"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    " + actionClassName + " a = (" + actionClassName + ")" + learnerVarName + ".selectAction(prevState);"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("    return a.stepIndex.intValue();"));
        codeBlock.addToBlockBody(new CodeStringDescriptor("}"));

        behUnit.writeRLPolicy(codeBlock, this);

        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("static Action[] " + actionsVarName + ";"));
        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("static QLearner " + learnerVarName + ";"));
        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("static {"));
        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("    " + actionsVarName + " = new Action[] {" + actionNames + "};"));
        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("    " + learnerVarName + " = new QLearner(" + actionsVarName + ");"));
        behUnit.writeRLPolicySuffix(new CodeStringDescriptor("}"));
    }

    /* Declares variables for all formal args appearing in the
        behavior definition. Currently doesn't perform any error
        checking (e.g. duplicate arg names, arg names conflicting with
        explicitly declared variables). */

    private void addArgumentDeclarations() {
        Object[] args = formalArgs.getArguments();
        for (int i = 0; i < args.length; i++) {
            MethodArgDescriptor arg = (MethodArgDescriptor) args[i];
            FieldDescriptor variableDecl = new FieldDescriptor();
            variableDecl.fieldType = arg.argType;
            variableDecl.addFieldName(arg.argName);
            variableDecl.initializer =
                    compileBehaviorArgument(arg.argName, arg.argType);
            addVariableDeclaration(variableDecl);
        }
    }

    // Given a hash set of all the signatures found in the behaving
    // entity, looks through each of the behavior steps looking for
    // subgoals whose signatures aren't matched by any behavior
    // signature in the set.

    void checkForSubgoalsWithNoMatchingBehaviors(
            Set individualSigSet,
            Set jointSigSet)
            throws CompileException {
        AblParseNode n;

        for (int i = 0; i < jjtGetNumChildren(); i++) {
            // Loop through children nodes looking for steps.
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTGOALSTEP) {
                ASTGoalStep goalStepDef = (ASTGoalStep) n;
                if (goalStepDef.isJointGoal()) {
                    if (!jointSigSet.contains(goalStepDef.getSignature()))
                        CompileWarning.warn(
                                getFirstLineNumber(),
                                "no matching joint behavior signature for joint goal signature "
                                        + goalStepDef.getSignature());
                } else {
                    if (!individualSigSet.contains(goalStepDef.getSignature()))
                        CompileWarning.warn(
                                getFirstLineNumber(),
                                "no matching behavior signature for goal signature "
                                        + goalStepDef.getSignature());
                }
            }
        }
    }

    ArrayList<Hashtable<Integer, HashSet<String>>> typeCheckSubgoalsWithInheritance(
            Hashtable<String, ArrayList<ArrayList<Object>>> individualBehaviors,
            Hashtable<String, ArrayList<ArrayList<Object>>> jointBehaviors,
            ASTBehaviorUnit caller) throws CompileException
    {

        //storage for the signatures found from subgoals off this behavior
        Hashtable<Integer, HashSet<String>> individualSignatures = new Hashtable<Integer, HashSet<String>>();
        Hashtable<Integer, HashSet<String>> jointSignatures = new Hashtable<Integer, HashSet<String>>();

        AblParseNode n;
        for (int i = 0; i < jjtGetNumChildren(); i++) {
            //loop through children nodes looking for steps
            n = (AblParseNode) jjtGetChild(i);
            if (n.id == JJTGOALSTEP) {
                ASTGoalStep goalStepDef = (ASTGoalStep) n;
                //find the signature of this goal step
                String signature = goalStepDef.getSignature();
                //extract the list of arguments from this string
                String[] args = new String[goalStepDef.getProcessedArgs().size()];
                for (int c = 0; c < args.length; c++) {
                    args[c] = ((AblArgument) goalStepDef.getProcessedArgs().get(c)).getType();
                }

                //construct the lookup key
                String lookup = signature.substring(0, signature.indexOf('(') + 1) + args.length + ")";
                //System.err.println("Looking up: " + lookup);
                //System.err.flush();

                //extract the Classes that are passed to this step
                Class[] goalStepClasses = new Class[args.length];
                for (int c = 0; c < args.length; c++) {
                    Class cl = caller.findClassFromShortName(args[c]);
                    if (cl == null)
                        throw new CompileException("Unable to find class " + args[c] +
                                " in the imported packages.");
                    goalStepClasses[c] = cl;
                }

                //need to check each argument to see if it's a match for any of the signatures stored in the
                //behavior sig table. Check with inheritance.
                if (args.length > 0) {
                    if (goalStepDef.isJointGoal()) {
                        //list of potential matches to this goal step
                        ArrayList<ArrayList<Object>> matches = jointBehaviors.get(lookup);
                        if (matches == null) {
                            CompileWarning.warn(getFirstLineNumber(),
                                    "no joint behavior signature for joint goal signature "
                                            + goalStepDef.getSignature());
                        } else {
                            ArrayList<Integer> matchingIDs = new ArrayList<Integer>();
                            boolean matchFound = false;
                            for (ArrayList<Object> match : matches) {
                                Integer id = (Integer) match.get(0);
                                boolean isMatch = true;
                                for (int c = 1; c < match.size(); c++) {
                                    if (!(((Class) match.get(c)).isAssignableFrom(goalStepClasses[c - 1]))) {
                                        isMatch = false;
                                        break;
                                    }
                                }
                                //if the signatures match with inheritance, then add the ID to the list of matching behavior IDs
                                if (isMatch) {
                                    matchFound = true;
                                    matchingIDs.add(id);
                                }
                            }
                            //if no match was found, then issue a compiler warning
                            if (!matchFound) {
                                CompileWarning.warn(getFirstLineNumber(),
                                        "no matching joint behavior signature for joint goal signature "
                                                + goalStepDef.getSignature());
                            }
                            else {
                                for (Integer id : matchingIDs) {
                                    if (jointSignatures.containsKey(id)) {
                                        jointSignatures.get(id).add(signature);
                                    }
                                    else {
                                        HashSet<String> temp = new HashSet<String>();
                                        temp.add(signature);
                                        jointSignatures.put(id, temp);
                                    }
                                }
                            }
                        }
                    }
                    else {
                        //list of potential matches to this goal step
                        ArrayList<ArrayList<Object>> matches = individualBehaviors.get(lookup);
                        if (matches == null) {
                            CompileWarning.warn(getFirstLineNumber(),
                                    "no behavior signature for goal signature "
                                            + goalStepDef.getSignature());
                        } else {
                            ArrayList<Integer> matchingIDs = new ArrayList<Integer>();
                            boolean matchFound = false;
                            for (ArrayList<Object> match : matches) {
                                Integer id = (Integer) match.get(0);
                                boolean isMatch = true;
                                for (int c = 1; c < match.size(); c++) {
                                    if (!(((Class) match.get(c)).isAssignableFrom(goalStepClasses[c - 1]))) {
                                        //System.err.println("Cannot match: " + ((Class)match.get(c)).getSimpleName() + " and " +
                                        //                    goalStepClasses[c-1].getSimpleName());
                                        //System.err.flush();
                                        isMatch = false;
                                        break;
                                    }
                                }
                                //if the signatures match with inheritance, then add the ID to the list of matching behavior IDs
                                if (isMatch) {
                                    matchFound = true;
                                    matchingIDs.add(id);
                                }
                            }
                            //if no match was found, then issue a compiler warning
                            if (!matchFound) {
                                CompileWarning.warn(getFirstLineNumber(),
                                        "no matching behavior signature for goal signature "
                                                + goalStepDef.getSignature());
                            }
                            else {
                                for (Integer id : matchingIDs) {
                                    if (individualSignatures.containsKey(id)) {
                                        individualSignatures.get(id).add(signature);
                                    }
                                    else {
                                        HashSet<String> temp = new HashSet<String>();
                                        temp.add(signature);
                                        individualSignatures.put(id, temp);
                                    }
                                }
                            }
                        }
                    }

                }
                else { //if there aren't any args, it's an automatic match
                    if (goalStepDef.isJointGoal()) {
                        if (jointBehaviors.containsKey(lookup)) {
                            for (ArrayList<Object> idArgs : jointBehaviors.get(lookup)) {
                                Integer id = (Integer) idArgs.get(0);
                                //this id number is the behavior ID, which is the key to the returned hashtable
                                if (jointSignatures.containsKey(id)) {
                                    jointSignatures.get(id).add(signature);
                                } else {
                                    HashSet<String> temp = new HashSet<String>();
                                    temp.add(signature);
                                    jointSignatures.put(id, temp);
                                }
                            }
                        } else {
                            CompileWarning.warn(getFirstLineNumber(),
                                    "no matching joint behavior signature for joint goal signature "
                                            + goalStepDef.getSignature());
                        }
                    } else {
                        if (individualBehaviors.containsKey(lookup)) {
                            for (ArrayList<Object> idArgs : individualBehaviors.get(lookup)) {
                                Integer id = (Integer) idArgs.get(0);
                                //this id number is the behavior ID, which is the key to the returned hashtable
                                if (individualSignatures.containsKey(id)) {
                                    individualSignatures.get(id).add(signature);
                                } else {
                                    HashSet<String> temp = new HashSet<String>();
                                    temp.add(signature);
                                    individualSignatures.put(id, temp);
                                }
                            }
                        } else {
                            CompileWarning.warn(getFirstLineNumber(),
                                    "no matching behavior signature for goal signature "
                                            + goalStepDef.getSignature());
                        }
                    }
                }
            }
        }


        //return the signatures
        ArrayList<Hashtable<Integer, HashSet<String>>> signatures = new ArrayList<Hashtable<Integer, HashSet<String>>>();
        signatures.add(individualSignatures);
        signatures.add(jointSignatures);
        return signatures;
    }

    // Compiles the code block initializing the behavior frame.

    private CodeBlockDescriptor compileBehaviorFrame() throws CompileException {
        CodeBlockDescriptor frameInit = new CodeBlockDescriptor();
        // behavior declares variables
        if (fieldDescriptors.size() > 0) {
            // fixme: Eliminate this restriction in later versions of ABL.
            assert !isNestedScope : "Attempt to declare a variable within an anonymous behavior. Currently this is not allowed. Future versions of ABL will lift this restriction";
            frameInit.addToBlockBody(
                    new CodeStringDescriptor(
                            "final Object[] __$behaviorFrame = new Object["
                                    + fieldDescriptors.size()
                                    + "];"));
        } else if (isNestedScope) {
            // fixme: this doesn't handle variables declared within the anonymous behavior scope (see fixme above).
            frameInit.addToBlockBody(new CodeStringDescriptor("final Object[] __$behaviorFrame = __$args;"));
        }
        final Iterator<FieldDescriptor> varIter = fieldDescriptors.iterator();
        HashSet boundVariables = getBoundVariables();
        while (varIter.hasNext()) {
            FieldDescriptor field = varIter.next();
            assert (field.getFieldNames().length == 1);
            String variableName = field.getFieldNames()[0];
            String initString = ""; // variable initializer
            if (boundVariables.contains(variableName)) {
                // initialize to the current binding
                if (primitiveType(field.fieldType)) {
                    if (field.fieldType.equals("int"))
                        initString =
                                "(Integer)__$boundVars.get(\""
                                        + variableName
                                        + "\")";
                    else if (field.fieldType.equals("float"))
                        initString =
                                "(Float)__$boundVars.get(\"" + variableName + "\")";
                    else if (field.fieldType.equals("char"))
                        initString =
                                "(Character)__$boundVars.get(\""
                                        + variableName
                                        + "\")";
                    else if (field.fieldType.equals("boolean"))
                        initString =
                                "(Boolean)__$boundVars.get(\""
                                        + variableName
                                        + "\")";
                    else if (field.fieldType.equals("long"))
                        initString =
                                "(Long)__$boundVars.get(\"" + variableName + "\")";
                    else if (field.fieldType.equals("short"))
                        initString =
                                "(Short)__$boundVars.get(\"" + variableName + "\")";
                    else if (field.fieldType.equals("byte"))
                        initString =
                                "(Byte)__$boundVars.get(\"" + variableName + "\")";
                    else if (field.fieldType.equals("double"))
                        initString =
                                "(Double)__$boundVars.get(\""
                                        + variableName
                                        + "\")";
                } else
                    initString =
                            "((ObjectWrapper)__$boundVars.get(\""
                                    + variableName
                                    + "\")).objectValue()";
            } else if (field.initializer != null)
                initString = field.initializer;

            String valueString = null;
            // Construct the value objects
            if (primitiveType(field.fieldType)) {
                if (field.fieldType.equals("int"))
                    valueString = "new __ValueTypes.IntVar(" + initString + ")";
                else if (field.fieldType.equals("float"))
                    valueString =
                            "new __ValueTypes.FloatVar(" + initString + ")";
                else if (field.fieldType.equals("char"))
                    valueString =
                            "new __ValueTypes.CharVar(" + initString + ")";
                else if (field.fieldType.equals("boolean"))
                    valueString =
                            "new __ValueTypes.BooleanVar(" + initString + ")";
                else if (field.fieldType.equals("long"))
                    valueString =
                            "new __ValueTypes.LongVar(" + initString + ")";
                else if (field.fieldType.equals("short"))
                    valueString =
                            "new __ValueTypes.ShortVar(" + initString + ")";
                else if (field.fieldType.equals("byte"))
                    valueString =
                            "new __ValueTypes.ByteVar(" + initString + ")";
                else if (field.fieldType.equals("double"))
                    valueString =
                            "new __ValueTypes.DoubleVar(" + initString + ")";
            } else
                valueString = initString;

            if (!(valueString.length() == 0))
                // There is an explicit initializer and/or the behavior variable is primitive.
                frameInit.addToBlockBody(
                        new CodeStringDescriptor(
                                "__$behaviorFrame["
                                        + getBehaviorFrameIndex(variableName)
                                        + "] = "
                                        + valueString
                                        + ";"));
        }
        return frameInit;
    }

    private CodeBlockDescriptor compileBehaviorFactory() throws CompileException {
        assert (stepIDs.size() == stepFactories.size());

        CodeBlockDescriptor behaviorFactory =
                new CodeBlockDescriptor("case " + behaviorID + ": {", "}");

        behaviorFactory.addToBlockBody(
                new CodeStringDescriptor("// " + uniqueBehaviorName));
        behaviorFactory.addToBlockBody(compileBehaviorFrame());

        // compile __StepDesc array
        StringBuffer stepArrayInit = new StringBuffer(2000);
        stepArrayInit.append("final __StepDesc[] __$steps = ");
        if (stepIDs.size() > 0) {
            stepArrayInit.append("{");
            final Iterator<Integer> stepIDIter = stepIDs.iterator();
            final Iterator<String> stepFactoryIter = stepFactories.iterator();
            while (stepIDIter.hasNext()) {
                stepArrayInit.append(
                        "new __StepDesc("
                                + stepIDIter.next()
                                + ", "
                                + getBehavingEntityField(stepFactoryIter.next())
                                + ")");
                if (stepIDIter.hasNext())
                    stepArrayInit.append(", ");
            }
            stepArrayInit.append("}");
        } else
            stepArrayInit.append("null");
        stepArrayInit.append(";");
        behaviorFactory.addToBlockBody(
                new CodeStringDescriptor(stepArrayInit.toString()));

        // Create the constructor.
        StringBuffer constructorCall = new StringBuffer(500);
        constructorCall.append("return new ");
        if (isAdaptive)
            constructorCall.append("AdaptiveBehavior");
        else if (behaviorType.equals("sequential"))
            constructorCall.append("SequentialBehavior");
        else if (behaviorType.equals("parallel"))
            constructorCall.append("ParallelBehavior");
        else if (behaviorType.equals("collection"))
            constructorCall.append("CollectionBehavior");

        if (Abl.debugLevel == GUI_DEBUGGER)
            // construct debug versions of behaviors only for debug level 2
            constructorCall.append("Debug(");
        else
            constructorCall.append("(");

        String contextCondition = null;
        if (contextConditionMethod_rField != null)
            contextCondition =
                    getBehavingEntityField(contextConditionMethod_rField);

        String contextConditionSensorFactory = null;
        if (contextConditionSensorFactoryMethod_rField != null)
            contextConditionSensorFactory =
                    getBehavingEntityField(contextConditionSensorFactoryMethod_rField);

        String successCondition = null;
        if (successConditionMethod_rField != null)
            successCondition =
                    getBehavingEntityField(successConditionMethod_rField);

        String successConditionSensorFactory = null;
        if (successConditionSensorFactoryMethod_rField != null)
            successConditionSensorFactory =
                    getBehavingEntityField(successConditionSensorFactoryMethod_rField);

        // fixme: may need to add uniqueName to the constructor as well, but most likely just for debug behaviors.
        constructorCall.append(
                "__$parentGoal, "
                        + contextCondition
                        + ", "
                        + contextConditionSensorFactory
                        + ", "
                        + successCondition
                        + ", "
                        + successConditionSensorFactory
                        + ", ");
        if (isAdaptive) {
            String reinforcementSignals = null;
            if (reinforcementSignalsMethod_rField != null)
                reinforcementSignals =
                        getBehavingEntityField(reinforcementSignalsMethod_rField);

            String reinforcementSignalsSensorFactory = null;
            if (reinforcementSignalsSensorFactoryMethod_rField != null)
                reinforcementSignalsSensorFactory =
                        getBehavingEntityField(reinforcementSignalsSensorFactoryMethod_rField);

            String reinforcementState = null;
            if (stateWMEMethod_rField != null)
                reinforcementState =
                        getBehavingEntityField(stateWMEMethod_rField);

            String reinforcementStateSensorFactory = null;
            if (stateWMESensorFactoryMethod_rField != null)
                reinforcementStateSensorFactory =
                        getBehavingEntityField(stateWMESensorFactoryMethod_rField);

            String rlPolicy = null;
            if (rlPolicyMethod_rField != null)
                rlPolicy =
                        getBehavingEntityField(rlPolicyMethod_rField);

            constructorCall.append(
                    reinforcementSignals
                            + ", "
                            + reinforcementSignalsSensorFactory
                            + ", "
                            + reinforcementState
                            + ", "
                            + reinforcementStateSensorFactory
                            + ", "
                            + rlPolicy
                            + ", ");
        }
        constructorCall.append(
                isAtomic
                        + ", "
                        + "__$signature"
                        + ", (short)"
                        + specificity
                        + ", "
                        + behaviorID
                        + ", ");
        // If behavior scope variables have been declared or we're in a nested scope (anonymous behavior), pass the behavior frame.
        if (fieldDescriptors.size() > 0 || isNestedScope)
            // There are behavior variables
            constructorCall.append("__$behaviorFrame, ");
        else
            constructorCall.append("null, ");

        constructorCall.append("__$steps");

        if (isAdaptive) {
            if (behaviorType.equals("sequential"))
                constructorCall.append(", Behavior.SEQUENTIAL");
            else if (behaviorType.equals("parallel"))
                constructorCall.append(", Behavior.PARALLEL");
            else
                throw new CompileException(getFirstLineNumber(), "Invalid behavior type for adaptive behavior");
        }

        if (behaviorType.equals("parallel")
                || behaviorType.equals("collection")) {
            if (numberNeededForSuccess == -1)
                // numberNeededForSuccess not explicitly defined - set it equal to the number of non-effect-only steps
                constructorCall.append(", " + nonEffectOnlySteps);
            else
                // numberNeededForSuccess explicitly defined
                constructorCall.append(", " + numberNeededForSuccess);
        }

        if (isJoint) {
            // joint behavior - add team members and close constructor call
            assert (teamMembers.size() > 0);
            StringBuffer teamArrayInit = new StringBuffer(500);
            teamArrayInit.append("BehavingEntity[] __$teamMembers = {");
            final Iterator<String> iter = teamMembers.iterator();
            while (iter.hasNext()) {
                teamArrayInit.append(
                        "BehavingEntity.getBehavingEntity(\""
                                + iter.next()
                                + "\")");
                if (iter.hasNext())
                    teamArrayInit.append(", ");
            }
            teamArrayInit.append("};");
            behaviorFactory.addToBlockBody(
                    new CodeStringDescriptor(teamArrayInit.toString()));
            constructorCall.append(", __$teamMembers");
        }

        if (Abl.debugLevel == GUI_DEBUGGER)
            // Add behaviorDesc only for debug level 2
            constructorCall.append(", __$behaviorDesc");

        constructorCall.append(");");

        behaviorFactory.addToBlockBody(
                new CodeStringDescriptor(constructorCall.toString()));

        return behaviorFactory;
    }

    void compileToJava() throws CompileException {
        assert (behaviorType != null && behaviorName != null);

        ASTBehaviorUnit behaviorUnitNode = ASTBehaviorUnit.getBehaviorUnit();

        behaviorID = behaviorUnitNode.getUniqueBehaviorID();

        // Determine the unique behavior name.
        if (behaviorName.equals("__RootCollectionBehavior")) {
            // If the root collection behavior, set the unique name for both the behavior and the concrete class.
            behaviorName =
                    behaviorUnitNode.getBehavingEntityClass()
                            + "_RootCollectionBehavior";
            uniqueBehaviorName = behaviorName;

            behaviorUnitNode.rootBehaviorID = behaviorID;
        } else {
            uniqueBehaviorName =
                    behaviorName
                            + "_"
                            + behaviorUnitNode.getBehaviorCount(behaviorName);
            behaviorUnitNode.incrementBehaviorCount(behaviorName);
        }

        // The behavior unit keeps track of which individual and joint behaviors have been defined.
        behaviorUnitNode.addBehaviorNode(this);

        /* Intialize the behavior variable scope. The behavior scope
                 must be initialized before compiling the steps because the
                 steps may make variable references. */
        scopeName = uniqueBehaviorName;
        setScopeType(AblScopeMaintainer.BEHAVIOR_SCOPE);
        initializeScope();

        // Add the variables for argument declarations, bound variables and declared variables, in that order.
        addArgumentDeclarations(); // Formal argument decls.
        addChildVariableDeclarationsWithReferenceRewrite();
        // Process all the children ASTAblVariableDecl nodes.
        compileTests();
        // Compile any precondition or context condition - declares new variables

        // Compile the behavior steps.
        compileSteps();

        if (isAdaptive) {
            compileReinforcementStateAndSignals();
        }

        // Now declare all the behavior class fields.

        if (hasContextCondition) {
            CodeBlockDescriptor contextConditionSensorFactory =
                    compileSensorActivationFactory(
                            ASTTestExpression.CONTEXT_CONDITION);
            if (contextConditionSensorFactory != null)
                // registered wmes referenced in context condition
                behaviorUnitNode.writeContextConditionSensorActivation(
                        contextConditionSensorFactory,
                        this);
        }

        if (hasSuccessCondition) {
            CodeBlockDescriptor successConditionSensorFactory =
                    compileSensorActivationFactory(
                            ASTTestExpression.SUCCESS_CONDITION);
            if (successConditionSensorFactory != null)
                // registered wmes referenced in success condition
                behaviorUnitNode.writeSuccessConditionSensorActivation(
                        successConditionSensorFactory,
                        this);
        }

        if (hasPrecondition) {
            CodeBlockDescriptor preconditionSensorFactory =
                    compileSensorActivationFactory(ASTTestExpression.PRECONDITION);
            if (preconditionSensorFactory != null)
                // registered wmes referenced in precondition
                behaviorUnitNode.writePreconditionSensorActivation(
                        preconditionSensorFactory,
                        this);
        }

        if (isAdaptive) {
            compileReinforcementPolicyAndAction();
        }

        // Compile the behavior factory at the end since the factory references the class and method names of the
        // other behavior parts (e.g. context condition, sensor factories).
        behaviorUnitNode.writeBehaviorFactory(compileBehaviorFactory(), this);
    }
}
