/* Parse node for ABL expressions. A ABL expression is either an ABL
 literal (single token) or a Java name. If it is a Java name, this
 node will have a child of type ASTJavaName. */

package abl.compiler;

public class ASTAblExpression extends AblParseNode {
	ASTAblExpression(int id) {
		super(id);
	}

	ASTAblExpression(AblParser p, int id) {
		super(p, id);
	}

	boolean isIdentifier() {
		if (jjtGetNumChildren() == 1)
			// The ASTAblExpression has a child - an ASTJavaName representing a
			// variable reference or constant.
			return true;
		else
			return false;
	}

	boolean isLiteral() {
		return !isIdentifier();
	}

	// Returns the JavaName representing the literal variable
	// reference (e.g. x.y). Returns null if the AblExpression is a
	// literal.
	ASTJavaName getRef() {
		if (isIdentifier())
			return (ASTJavaName) jjtGetChild(0);
		else
			return null;
	}

	String getVariableReferenceConstantOrLiteral(AblScopeMaintainer scope)
			throws CompileException {
		if (isIdentifier())
			return getRef().getVariableReferenceOrConstant(scope);
		else
			// AblExpression is a literal
			return firstToken.image;
	}

	// If the variable is of primitive type, or this expression is a constant or
	// literal, wraps the string representation of the primitive
	// variable, returns a string wrapping the primitive variable, constant or
	// literal in an appropriate Object wrapper
	// (e.g. "new Integer(((Foo)parent).bar)" ).
	// fixme: this method doesn't seem to be called during compiles - is it redundant?
	String getVariableReferenceConstantOrLiteralAsObject(AblScopeMaintainer scope) throws CompileException {
		if (isIdentifier())
			return getRef().getVariableReferenceOrConstantAsObject(scope);
		else
			// AblExpression is a literal
			switch (firstToken.kind) {
			case INTEGER_LITERAL:
				return "new Integer(" + firstToken.image + ")";
			case FLOATING_POINT_LITERAL:
				return "new Float(" + firstToken.image + ")";
			case DOUBLE_LITERAL:
				return "new Double(" + firstToken.image + ")";
			case CHARACTER_LITERAL:
				return "new Character(" + firstToken.image.charAt(0) + ")";
			case STRING_LITERAL:
				return "new String(" + firstToken.image + ")";
			case TRUE:
			case FALSE:
				return "new Boolean(" + firstToken.image + ")";
			case NULL:
				return "null";
			default:
				throw new CompileError("Unexpected literal token");
			}
	}

	// If the AblExpression represents a literal, returns an object
	// representing that literal. Otherwise throws an error.
	Object getLiteralValue() {
		if (isLiteral()) {
			switch (firstToken.kind) {
			case INTEGER_LITERAL:
				return new Integer(firstToken.image);
			case FLOATING_POINT_LITERAL:
				return new Float(firstToken.image);
			case DOUBLE_LITERAL:
				return new Double(firstToken.image);
			case CHARACTER_LITERAL:
				return new Character(firstToken.image.charAt(0));
			case STRING_LITERAL:
				return new String(firstToken.image);
			case TRUE:
			case FALSE:
				return new Boolean(firstToken.image);
			case NULL:
				return null;
			default:
				throw new CompileError("Unexpected literal token");
			}
		} else
			throw new CompileError(dumpTokens() + " is not a literal.");
	}

	String getFullNameOrLiteral() {
		if (isIdentifier())
			return getRef().getFullName();
		else
			return firstToken.image;
	}

	/** returns the type name of this abl expression */
	// fixme: this method doesn't seem to be called during compiles - is it redundant?
	public String getType(AblScopeMaintainer scope) throws CompileException {
		if (isIdentifier())
			return getRef().getType(scope);
		else
			switch (firstToken.kind) {
			case INTEGER_LITERAL:
				return "int";
			case FLOATING_POINT_LITERAL:
				return "float";
			case DOUBLE_LITERAL:
				return "double";
			case CHARACTER_LITERAL:
				return "char";
			case STRING_LITERAL:
				return "String";
			case TRUE:
			case FALSE:
				return "boolean";
			case NULL:
				return "Object";
			default:
				throw new CompileError("Unexpected literal token");
			}
	}
}
