#include "TrainTask.h"
#include "Environment.h"
#include "Agent.h"

#define TRAIN_TASK_VALIDATION_FREQ	1
#define RESOURCE_BONUS				8

TrainTask::TrainTask(UnitType unitType, Environment* env) : Task(env, "Train ", false)
{
	_types.insert(TRAIN_TASK);

	_name.append(unitType.getName());

	_unitType = unitType;

	_trainerType = _unitType.whatBuilds().first;

	_requiredAddon = UnitTypes::None;

	UnitTypeCountMap requiredUnits = UnitTypes::Terran_Comsat_Station.requiredUnits();
	for (UnitTypeCountMap::iterator i = requiredUnits.begin(); i != requiredUnits.end(); i++)
	{
		UnitType type = (*i).first;
		
		if (type.isAddon() && (type.whatBuilds().first == _trainerType))
		{
			_requiredAddon = type;
			break;
		}
	}

	_rallyPosition = Positions::None;

	_unitTypeTrained = NULL;

	//_execValidationFreq = TRAIN_TASK_VALIDATION_FREQ;

	env->incRequiredMinerals(_unitType.mineralPrice());
	env->incRequiredGas(_unitType.gasPrice());
	env->incRequiredSupply(_unitType.supplyRequired());
}

TrainTask::TrainTask(UnitType unitType, Position rallyPosition, Environment* env) : Task(env, "Train ", false)
{
	_types.insert(TRAIN_TASK);

	_name.append(unitType.getName());

	_unitType = unitType;

	_trainerType = _unitType.whatBuilds().first;

	_rallyPosition = rallyPosition;

	_unitTypeTrained = UnitTypes::None;

	_env->incRequiredMinerals(_unitType.mineralPrice());
	_env->incRequiredGas(_unitType.gasPrice());
	_env->incRequiredSupply(_unitType.supplyRequired());
}

TrainTask::~TrainTask(void)
{
	_env->decRequiredMinerals(_unitType.mineralPrice());
	_env->decRequiredGas(_unitType.gasPrice());
	_env->decRequiredSupply(_unitType.supplyRequired());
}

double TrainTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType unitType = unit->getType();

	if (unitType != _trainerType) return NO_APTITUDE;

	if (_requiredAddon != UnitTypes::None)
	{
		Unit* addon = unit->getAddon();

		if (addon == NULL) return NO_APTITUDE;
		if (addon->getType() != _requiredAddon) return NO_APTITUDE;
	}

	if (unit->isLifted()) return NO_APTITUDE;

	if (unit->isUnpowered()) return NO_APTITUDE;

	int timeLeft = unit->getRemainingTrainTime();
	timeLeft += unit->getRemainingBuildTime();
	timeLeft += unit->getRemainingUpgradeTime();
	timeLeft += unit->getRemainingResearchTime();

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + (double)timeLeft);

	if (_rallyPosition == Positions::None) return aptitude;

	double dist = _rallyPosition.getDistance(unit->getPosition());

	aptitude = aptitude/(1.0 + dist);

	return aptitude;
}

void TrainTask::evaluateStatus()
{
	_canExecute = true;
	_statusMessage = "None specified.";
	_requiresUnits = false;

	bool hasUnits = _agents.size() >= 1;
	
	if (_status == IN_PROGRESS)
	{
		Unit* trainer = NULL;

		if (hasUnits)
		{
			trainer = (*_agents.begin())->getUnit();

			if (trainer->getRemainingTrainTime() <= 0)
			{
				_statusMessage = "Finished.";
				_status = FINISHED;
				return;
			}
		}
	}
	else if (_status == NOT_STARTED)
	{
		int mineralCost = _unitType.mineralPrice();
		int gasCost = _unitType.gasPrice();
		int supplyCost = _unitType.supplyRequired();

		int currentMinerals = _env->getCurrentMinerals();
		int reservedMinerals = _env->getReservedMinerals();

		int currentGas = _env->getCurrentGas();
		int reservedGas = _env->getReservedGas();

		int currentSupply = _env->getCurrentSupply();
		int reservedSupply = _env->getReservedSupply();
		
		int excessMinerals = max(0, currentMinerals - reservedMinerals);
		int excessGas = max(0, currentGas - reservedGas);
		int excessSupply = max(0, currentSupply - reservedSupply);

		_canExecute =
			(excessMinerals >= mineralCost) &&
			(excessGas >= gasCost) &&
			(excessSupply >= supplyCost);

		double percentCollected = 1.0;

		if ((mineralCost > 0) && (gasCost > 0))
		{
			percentCollected = min(excessMinerals/(double)mineralCost, excessGas/(double)gasCost);
			percentCollected = min(percentCollected, 1.0);
		}

		_env->reserveMinerals((int)(mineralCost*percentCollected + RESOURCE_BONUS));
		_env->reserveGas((int)(gasCost*percentCollected + RESOURCE_BONUS));
		
		_env->reserveSupply(supplyCost);


		if (!_canExecute)
		{
			_statusMessage = "Not enough resources.";
			return;
		}
	}
	
	_requiresUnits = true;

	_canExecute = hasUnits;

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		return;
	}

	_statusMessage = "No issues.";
}

UnitType TrainTask::getUnitTypeTrained()
{
	return _unitTypeTrained;
}

void TrainTask::evaluateNeededUnits()
{
	if (_agents.size() < 1)	_needsMoreUnits = _requiresUnits;
	else					_needsMoreUnits = false;
}

bool TrainTask::execute(Agent* agent)
{
	if (_status == IN_PROGRESS) return true;

	Unit* unit = agent->getUnit();

	if (!unit->isIdle()) return true;

	if (!unit->train(_unitType))
	{
		_status = CANCELLED;
		return false;
	}

	_unitTypeTrained = _unitType;

	if (_rallyPosition == Positions::None)
	{
		_rallyPosition = unit->getPosition();
	}

	unit->setRallyPoint(_rallyPosition);
	_status = IN_PROGRESS;

	return true;
}
