#include "Agent.h"
#include "Task.h"
#include "Environment.h"

TaskSet Task::filterSet(TaskSet* tasks, TaskType type)
{
	TaskSet filteredSet;

	for (TaskSet::iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = (*i);

		if (task->isType(type)) filteredSet.insert(task);
	}

	return filteredSet;
}

TaskSet Task::filterSet(TaskSet* tasks, TaskType type, int maxCreationFrame)
{
	TaskSet filteredSet;

	for (TaskSet::iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = (*i);

		if (task->getCreationFrame() >= maxCreationFrame) continue;

		if (task->isType(type)) filteredSet.insert(task);
	}

	return filteredSet;
}

TaskSet Task::filterSetByPriority(TaskSet* tasks, TaskType type, double priority)
{
	TaskSet filteredSet;

	for (TaskSet::iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = (*i);

		if (task->getPriority() <= priority) continue;

		if (task->isType(type)) filteredSet.insert(task);
	}

	return filteredSet;
}

double Task::scalePriority(double value, double maxPriority, double minPriority)
{
	double sValue = value/(1000 + value);

	return minPriority + sValue*(maxPriority - minPriority);
}

Task::Task(Environment* env, string name, bool canBePreempted, int execValidationFreq)
{
	_env = env;
	_name = name;

	_types.insert(TASK);

	_canExecute = false;
	_needsMoreUnits = false;

	_status = NOT_STARTED;
	_statusMessage = "None specified.";

	_canBePreempted = canBePreempted;

	_creationFrame = Broodwar->getFrameCount();

	_id = env->getNextTaskId();

	_filterTaskInfo = false;

	_execValidationFreq = execValidationFreq;

	_syncUp = false;
}

Task::~Task(void)
{
	//for(AgentSet::const_iterator i = _agents.begin(); i != _agents.end(); i++)
	//{
	//	(*i)->removeDeletedTask();
	//}
}

void Task::cleanup()
{
	AgentList agentsToRemove;

	for(AgentSet::const_iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		agentsToRemove.push_back(*i);
	}

	for(AgentList::iterator i = agentsToRemove.begin(); i != agentsToRemove.end(); i++)
	{
		(*i)->assignTask(NULL);
	}
}

bool Task::isType(TaskType type)
{
	for(TaskTypeSet::const_iterator i = _types.begin(); i != _types.end(); i++)
	{
		if (*i == type) return true;
	}

	return false;
}

double Task::getPriority()
{
	return _priority;
}

TaskStatus Task::getStatus()
{
	return _status;
}

void Task::setStatus(TaskStatus status)
{
	_status = status;
	_statusMessage = "None.";
}

void Task::setStatus(TaskStatus status, string statusMessage)
{
	_status = status;
	_statusMessage = statusMessage;
}

int Task::getId()
{
	return _id;
}

string Task::getName()
{
	return _name;
}

int Task::getCreationFrame()
{
	return _creationFrame;
}

int Task::getExecValidationFreq()
{
	return _execValidationFreq;
}

bool Task::mustSyncUp()
{
	return _syncUp;
}

bool Task::canBePreempted()
{
	return _canBePreempted;
}

void Task::addAgent(Agent* agent)
{
	_agents.insert(agent);
}

void Task::removeAgent(Agent* agent)
{
	_agents.erase(agent);
}

bool Task::needsMoreUnits()
{
	return _needsMoreUnits;
}

bool Task::canExecute()
{
	return _canExecute;
}

string Task::statusMessage()
{
	return _statusMessage;
}

bool compareTempValues(Agent* first, Agent* second)
{
	return (first->getTempValue() > second->getTempValue());
}

void Task::assignNeededAgents(AgentMap* agents)
{
	evaluateNeededUnits();

	if (!_needsMoreUnits && !_canBePreempted) return;
	if (!_needsMoreUnits) return;

//	_agents.clear();

	AgentList agentList;

	for(AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;
		Task* currentTask = agent->getCurrentTask();

		if (currentTask != this)
		{
			if (!agent->isFinished()) continue;

			if (agent->getUnit()->isStuck()) continue;

			double aptitude = evaluateAptitude(agent);
			
			if (aptitude == NO_APTITUDE) continue;
			
			if (currentTask != NULL)
			{
				if (!currentTask->canBePreempted() ||
					(_priority <= currentTask->getPriority()) ||
					(agent->getTaskHeldTime() <= MIN_TASK_HOLD_FRAMES)) continue;

			//	double currentTaskApt = currentTask->evaluateAptitude(agent)*currentTask->getPriority();
			//	double taskApt = aptitude*_priority;

			//	if (taskApt <= currentTaskApt) continue;
			}

			agent->setTempValue(aptitude);
		}
		else
		{
			//agent->setTempValue(agent->getCurrentAptitude());
			continue;
		}
		
		agentList.push_back(agent);
	}

	if (agentList.empty()) return;

	agentList.sort(compareTempValues);

	for(AgentList::iterator i = agentList.begin(); i != agentList.end(); i++)
	{
		Agent* agent = (*i);

		evaluateNeededUnits();

		if (_needsMoreUnits)
			agent->assignTask(this);
		//else if (agent->getCurrentTask() == this)
		//	agent->assignTask(NULL);
		else break;
	}
}

void Task::displayInfo(int &row, Agent* agent)
{
	if (_filterTaskInfo) return;

	char message[400];

	if (agent == NULL)
	{
		sprintf_s(message, 400, "Task '%s': priority = %0.3f, canExecute = %d, status = %d, needsMoreUnits = %d, message = '%s'",
			_name.c_str(),
			_priority,
			_canExecute,
			_status,
			_needsMoreUnits,
			_statusMessage.c_str());
	}
	else
	{
		sprintf_s(message, 400, "Task '%s': agent aptitude %0.3f, priority = %0.3f, canExecute = %d, status = %d, needsMoreUnits = %d, message = '%s'",
			_name.c_str(),
			evaluateAptitude(agent),
			_priority,
			_canExecute,
			_status,
			_needsMoreUnits,
			_statusMessage.c_str());
	}

	//Print miscellaneous task info.
	Broodwar->drawTextScreen(5,10 + (row*10), message);

	//_logger->Log(message);
	
	row++;
}

