#include "ScanTargetTask.h"
#include "Environment.h"
#include "Agent.h"

#define TASK_PRIORITY_FACTOR		1
#define ENERGY_APTITUDE_FACTOR		1
#define SIGHT_RANGE_FACTOR			1.5
#define SCAN_TASK_VALIDATION_FREQ	10
#define MIN_ATTACKERS				5

void ScanTargetTask::generateNecessaryTasks(Environment* env)
{
	UnitSet allUnits = Broodwar->getAllUnits();

	for(UnitSet::iterator i = allUnits.begin(); i != allUnits.end(); i++)
	{
		Unit* unit = *i;

		if (!env->isEnemyUnit(unit)) continue;

		if (!unit->isVisible()) continue;

		if (unit->isDetected()) continue;
		
		if (env->isTaskTarget(unit, SCAN_TARGET_TASK)) continue;

		env->addTask(new ScanTargetTask(unit, env));
	}
}

ScanTargetTask::ScanTargetTask(Unit* target, Environment* env) : Task(env, "Scan target")
{
	_types.insert(SCAN_TARGET_TASK);
	
	_target = target;

	_execValidationFreq = SCAN_TASK_VALIDATION_FREQ;

	_env->addTaskTarget(_target, SCAN_TARGET_TASK);
}

ScanTargetTask::~ScanTargetTask(void)
{
	_env->removeTaskTarget(_target, SCAN_TARGET_TASK);
}

Unit* ScanTargetTask::getTarget()
{
	return _target;
}

double ScanTargetTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (type != UnitTypes::Terran_Comsat_Station) return NO_APTITUDE;

	double aptitude = ENERGY_APTITUDE_FACTOR*unit->getEnergy()/(double)type.maxEnergy();

	return aptitude;
}

void ScanTargetTask::evaluateStatus()
{
	if (!_target->exists())
	{
		if (Broodwar->isVisible(TilePosition(_target->getPosition())))
		{
			_statusMessage = "Target doesn't exist.";
			_status = CANCELLED;
			return;
		}
	}

	if (!_target->isVisible())
	{
		_statusMessage = "Target is no longer visible.";
		_status = CANCELLED;
		return;
	}

	if (_target->isDetected())
	{
		_statusMessage = "Target is dectected.";
		_status = FINISHED;
		return;
	}

	_canExecute = (_agents.size() >= 1);

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		_status = HALTED;
		return;
	}

	int energyCost = TechTypes::Scanner_Sweep.energyUsed();

	Unit* scanner = (*_agents.begin())->getUnit();

	_canExecute = energyCost <= scanner->getEnergy();

	if (!_canExecute)
	{
		_statusMessage = "Not enough energy.";
		_status = HALTED;
		return;
	}

	UnitType targetType = _target->getType();

	AgentSet agentsInRange = _env->getAgentsInRadius(_target->getPosition(), MAX_THREAT_RANGE);

	int attackerCount = 0;
	bool canBeAttacked = false;
	for(AgentSet::iterator i = agentsInRange.begin(); i != agentsInRange.end(); i++)
	{
		Agent* agent = *i;

		if (targetType.isFlyer() && !agent->canAttackAir()) continue;

		if (!targetType.isFlyer() && !agent->canAttackGround()) continue;

		if (!agent->isType(ATTACKER_AGENT)) continue;

		attackerCount++;
	}

	if (MIN_ATTACKERS <= attackerCount)
	{
		_statusMessage = "No unit in range of target.";
		_status = HALTED;
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void ScanTargetTask::evaluateNeededUnits()
{
	if (_agents.size() < 1)	_needsMoreUnits = true;
	else					_needsMoreUnits = false;
}

void ScanTargetTask::calculatePriority()
{
	UnitType tType = _target->getType();
	double tValue = Environment::getUnitHealthValue(_target)/Environment::getUnitTypeValue(tType);
	double wValue = Environment::getUnitAirAttackValue(tType) + Environment::getUnitGroundAttackValue(tType);

	double prePriority = TASK_PRIORITY_FACTOR*tValue*(1 + wValue);

	_priority = scalePriority(prePriority, MAX_SCAN_PRIORITY, MIN_SCAN_PRIORITY);
}

bool ScanTargetTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	return unit->useTech(TechTypes::Scanner_Sweep, _target->getPosition());
}

