#include "ResearchTask.h"
#include "Environment.h"
#include "Agent.h"
#include "AiLogger.h"

#define TASK_PRIORITY_FACTOR		1000
#define CONCURRENT_TASK_FACTOR		1
#define MIN_ATTACKER_AGENTS			15
#define RESOURCE_BONUS				8

static std::set<BWAPI::UpgradeType>		upgradeInProgress;
static std::set<BWAPI::TechType>		techInProgress;

static bool generateTechs(Environment* env, BWAPI::Player* player)
{
	std::set<BWAPI::TechType> raceTechs, posibleTechs;

	for(std::set<BWAPI::TechType>::iterator i = BWAPI::TechTypes::allTechTypes().begin(); i != BWAPI::TechTypes::allTechTypes().end(); ++i)
	{
		TechType type = (*i);

		if(type.getRace() != player->getRace()) continue;
		if (type == TechTypes::None) continue;

		raceTechs.insert(type);
	}

	for(std::set<BWAPI::TechType>::iterator i = raceTechs.begin(); i!=raceTechs.end(); ++i)
	{
		const BWAPI::TechType& tech = *i;
		bool inProgress = techInProgress.find(tech) != techInProgress.end();

		if(inProgress || player->hasResearched(tech) ) continue;

		bool hasTypeTeched = false;
		std::set<UnitType> typesTeched = tech.whatUses();

		for(std::set<UnitType>::iterator it = typesTeched.begin(); it != typesTeched.end(); it++)
			if (hasTypeTeched = env->getFilteredAgents(*it).size() > 0) break;

		if(!hasTypeTeched) continue;

		UnitType buildingType = tech.whatResearches();
		AgentMap buildings = env->getFilteredAgents(buildingType);
		bool hasBuilding = buildings.size() > 0;

		if(hasBuilding)
		{
			for(AgentMap::iterator it = buildings.begin(); it != buildings.end(); ++it)
				if((*it).first->isIdle())
				{
					//AiLogger::taskLog.Log(upgrade.getName().append(" Research created").c_str());
					env->addTask(new ResearchTask(BWAPI::UpgradeTypes::None, tech, env));
					techInProgress.insert(tech);
					return true;
				}
		}
	}
	return false;
}

static bool generateUpgrades(Environment* env, BWAPI::Player* player)
{
	std::set<BWAPI::UpgradeType> raceUpgrades, posibleUpgrades;

	for(std::set<BWAPI::UpgradeType>::iterator i = BWAPI::UpgradeTypes::allUpgradeTypes().begin(); i != BWAPI::UpgradeTypes::allUpgradeTypes().end(); ++i)
	{
		UpgradeType type = (*i);

		if(type.getRace() != player->getRace()) continue;
		if (type == UpgradeTypes::None) continue;

		raceUpgrades.insert(type);
	}

	for(std::set<BWAPI::UpgradeType>::iterator i = raceUpgrades.begin(); i!=raceUpgrades.end(); ++i)
	{
		const BWAPI::UpgradeType& upgrade = *i;
		bool inProgress = upgradeInProgress.find(upgrade) != upgradeInProgress.end();

		if(inProgress || player->getUpgradeLevel(upgrade) >= upgrade.maxRepeats() ) continue;

		bool hasTypeUpgraded = false;
		std::set<UnitType> typesUpgraded = upgrade.whatUses();

		for(std::set<UnitType>::iterator it = typesUpgraded.begin(); it != typesUpgraded.end(); it++)
			if (hasTypeUpgraded = env->getFilteredAgents(*it).size() > 0) break;

		if(!hasTypeUpgraded) continue;

		UnitType buildingType = upgrade.whatUpgrades();
		AgentMap buildings = env->getFilteredAgents(buildingType);
		bool hasBuilding = buildings.size() > 0;

		if(hasBuilding)
		{
			UnitType requiredBuildingType = upgrade.whatsRequired(player->getUpgradeLevel(upgrade) + 1);
			if(requiredBuildingType != UnitTypes::None)
			{
				AgentMap requiredBuildings = env->getFilteredAgents(requiredBuildingType);
				bool hasRequiredBuilding = requiredBuildings.size() > 0;
				if(!hasRequiredBuilding)
					continue;
			}

			for(AgentMap::iterator it = buildings.begin(); it != buildings.end(); ++it)
				if((*it).first->isIdle())
				{
					
					//AiLogger::taskLog.Log(upgrade.getName().append(" Research created").c_str());
					env->addTask(new ResearchTask(upgrade, BWAPI::TechTypes::None, env));
					upgradeInProgress.insert(upgrade);
					return true;
				}
		}
	}
	return false;
}

void ResearchTask::generateNecessaryTasks(Environment* env)
{
	AgentMap attackers = env->getFilteredAgents(ATTACKER_AGENT);
	int attackerCount = attackers.size() - Agent::filterMap(&attackers, WORKER_AGENT).size();

	int minAttackers = MIN_ATTACKER_AGENTS;

	if (attackerCount < minAttackers) return;

	BWAPI::Player* player = env->getPlayer();
	if(!generateTechs(env, player))
		generateUpgrades(env, player);
}

ResearchTask::ResearchTask(const BWAPI::UpgradeType upgrade, const BWAPI::TechType tech, Environment* env) : Task(env, "Research"), _upgrade(upgrade), _tech(tech)
{
	std::string name = (upgrade != BWAPI::UpgradeTypes::None) ? upgrade.getName() : tech.getName();
	_name = name.insert(0,"Research ").c_str();
	_types.insert(RESEARCH_TASK);
}

ResearchTask::~ResearchTask(void)
{
	if(_upgrade != BWAPI::UpgradeTypes::None)
		upgradeInProgress.erase(_upgrade);
	else
		techInProgress.erase(_tech);
}

double ResearchTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	UnitType buildingType =  _upgrade != BWAPI::UpgradeTypes::None ? _upgrade.whatUpgrades() : _tech.whatResearches();
	if (type != buildingType)	return NO_APTITUDE;
	if (unit->isUpgrading())	return NO_APTITUDE;
	if (unit->isResearching())  return NO_APTITUDE;

	double aptitude = 1.0f;

	return aptitude;
}

void ResearchTask::evaluateStatus()
{
	_canExecute = (_agents.size() >= 1);
	if (!_canExecute)
	{
		//AiLogger::taskLog.Log("Research No unit assigned.");
		_statusMessage = "No unit assigned.";
		_status = HALTED;
		return;
	}

	Unit* unit = (*_agents.begin())->getUnit();
	if (!unit->exists())
	{
		//AiLogger::taskLog.Log("Research Cancelled no units");
		_statusMessage = "Unit doesn't exist.";
		_status = CANCELLED;
		return;
	}

	if(_status != IN_PROGRESS)
	{
		if(_status == NOT_STARTED/*!unit->isUpgrading()*/)
		{
			int gasCost = _upgrade != BWAPI::UpgradeTypes::None ? _upgrade.gasPrice() : _tech.gasPrice();
			int mineralCost = _upgrade != BWAPI::UpgradeTypes::None ? _upgrade.mineralPrice() : _tech.mineralPrice();

			int currentMinerals = _env->getPlayer()->minerals();
			int reservedMinerals = _env->getReservedMinerals();

			int currentGas = _env->getPlayer()->gas();
			int reservedGas = _env->getReservedGas();

			int excessMinerals = max(0, currentMinerals - reservedMinerals);
			int excessGas = max(0, currentGas - reservedGas);

			_canExecute = ((excessMinerals >= mineralCost) && (excessGas >= gasCost));

			double percentCollected = 1.0;

			if ((mineralCost > 0) && (gasCost > 0))
			{
				percentCollected = min(excessMinerals/(double)mineralCost, excessGas/(double)gasCost);
				percentCollected = min(percentCollected, 1.0);
			}

			_env->reserveMinerals((int)(mineralCost*percentCollected + RESOURCE_BONUS));
			_env->reserveGas((int)(gasCost*percentCollected + RESOURCE_BONUS));

			if (!_canExecute)
			{
				//AiLogger::taskLog.Log("Research No resources.");
				_statusMessage = "Not enough resources.";
				_status = HALTED;
			}
			return;
		}
	}
	else if ((_upgrade != BWAPI::UpgradeTypes::None  && unit->getRemainingUpgradeTime() <= 0) ||
		(_tech != BWAPI::TechTypes::None && unit->getRemainingResearchTime() <= 0))
	{
		//AiLogger::taskLog.Log("Research Done.");
		_statusMessage = "Done.";
		_status = FINISHED;
		return;
	}

	//AiLogger::taskLog.Log("Research in progress.");
	_statusMessage = "No issues.";
	//_status = IN_PROGRESS;
}

void ResearchTask::evaluateNeededUnits()
{
	_needsMoreUnits = (_agents.size() < 1);
}

void ResearchTask::calculatePriority()
{
	int researchesInProgress = _env->getFilteredTasks(RESEARCH_TASK, _creationFrame).size();

	AgentMap attackers = _env->getFilteredAgents(ATTACKER_AGENT);
	int attackerCount = attackers.size() - Agent::filterMap(&attackers, WORKER_AGENT).size() - MIN_ATTACKER_AGENTS;
	attackerCount = max(0, attackerCount);

	double prePriority = TASK_PRIORITY_FACTOR*attackerCount/(double)(CONCURRENT_TASK_FACTOR + researchesInProgress);

	_priority = scalePriority(prePriority, MAX_RESEARCH_PRIORITY, MIN_RESEARCH_PRIORITY);
}

bool ResearchTask::execute(Agent* agent)
{
	//AiLogger::taskLog.Log("Execute()");
	if (_status == IN_PROGRESS) return true;
	//AiLogger::taskLog.Log("Execute not in progress");

	Unit* unit = agent->getUnit();
	if (!unit->exists())
	{
		//AiLogger::taskLog.Log("Research Cancelled no units");
		_statusMessage = "Unit doesn't exist.";
		_status = CANCELLED;
		return false;
	}
	if (!unit->isIdle()) return true;
	//AiLogger::taskLog.Log("Execute upgrade");

	bool executeSucessful = _upgrade != BWAPI::UpgradeTypes::None ? unit->upgrade(_upgrade) : unit->research(_tech);
	if(!executeSucessful)
	{	//AiLogger::taskLog.Log("Execute cancelled");

		_status = CANCELLED;
		return false;
	}
	//AiLogger::taskLog.Log("Execute set to in progress");
	_status = IN_PROGRESS;
	return true;
}

