#include "RecollectMinTask.h"
#include "Environment.h"
#include "BaseAgent.h"
#include "Agent.h"

#define TASK_PRIORITY_FACTOR		5.0
#define WORKER_AGENTS_PER_PATCH		3
#define MINERALS_PER_WORKER_AGENT	8
#define MIN_MINERAL_WORKERS			5

void RecollectMinTask::generateNecessaryTasks(Environment* env)
{
	AgentMap bases = env->getFilteredAgents(BASE_AGENT);

	for(AgentMap::iterator i = bases.begin(); i != bases.end(); i++)
	{
		BaseAgent* base = (BaseAgent*)(*i).second;

		if (base->getUnit()->isLifted()) continue;
	
		UnitSet* minerals = base->getMineralPatches();

		for(UnitSet::iterator i = minerals->begin(); i != minerals->end(); i++)
		{
			Unit* mineral = (*i);

			if (!mineral->exists()) continue;

			if (env->isTaskTarget(mineral, RECOLLECT_MIN_TASK)) continue;

			//Broodwar->printf("Generating RecollectMinTask task.");
			env->addTask(new RecollectMinTask(mineral, env));

			////Create just one task at a time
			//return;
		}
	}
}

RecollectMinTask::RecollectMinTask(Unit* resource, Environment* env) : Task(env, "Recollect Minerals")
{
	_types.insert(RECOLLECT_TASK);
	_types.insert(RECOLLECT_MIN_TASK);

	_resource = resource;

	_maxReqUnits = WORKER_AGENTS_PER_PATCH;

	_env->incRequiredMineralWorkers(_maxReqUnits);

	_env->addTaskTarget(_resource, RECOLLECT_MIN_TASK);

	_reqUnits = 0;
}

RecollectMinTask::~RecollectMinTask(void)
{
	_env->removeTaskTarget(_resource, RECOLLECT_MIN_TASK);

	_env->decRequiredMineralWorkers(_maxReqUnits);
}

Unit* RecollectMinTask::getResource()
{
	return _resource;
}

double RecollectMinTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (!unit->getType().isWorker()) return NO_APTITUDE;

	if (!unit->hasPath(_resource)) return NO_APTITUDE;

	double dist = _resource->getDistance(unit);

	return (double)DIST_APTITUDE_FACTOR/(double)(1 + dist);
}

void RecollectMinTask::calculatePriority()
{
	int gasWorkers = _env->getFilteredAgents(RECOLLECT_GAS_TASK).size() + MIN_MINERAL_WORKERS;
	int minWorkers = _env->getFilteredAgents(RECOLLECT_MIN_TASK).size();
	int recWorkers = gasWorkers + minWorkers;

	int workers = max(gasWorkers - (int)_agents.size(), 0);

	double workerFactor = (double)workers/(double)recWorkers;

	int currentMinerals = max(1, _env->getCurrentMinerals());

	int requiredMinerals = _env->getRequiredMinerals();

	double mineralFactor = requiredMinerals/(double)(currentMinerals + requiredMinerals);

	double prePriority = TASK_PRIORITY_FACTOR*mineralFactor*workerFactor;

	_priority = scalePriority(prePriority, MAX_MINERAL_PRIORITY, MIN_MINERAL_PRIORITY);
}

void RecollectMinTask::evaluateStatus()
{
	if (!_resource->exists())
	{
		_statusMessage = "No more minerals.";
		_status = FINISHED;
		return;
	}

	_canExecute = true;
	_statusMessage = "No issues.";
	_status = IN_PROGRESS;

	int requiredMineralWorkers = _env->getRequiredMineralWorkers();
	int requiredWorkers = requiredMineralWorkers + _env->getRequiredGasWorkers();

	double reqRatio = (double)requiredMineralWorkers/(double)requiredWorkers;

	int workerCount = _env->getFilteredAgents(WORKER_AGENT).size();

	_reqUnits = max((int)(workerCount*reqRatio), 1); //there should be always need for at least one worker

	_reqUnits = min(_reqUnits, _maxReqUnits); // don't require more workers than _maxReqUnits
}

void RecollectMinTask::evaluateNeededUnits()
{
	_needsMoreUnits = (_agents.size() < (unsigned int)_reqUnits);
}

bool RecollectMinTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (unit->isGatheringMinerals()) return true;

	return unit->gather(_resource);
}

