#include "RecollectGasTask.h"
#include "Environment.h"
#include "BaseAgent.h"
#include "Agent.h"

#define TASK_PRIORITY_FACTOR		500.0
#define WORKER_AGENTS_PER_REFINERY	3
//#define GAS_PER_WORKER_AGENT		8
#define MIN_MINERAL_WORKERS			1

void RecollectGasTask::generateNecessaryTasks(Environment* env)
{
	AgentMap bases = env->getFilteredAgents(BASE_AGENT);

	for(AgentMap::iterator i = bases.begin(); i != bases.end(); i++)
	{
		BaseAgent* base = (BaseAgent*)(*i).second;

		if (base->getUnit()->isLifted()) continue;
	
		UnitSet* refineries = base->getRefineries();

		for(UnitSet::iterator i = refineries->begin(); i != refineries->end(); i++)
		{
			Unit* refinery = (*i);

			if (!refinery->exists()) continue;

			if (env->isTaskTarget(refinery, RECOLLECT_GAS_TASK)) continue;

			//Broodwar->printf("Generating RecollectGasTask task.");
			env->addTask(new RecollectGasTask(refinery, env));

			//Create just one task at a time
			return;
		}
	}
}

RecollectGasTask::RecollectGasTask(Unit* resource, Environment* env) : Task(env, "Recollect Gas")
{
	_types.insert(RECOLLECT_TASK);
	_types.insert(RECOLLECT_GAS_TASK);

	_resource = resource;

	_maxReqUnits = WORKER_AGENTS_PER_REFINERY;

	_env->incRequiredGasWorkers(_maxReqUnits);

	_env->addTaskTarget(_resource, RECOLLECT_GAS_TASK);

	_reqUnits = 0;
}

RecollectGasTask::~RecollectGasTask(void)
{
	_env->removeTaskTarget(_resource, RECOLLECT_GAS_TASK);

	_env->decRequiredGasWorkers(_maxReqUnits);
}

Unit* RecollectGasTask::getResource()
{
	return _resource;
}

double RecollectGasTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (!unit->getType().isWorker()) return NO_APTITUDE;

	if (!unit->hasPath(_resource)) return NO_APTITUDE;

	double dist = _resource->getDistance(unit);

	return (double)DIST_APTITUDE_FACTOR/(double)(1 + dist);
}

void RecollectGasTask::calculatePriority()
{
	int gasWorkers = _env->getFilteredAgents(RECOLLECT_GAS_TASK).size() + MIN_MINERAL_WORKERS;
	int minWorkers = _env->getFilteredAgents(RECOLLECT_MIN_TASK).size();
	int recWorkers = gasWorkers + minWorkers;

	int workers = max(minWorkers - (int)_agents.size(), 0);

	double workerFactor = (double)workers/(double)recWorkers;

	int currentGas = max(1, _env->getCurrentGas());
	int requiredGas = max(1, _env->getRequiredGas());

	double gasFactor = requiredGas/(double)(currentGas + requiredGas);

	double prePriority = TASK_PRIORITY_FACTOR*gasFactor*workerFactor;

	_priority = scalePriority(prePriority, MAX_GAS_PRIORITY, MIN_GAS_PRIORITY);
}

void RecollectGasTask::evaluateStatus()
{
	//NOTE: Refineries still generate some gas after being spent
	if (!_resource->exists())
	{
		_statusMessage = "No more gas.";
		_status = FINISHED;
		return;
	}

	_canExecute = true;
	_statusMessage = "No issues.";
	_status = IN_PROGRESS;

	int requiredGasWorkers = _env->getRequiredGasWorkers();
	int requiredWorkers = requiredGasWorkers + _env->getRequiredMineralWorkers();

	double reqRatio = (double)requiredGasWorkers/(double)requiredWorkers;

	int workerCount = _env->getFilteredAgents(WORKER_AGENT).size();

	_reqUnits = max((int)(workerCount*reqRatio), 1); //there should be always need for at least one worker

	_reqUnits = min(_reqUnits, _maxReqUnits); // don't require more workers than _maxReqUnits
}

void RecollectGasTask::evaluateNeededUnits()
{
	_needsMoreUnits = (_agents.size() < (unsigned int)_reqUnits);
}

bool RecollectGasTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (unit->isGatheringGas())
	{
		Unit* target = unit->getTarget();

		if (target == NULL) return true;

		if (!target->getType().isRefinery()) return true;
		
		if (target == _resource) return true;
	}

	return unit->gather(_resource);
}

