#include "ProtectUnitTask.h"
#include "Environment.h"
#include "Agent.h"

#define TASK_PRIORITY_FACTOR			1
#define VISION_FACTOR					1
#define FLYING_FACTOR					2
#define SPEED_FACTOR					0.25
#define ENEMY_VALUE_FACTOR				1
#define PROTECTOR_VALUE_FACTOR			1
#define PASSIVE_MINIMUM_RANGE_FACTOR	0.5
#define ACTIVE_MINIMUM_RANGE_FACTOR		0.2
#define MINIMUM_DIST					64.0
#define ACTIVE_TASK_VALIDATION_FREQ		10
#define PASSIVE_TASK_VALIDATION_FREQ	DEFAULT_TASK_VALIDATION_FREQ
#define MAX_AGENTS						200

#define VIEW_RANGE_FACTOR				2.0

#define BASE_RADIUS						10
#define MINIMUM_RANGE_FACTOR			10
#define MINIMUM_RANGE_INCREASE			0.25

#define CHANGE_CENTER_AGENT_FREQ		200

void ProtectUnitTask::generateNecessaryTasks(Environment* env)
{
	AgentMap* agents = env->getAgents();

	for(AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;
		Unit* unit = agent->getUnit();
		UnitType type = unit->getType();

		//if (!type.isBuilding() && !type.isWorker()) continue;
		if (!type.isResourceDepot()) continue;

		//if (!agent->isFinished()) continue;

		if (agent->getProtectUnitTask() != NULL) continue;

		env->addTask(new ProtectUnitTask(agent, env));
	}
}

ProtectUnitTask::ProtectUnitTask(Agent* agent, Environment* env) : Task(env, "Protect Unit ")
{
	_types.insert(PROTECT_UNIT_TASK);

	Unit* vUnit = agent->getUnit();

	Position pos = vUnit->getPosition();

	char coordinates[20];

	sprintf_s(coordinates, 20, "%d,%d", pos.x(), pos.y());
	
	_name.append(coordinates);
	
	_protectedAgent = agent;
	_protectedAgent->setProtectUnitTask(this);

	_groupPosition = Positions::None;
	_readyRange = 0;
	_centerAgent = NULL;
}

ProtectUnitTask::~ProtectUnitTask(void)
{
	_protectedAgent->setProtectUnitTask(NULL);
}

void ProtectUnitTask::addAgent(Agent* agent)
{
	_agents.insert(agent);
	agent->setDisposition(BRAVE);
}

void ProtectUnitTask::removeAgent(Agent* agent)
{
	if (agent == _centerAgent) _centerAgent = NULL;

	agent->setDisposition(COWARD);
	_agents.erase(agent);
}

Agent* ProtectUnitTask::getProtectedAgent()
{
	return _protectedAgent;
}

Unit* ProtectUnitTask::getTargetEnemy()
{
	return _targetEnemy;
}

int ProtectUnitTask::getReadyRange()
{
	return _readyRange;
}

Position ProtectUnitTask::getGroupPosition()
{
	return _groupPosition;
}

double ProtectUnitTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();
	Unit* vUnit = _protectedAgent->getUnit();

	if (type.isBuilding()) return NO_APTITUDE;

	if (type.isWorker()) return NO_APTITUDE;

	Position pos = vUnit->getPosition();

	if (pos == Positions::None) return NO_APTITUDE;

	double dist = unit->getDistance(pos);

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	if (type.isFlyer())
	{
		aptitude *= FLYING_FACTOR;
	}
	else
	{
		if (!pos.hasPath(unit->getPosition())) return NO_APTITUDE;
	}

	aptitude *= VISION_FACTOR*type.sightRange();
	aptitude *= SPEED_FACTOR*type.topSpeed();

	return aptitude;
}

void ProtectUnitTask::calculatePriority()
{
	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	double agentValue = Environment::getUnitTypeValue(vType);
	//double agentValue = Environment::getUnitHealthValue(vUnit)*Environment::getUnitTypeValue(vUnit->getType());

	UnitSet enemies = _protectedAgent->getEnemiesInRange((int)(vType.sightRange()*VIEW_RANGE_FACTOR));
	
	_targetEnemy = NULL;

	_execValidationFreq = PASSIVE_TASK_VALIDATION_FREQ;

	double enemiesValue = 0.0;
	double highestValue = 0.0;
	for(UnitSet::iterator i = enemies.begin(); i != enemies.end(); i++)
	{
		Unit* enemy = (*i);

		double value = Environment::getUnitCombatValue(enemy);
		//double value = Environment::getUnitHealthValue(enemy)*Environment::getUnitTypeValue(enemy->getType());
		enemiesValue += value;

		if (highestValue < value)
		{
			highestValue = value;
			_targetEnemy = enemy;
		}

		_execValidationFreq = ACTIVE_TASK_VALIDATION_FREQ;
	}

	double protectorsValue = 0.0;
	for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Unit* protector = (*i)->getUnit();

		protectorsValue += Environment::getUnitCombatValue(protector);
		//protectorsValue += Environment::getUnitHealthValue(protector)*Environment::getUnitTypeValue(protector->getType());
	}

	double unitsValue = 1.0 + max(0.0, ENEMY_VALUE_FACTOR*enemiesValue - PROTECTOR_VALUE_FACTOR*protectorsValue);
	double prePriority = TASK_PRIORITY_FACTOR*agentValue*unitsValue;

	if (_targetEnemy != NULL)
	{
		_priority = scalePriority(prePriority, MAX_ACTIVE_PROTECT_PRIORITY, MIN_ACTIVE_PROTECT_PRIORITY);
	}
	else
	{
		_priority = scalePriority(prePriority, MAX_PASSIVE_PROTECT_PRIORITY, MIN_PASSIVE_PROTECT_PRIORITY);
	}
}

void ProtectUnitTask::evaluateStatus()
{
	_readyRange = 0;
	_groupPosition = Positions::None;
	_canExecute = true;

	_neededUnits = 0;

	TaskSet tasks = _env->getFilteredTasks(PROTECT_UNIT_TASK);
	AgentMap agents = _env->getFilteredAgents(ATTACKER_AGENT);
	AgentMap wAgents = Agent::filterMap(&agents, WORKER_AGENT);

	int numAgents = agents.size() - wAgents.size();

	double totalPriority = 0.0;
	for(TaskSet::iterator i = tasks.begin(); i != tasks.end(); i++)
	{
		Task* task = (*i);

		totalPriority += task->getPriority();
	}

	double prioFactor = 0;
	if (totalPriority > 0) prioFactor = _priority/totalPriority;

	_neededUnits = min(MAX_AGENTS, (int)(numAgents*prioFactor));

	// Calculate group center
	int agentCount = _agents.size();

	if (agentCount == 0) return;

	double rangeFactor = MINIMUM_RANGE_FACTOR + MINIMUM_RANGE_INCREASE*agentCount;
	_readyRange = (int)(BASE_RADIUS * rangeFactor);

	Position bestPosition = Positions::None;

	if (_targetEnemy == NULL)
	{
		bestPosition = _env->getBestOpenSpace(_protectedAgent->getUnitPosition());
		
		if (bestPosition != Positions::None) _groupPosition = bestPosition;

		return;
	}

	int sumX = 0;
	int sumY = 0;
	for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Position pos = (*i)->getUnitPosition();

		sumX += pos.x();
		sumY += pos.y();
	}

	Position centroid = Position(sumX/agentCount, sumY/agentCount);
	_groupPosition = centroid;

	if ((_centerAgent == NULL) ||
		((Broodwar->getFrameCount()%CHANGE_CENTER_AGENT_FREQ == 0) &&
		(_targetEnemy == NULL)))
	{
		double closestDistance = MAX_POS_DISTANCE;
		double slowestSpeed = MAX_POS_DISTANCE;
		int healthiest = 0;
		for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
		{
			Agent* agent = *i;
			Unit* unit = agent->getUnit();
			Position pos = unit->getPosition();
			double distance = pos.getDistance(centroid);
			UnitType type = unit->getType();

			if (type.isFlyer()) continue;
			if (!type.canAttack()) continue;

			int health = unit->getHitPoints() + unit->getShields();
			double speed = type.topSpeed();

			if (health < healthiest) continue;
			if ((health == healthiest) && (slowestSpeed < speed)) continue;
			if ((health == healthiest) && (slowestSpeed == speed) && (closestDistance < distance)) continue;

			healthiest = health;
			slowestSpeed = speed;
			_centerAgent = agent;
			closestDistance = distance;
			bestPosition = pos;
		}
	}
	else
	{
		bestPosition = _centerAgent->getUnitPosition();
	}
		
	if (bestPosition != Positions::None) _groupPosition = bestPosition;
}

void ProtectUnitTask::evaluateNeededUnits()
{
	_needsMoreUnits = ((int)_agents.size() < _neededUnits);
}

//bool ProtectUnitTask::execute(Agent* agent)
//{
//	Unit* unit = agent->getUnit();
//	UnitType type = unit->getType();
//	Unit* vUnit = _protectedAgent->getUnit();
//	UnitType vType = vUnit->getType();
//
//	Position uPos = unit->getPosition();
//
//	double distance = uPos.getDistance(_groupPosition);
//	double vDistance = unit->getDistance(vUnit);
//
//	double readyRange = _readyRange;
//
//	if (type == UnitTypes::Terran_Medic)
//	{
//		readyRange *= 0.30;
//	}
//
//	bool shouldAct = false;
//	if ((_targetEnemy != NULL) && (vDistance < distance) && type.canAttack()) shouldAct = true;
//	if (distance <= readyRange)	shouldAct = true;
//	
//	if (shouldAct)
//	{
//		if (_targetEnemy != NULL)
//		{
//			if (type.canAttack())
//			{
//				Unit* target = agent->getBestTargetInAttackRange();
//
//				if (target == NULL) target = _targetEnemy;
//
//				if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
//				
//				return unit->attack(target);
//			}
//			else if (type == UnitTypes::Terran_Medic)
//			{
//				Order order = unit->getOrder();
//
//				if (order != Orders::HealMove) return unit->stop();
//			}
//			else
//			{
//				Order order = unit->getOrder();
//
//				if (order != Orders::None) return unit->stop();
//			}
//		}
//
//		return true;
//	}
//	
//	//If not in range, move to range
//	return unit->move(_groupPosition);
//}

bool ProtectUnitTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (type == UnitTypes::Terran_Marine) return marineExecute(agent);
	if (type == UnitTypes::Terran_Medic) return medicExecute(agent);
	if (type == UnitTypes::Terran_Wraith) return wraithExecute(agent);
	if (type == UnitTypes::Terran_Siege_Tank_Siege_Mode) return siegeExecute(agent);
	if (type == UnitTypes::Terran_Siege_Tank_Tank_Mode) return tankExecute(agent);

	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool ProtectUnitTask::marineExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();
	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	Position uPos = unit->getPosition();

	double distance = uPos.getDistance(_groupPosition);
	double vDistance = unit->getDistance(vUnit);

	double readyRange = _readyRange;

	bool shouldAct = false;
	if ((_targetEnemy != NULL) && (vDistance < distance)) shouldAct = true;
	if (distance <= readyRange)	shouldAct = true;
	
	if (shouldAct)
	{
		if (_targetEnemy != NULL)
		{
			Unit* target = agent->getBestTargetInAttackRange();

			if (target == NULL) target = _targetEnemy;

			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
			
			return unit->attack(target);
		}

		return true;
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool ProtectUnitTask::medicExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();
	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	Position uPos = unit->getPosition();

	double distance = uPos.getDistance(_groupPosition);
	double vDistance = unit->getDistance(vUnit);

	double readyRange = _readyRange*0.30;

	bool shouldAct = false;
	if (distance <= readyRange)	shouldAct = true;
	
	if (shouldAct)
	{
		if (_targetEnemy != NULL)
		{
			Order order = unit->getOrder();

			if (order != Orders::HealMove) return unit->stop();
		}

		return true;
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool ProtectUnitTask::wraithExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();
	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	Position uPos = unit->getPosition();

	double distance = uPos.getDistance(_groupPosition);
	double vDistance = unit->getDistance(vUnit);

	double readyRange = _readyRange;

	bool shouldAct = false;
	if ((_targetEnemy != NULL) && (vDistance < distance)) shouldAct = true;
	if (distance <= readyRange)	shouldAct = true;
	
	if (shouldAct)
	{
		if (_targetEnemy != NULL)
		{
			if (!unit->isCloaked() && unit->canIssueCommand(UnitCommand(unit, UnitCommandTypes::Cloak, NULL, 0, 0, 0)))
			{
				return unit->cloak();
			}

			Unit* target = _targetEnemy;

			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
			
			return unit->attack(target);
		}
	}

	if (unit->isCloaked())
	{
		return unit->decloak();
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool ProtectUnitTask::tankExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (unit->isSieged()) return true;

	if (!unit->canIssueCommand(UnitCommand(unit, UnitCommandTypes::Siege, NULL, 0, 0, 0)))
	{
		return marineExecute(agent);
	}

	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	Position uPos = unit->getPosition();

	double distance = uPos.getDistance(_groupPosition);
	double vDistance = unit->getDistance(vUnit);

	double readyRange = _readyRange;

	bool shouldAct = false;
	if ((_targetEnemy != NULL) && (vDistance < distance)) shouldAct = true;
	if (distance <= readyRange)	shouldAct = true;
	
	if (shouldAct)
	{
		if (_targetEnemy != NULL)
		{
			Unit* target = agent->getBestTargetInAttackRange(true, true);

			if (target != NULL)
			{
				return unit->siege();
			}

			target = agent->getBestTargetInAttackRange(true, false);

			if (target != NULL)
			{
				if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
				
				return unit->attack(target);
			}
			
			target = agent->getBestTargetInAttackRange(false, true);

			if (target != NULL)
			{
				return unit->siege();
			}

			target = agent->getBestTargetInAttackRange(false, false);

			if (target != NULL)
			{
				if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
				
				return unit->attack(target);
			}

			target = _targetEnemy;

			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
			
			return unit->attack(target);
		}

		return unit->siege();
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool ProtectUnitTask::siegeExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (!unit->isSieged()) return true;

	Unit* vUnit = _protectedAgent->getUnit();
	UnitType vType = vUnit->getType();

	Position uPos = unit->getPosition();

	double distance = uPos.getDistance(_groupPosition);
	double vDistance = unit->getDistance(vUnit);

	double readyRange = _readyRange*2;

	bool shouldAct = false;
	if ((_targetEnemy != NULL) && (vDistance < distance)) shouldAct = true;
	if (distance <= readyRange)	shouldAct = true;
	
	if (shouldAct)
	{
		Unit* target = agent->getBestTargetInAttackRange(true, true);

		if (target != NULL)
		{
			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;

			return unit->attack(target);
		}

		target = agent->getBestTargetInAttackRange(true, false);

		if (target != NULL) return unit->unsiege();

		target = agent->getBestTargetInAttackRange(false, true);

		if (target != NULL)
		{
			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;

			return unit->attack(target);
		}

		target = agent->getBestTargetInAttackRange(false, false);

		if (target != NULL) return unit->unsiege();

		return true;
	}

	return unit->unsiege();
}

