#include "MoveCommandTask.h"
#include "Environment.h"
#include "BaseAgent.h"
#include "Agent.h"
#include "AiLogger.h"

#define TASK_PRIORITY_FACTOR		1000.0
#define MAX_MOVE_TASKS				1
#define MAX_MINERALS				0
#define GENERATE_FREQ				1000

void MoveCommandTask::generateNecessaryTasks(Environment* env)
{
	if (Broodwar->getFrameCount()%GENERATE_FREQ != 0) return;

	AgentMap bases = env->getFilteredAgents(BASE_AGENT);
	
	int futureMoves = env->getFilteredTasks(MOVE_COMMAND_TASK).size();

	if (futureMoves >= MAX_MOVE_TASKS) return;

	int exhaustedBases = 0;
	for(AgentMap::iterator i = bases.begin(); i != bases.end(); i++)
	{
		BaseAgent* base = (BaseAgent*)(*i).second;
		
		if (base->getMineralPatches()->size() <= MAX_MINERALS) exhaustedBases++;
	}

	if (exhaustedBases <= 0) return;

	while (futureMoves < MAX_MOVE_TASKS)
	{
		TilePosition position = env->getBestCommandPosition();

		if (position == TilePositions::None) break;

		//Broodwar->printf("Generating MoveCommandTask task.");
		env->addTask(new MoveCommandTask(position, env));

		env->generateValidPositions();
		futureMoves++;
	}
}

MoveCommandTask::MoveCommandTask(TilePosition pos, Environment* env) : Task(env, "Move ")
{
	_types.insert(MOVE_COMMAND_TASK);

	_buildingType = UnitTypes::Terran_Command_Center;
	_name.append(_buildingType.getName());

	_buildingPos = pos;

	_plannedBuilding = new BuildingSpaceInfo(pos, _buildingType);

	_env->addPlannedBuilding(_plannedBuilding);
		
	int posX = _buildingPos.x() + _buildingType.tileWidth()/2;
	int posY = _buildingPos.y() + _buildingType.tileHeight()/2;

	_targetPosition = Position(posX*32, posY*32);
}

MoveCommandTask::~MoveCommandTask(void)
{
	_env->removePlannedBuilding(_plannedBuilding);

	delete _plannedBuilding;
}

void MoveCommandTask::calculatePriority()
{
	double prePriority = TASK_PRIORITY_FACTOR;

	_priority = scalePriority(prePriority, MAX_MOVE_PRIORITY, MIN_MOVE_PRIORITY);
}

double MoveCommandTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (unit->getType() != _buildingType) return NO_APTITUDE;

	BaseAgent* base = (BaseAgent*)agent;

	if (base->getMineralPatches()->size() > MAX_MINERALS) return NO_APTITUDE;

	if (!base->isFinished() || unit->isTraining() || unit->isConstructing()) return NO_APTITUDE;

	Position bPos = Position(_buildingPos);
	Position unitPos = unit->getPosition();

	double dist = bPos.getDistance(unitPos);

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	return aptitude;
}

void MoveCommandTask::evaluateStatus()
{
	_canExecute = true;
	_statusMessage = "None specified.";

	if (_agents.size() < 1)
	{
		_canExecute = false;
		_statusMessage = "No unit assigned.";
		return;
	}

	Agent* command = *(_agents.begin());
	Unit* cmdUnit = command->getUnit();

	if (_buildingPos == cmdUnit->getTilePosition())
	{
		if (!cmdUnit->isLifted())
		{
			_statusMessage = "Finished.";
			_status = FINISHED;
			return;
		}
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void MoveCommandTask::evaluateNeededUnits()
{
	if (_agents.size() < 1)	_needsMoreUnits = true;
	else					_needsMoreUnits = false;
}

TilePosition MoveCommandTask::getBuildingPosition()
{
	return _buildingPos;
}

UnitType MoveCommandTask::getBuildingType()
{
	return _buildingType;
}

bool MoveCommandTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (!unit->isLifted())
	{
		bool success = unit->lift();
		
		if (!success) _status = CANCELLED;
		return success;
	}

	if (_targetPosition != unit->getPosition())
	{
		if (_targetPosition != unit->getOrderTargetPosition())
		{
			bool success = unit->move(_targetPosition);
			
			if (!success) _status = CANCELLED;
			return success;
		}

		return true;
	}

	bool success = unit->land(_buildingPos);

	if (!success) _status = CANCELLED;
	return success;
}
