#include "ExploreSectorTask.h"
#include "Environment.h"
#include "Agent.h"
#include "MapSector.h"

#define TASK_PRIORITY_FACTOR		1
#define FLYING_FACTOR				2
#define SPEED_FACTOR				1
#define VISION_FACTOR				1
#define PERCENT_ACTIVE_UNITS		0.05
#define MIN_ACTIVE_UNITS			6
#define UNIT_COUNT_FACTOR			1
#define TASK_COUNT_FACTOR			10
#define WORKER_FACTOR				0.05
#define CORNER_FACTOR				1
#define TIMEOUT_EXPLORE_SECTOR_TASK 1200

#define COMSAT_APTITUDE				100000

void ExploreSectorTask::generateNecessaryTasks(Environment* env)
{
	int exploreSectorTaskCount = env->getFilteredTasks(EXPLORE_SECTOR_TASK).size();

	AgentMap agents = env->getFilteredAgents(WORKER_AGENT);

	int unitCount = (int)agents.size() - MIN_ACTIVE_UNITS;

	int comsatCount = env->getFilteredAgents(UnitTypes::Terran_Comsat_Station).size();
	int maxTasks = (int)ceil(max(0.0, (double)unitCount*PERCENT_ACTIVE_UNITS)) + comsatCount;

	if (exploreSectorTaskCount >= maxTasks) return;

	SectorList mapSectors = prioritizeSectors(env);

	bool needToReset = true;
	for(SectorList::iterator i = mapSectors.begin(); i != mapSectors.end(); i++)
	{
		needToReset = false;
		if (exploreSectorTaskCount >= maxTasks) break;

		MapSector* sector = (*i);

		//Broodwar->printf("Generating ExploreSectorTask task.");
		env->addTask(new ExploreSectorTask(sector, env));
		exploreSectorTaskCount++;
	}

	if (needToReset) env->resetQuadrants();
}

bool compareSectors(MapSector* first, MapSector* second)
{
	double firstVal = BASE_SECTOR_VALUE + first->getResourceValue() + first->getEnemyValue();
	if (first->getLastVisible() < 2) firstVal += NOT_EXPLORED_SECTOR_VALUE;

	firstVal *= first->getTimeNotVisible()*first->getDistanceFromStart()/(1.0 + first->getCornerDistance());

	double secondVal = BASE_SECTOR_VALUE + second->getResourceValue() + second->getEnemyValue();
	if (second->getLastVisible() < 2) secondVal += NOT_EXPLORED_SECTOR_VALUE;

	secondVal *= second->getTimeNotVisible()*second->getDistanceFromStart()/(1.0 + second->getCornerDistance());

	if (firstVal > secondVal) return true;

	return false;
}

SectorList ExploreSectorTask::prioritizeSectors(Environment* env)
{
	SectorList sectorList;

	MapSectorMap* mapSectors = env->getMapSectors();

	AgentMap agents = env->getFilteredAgents(EXPLORER_AGENT);

	for(MapSectorMap::iterator i = mapSectors->begin(); i != mapSectors->end(); i++)
	{
		MapSector* sector = (*i).second;

		TilePosition sectorCenter = sector->getCenterPosition();

		if (sectorCenter == TilePositions::None) continue;

		if (sector->getExploreSectorTask() != NULL) continue;

		if (sector->getEnemyValue() > 0) continue;

		int timeNotVisible = sector->getTimeNotVisible();

		if (timeNotVisible <= SECTOR_FRAMES_PER_EXPLORE) continue;

		Quadrant* quadrant = env->getQuadrant(sector);

		if (quadrant->beingExplored) continue;
		if (quadrant->visited) continue;
		//if (quadrant->dangerous) continue;

		bool canExplore = false;
		for(AgentMap::iterator i = agents.begin(); i != agents.end(); i++)
		{
			Agent* agent = (*i).second;

			if (evaluateAptitudeForTask(sector, agent) > NO_APTITUDE)
			{
				canExplore = true;
				break;
			}
		}
		
		if (canExplore) sectorList.push_back(sector);
	}

	sectorList.sort(compareSectors);

	return sectorList;
}

ExploreSectorTask::ExploreSectorTask(MapSector* sector, Environment* env) : Task(env, "Explore ")
{
	_types.insert(EXPLORE_SECTOR_TASK);

	TilePosition tPos = sector->getPosition();

	char coordinates[20];

	sprintf_s(coordinates, 20, "%d,%d", tPos.x()/SECTOR_LENGTH, tPos.y()/SECTOR_LENGTH);
	
	_name.append(coordinates);
	
	sector->setExploreSectorTask(this);
	_sector = sector;

	double sqMapWidth = Broodwar->mapWidth();
	sqMapWidth *= sqMapWidth;
	double sqMapHeight = Broodwar->mapHeight();
	sqMapHeight *= sqMapHeight;

	_maxDistance = sqrt(sqMapWidth + sqMapHeight);
	
	Quadrant* quadrant = _env->getQuadrant(_sector);
	quadrant->beingExplored = true;
}

ExploreSectorTask::~ExploreSectorTask(void)
{
	_sector->setExploreSectorTask(NULL);

	Quadrant* quadrant = _env->getQuadrant(_sector);
	quadrant->beingExplored = false;
}

MapSector* ExploreSectorTask::getSector()
{
	return _sector;
}

void ExploreSectorTask::removeAgent(Agent* agent)
{
	Quadrant* quadrant = _env->getQuadrant(_sector);

	Unit* unit = agent->getUnit();

	if ((unit->getHitPoints() < 1) ||
		(!unit->exists()) ||
		(unit->isUnderAttack()))
		_sector->incPossibleEnemyValue();

	quadrant->visited = true;

	_agents.erase(agent);
}

double ExploreSectorTask::evaluateAptitudeForTask(MapSector* sector, Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (sector->getEnemyValue() > 0)
	{
		if (type == UnitTypes::Terran_Comsat_Station)
		{
			if (unit->getEnergy() < type.maxEnergy()) return NO_APTITUDE;
			{
				return COMSAT_APTITUDE;
			}
		}

		return NO_APTITUDE;
	}

	if (type == UnitTypes::Terran_Comsat_Station)
	{
		return NO_APTITUDE;
	}

	if (!agent->isType(EXPLORER_AGENT)) return NO_APTITUDE;

	if (sector->getEnemyValue() > 0) return NO_APTITUDE;

	TilePosition tPos = sector->getCenterPosition();
	Position pos(tPos);

	if (tPos == TilePositions::None) return NO_APTITUDE;

	double dist = unit->getDistance(pos);

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	if (type.isFlyer())
	{
		aptitude *= FLYING_FACTOR;
	}
	else
	{
		if (!pos.hasPath(unit->getPosition())) return NO_APTITUDE;
	}

	if (type.isWorker()) aptitude *= WORKER_FACTOR;

	aptitude *= VISION_FACTOR*type.sightRange();
	aptitude *= SPEED_FACTOR*type.topSpeed();

	return aptitude;
}

double ExploreSectorTask::evaluateAptitude(Agent* agent)
{
	return evaluateAptitudeForTask(_sector, agent);
}

void ExploreSectorTask::calculatePriority()
{
	double value = BASE_SECTOR_VALUE + _sector->getResourceValue();

	if (_sector->getLastVisible() < 2) value += NOT_EXPLORED_SECTOR_VALUE;

	int exploreSectorTaskCountFactor = TASK_COUNT_FACTOR*_env->getFilteredTasks(EXPLORE_SECTOR_TASK, _creationFrame).size();
	int unitCountFactor = UNIT_COUNT_FACTOR*_env->getFilteredAgents(EXPLORER_AGENT).size();

	int countFactor = max(0, unitCountFactor - exploreSectorTaskCountFactor);

	double prePriority = TASK_PRIORITY_FACTOR*value*countFactor;

	_priority = scalePriority(prePriority, MAX_EXPLORE_PRIORITY, MIN_EXPLORE_PRIORITY);
}

void ExploreSectorTask::evaluateStatus()
{
	Quadrant* quadrant = _env->getQuadrant(_sector);

	int frameCount = Broodwar->getFrameCount();
	int timeActive = frameCount - _creationFrame;
	
	if (timeActive >= TIMEOUT_EXPLORE_SECTOR_TASK)
	{
		quadrant->visited = true;

		_statusMessage = "Timeout.";
		_status = CANCELLED;
		return;
	}
	
	if (Broodwar->isVisible(_sector->getCenterPosition()))
	{
		quadrant->visited = true;

		_statusMessage = "Completed.";
		_status = FINISHED;
		return;
	}

	if (quadrant->visited)
	{
		_statusMessage = "Quadrant dangerous.";
		_status = CANCELLED;
		return;
	}

	_canExecute = (_agents.size() >= 1);

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		_status = HALTED;
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void ExploreSectorTask::evaluateNeededUnits()
{
	_needsMoreUnits = (_agents.size() < 1);
}

bool ExploreSectorTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	TilePosition cTPos = _sector->getCenterPosition();
	Position cPos = Position(cTPos);
	Position tPos = unit->getOrderTargetPosition();

	if (type == UnitTypes::Terran_Comsat_Station)
	{
		return unit->useTech(TechTypes::Scanner_Sweep, cPos);
	}
	
	if (Broodwar->isVisible(cTPos))
	{
		//if (unit->isMoving()) unit->stop();
		return true;
	}
	
	if (unit->getOrder() == Orders::Move)
	{
		if (_sector->isInSector(tPos)) return true;
	}

	return unit->move(cPos);
}

