#include "BuildTask.h"
#include "BuildCommandTask.h"
#include "BuildBarracksTask.h"
#include "BuildSupplyTask.h"
#include "BuildRefineryTask.h"
#include "Environment.h"
#include "BaseAgent.h"
#include "Agent.h"

#define TIMEOUT_BUILD_TASK	3000
#define RESOURCE_BONUS		8

void BuildTask::generateUnfinishedBuildTasks(Environment* env)
{
	AgentMap buildings = env->getFilteredAgents(BUILDING_AGENT);

	for(AgentMap::iterator i = buildings.begin(); i != buildings.end(); i++)
	{
		Agent* buildingAgent = (*i).second;
		Unit* building = buildingAgent->getUnit();

		if (env->isTaskTarget(building, BUILD_TASK)) continue;

		if (!building->isConstructing() && !buildingAgent->isFinished())
		{
			UnitType type = building->getType();

			if (type == UnitTypes::Terran_Barracks)
			{
				env->addTask(new BuildBarracksTask(building, env));
			}
			else if (type == UnitTypes::Terran_Supply_Depot)
			{
				env->addTask(new BuildSupplyTask(building, env));
			}
			else if (type == UnitTypes::Terran_Refinery)
			{
				env->addTask(new BuildRefineryTask(building, env));
			}
			else if (type == UnitTypes::Terran_Command_Center)
			{
				env->addTask(new BuildCommandTask(building, env));
			}
		}
	}
}

BuildTask::BuildTask(UnitType buildingType, TilePosition pos, Environment* env) : Task(env, "Build ")
{
	_types.insert(BUILD_TASK);

	_workerType = Broodwar->self()->getRace().getWorker();

	_buildingType = buildingType;

	_buildingPos = pos;

	_building = NULL;

	_plannedBuilding = new BuildingSpaceInfo(pos, buildingType);

	initialize();
}

BuildTask::BuildTask(Unit* building, Environment* env) : Task(env, "Build ", false)
{
	_types.insert(BUILD_TASK);

	_workerType = Broodwar->self()->getRace().getWorker();

	_buildingPos = building->getTilePosition();

	//If it's not a vespene geyser, then it's an unfinished building we want to complete
	UnitType buildingType = building->getType();
	if (buildingType != UnitTypes::Resource_Vespene_Geyser)
	{
		_building = building;
		_buildingType = buildingType;

		_env->addTaskTarget(_building, BUILD_TASK);
	}
	else
	{
		_building = NULL;
		_buildingType = _env->getPlayer()->getRace().getRefinery();
	}

	_plannedBuilding = new BuildingSpaceInfo(_buildingPos, buildingType);

	initialize();
}

BuildTask::~BuildTask(void)
{
	_env->removePlannedBuilding(_plannedBuilding);

	delete _plannedBuilding;

	_env->decRequiredWorkers(1);
	
	_env->decRequiredMinerals(_buildingType.mineralPrice());
	_env->decRequiredGas(_buildingType.gasPrice());

	if (_building != NULL) _env->removeTaskTarget(_building, BUILD_TASK);
}

void BuildTask::initialize()
{
	_name.append(_buildingType.getName());

	_env->addPlannedBuilding(_plannedBuilding);

	_env->incRequiredWorkers(1);

	_env->incRequiredMinerals(_buildingType.mineralPrice());
	_env->incRequiredGas(_buildingType.gasPrice());

	_targetPosition = Positions::None;

	_requiresUnits = false;
}

double BuildTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (!unit->getType().isWorker()) return NO_APTITUDE;

	Position bPos = Position(_buildingPos);
	Position unitPos = unit->getPosition();

	double dist = bPos.getDistance(unitPos);

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	return aptitude;
}

void BuildTask::evaluateStatus()
{
	_canExecute = true;
	_statusMessage = "None specified.";
	_requiresUnits = false;

	bool hasUnits = _agents.size() >= 1;
	
	Player* player = _env->getPlayer();

	if (hasUnits)
	{
		Unit* worker = (*_agents.begin())->getUnit();

		if (_building == NULL) _building = worker->getBuildUnit();
	}

	if ((_building == NULL) || (_building->getRemainingBuildTime() == _buildingType.buildTime()))
	{
		int frameCount = Broodwar->getFrameCount();
		int timeActive = frameCount - _creationFrame;
		
		if (timeActive >= TIMEOUT_BUILD_TASK)
		{
			_statusMessage = "Timeout.";
			_status = CANCELLED;
			return;
		}

		int currentMinerals = player->minerals();
		int reservedMinerals = _env->getReservedMinerals();
		int mineralCost = _buildingType.mineralPrice();
		
		int currentGas = player->gas();
		int reservedGas = _env->getReservedGas();
		int gasCost = _buildingType.gasPrice();
		
		int excessMinerals = max(0, currentMinerals - reservedMinerals);
		int excessGas = max(0, currentGas - reservedGas);

		_canExecute = ((excessMinerals >= mineralCost) && (excessGas >= gasCost));

		double percentCollected = 1.0;

		if ((mineralCost > 0) && (gasCost > 0))
		{
			percentCollected = min(excessMinerals/(double)mineralCost, excessGas/(double)gasCost);
			percentCollected = min(percentCollected, 1.0);
		}

		_env->reserveMinerals((int)(mineralCost*percentCollected + RESOURCE_BONUS));
		_env->reserveGas((int)(gasCost*percentCollected + RESOURCE_BONUS));

		if (!_canExecute)
		{
			_statusMessage = "Not enough resources.";
			return;
		}

		_requiresUnits = true;
	}
	else
	{
		_requiresUnits = true;

		if (!_env->isTaskTarget(_building, BUILD_TASK))
		{
			_env->addTaskTarget(_building, BUILD_TASK);
		}

		if (_building->getRemainingBuildTime() <= 0)
		{
			_statusMessage = "Finished.";
			_status = FINISHED;
			return;
		}

		if (_building->isBeingConstructed())
			_canBePreempted = false;
		else
			_canBePreempted = true;
	}

	if (!hasUnits)
	{
		_canExecute = false;
		_statusMessage = "No unit assigned.";
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void BuildTask::evaluateNeededUnits()
{
	if (_agents.size() < 1)	_needsMoreUnits = _requiresUnits;
	else					_needsMoreUnits = false;
}

TilePosition BuildTask::getBuildingPosition()
{
	return _buildingPos;
}

UnitType BuildTask::getBuildingType()
{
	return _buildingType;
}

Unit* BuildTask::getBuilding()
{
	return _building;
}

bool BuildTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	Position tPos = unit->getOrderTargetPosition();

	int width = _buildingType.tileWidth();
	int height = _buildingType.tileHeight();

	Position bPos(_buildingPos);

	if (unit->isConstructing()) return true;

	if (_building != NULL) return unit->rightClick(_building);

	bool visible = true;
	for (int m = 0; m < width; m++)
	{
		for (int n = 0; n < height; n++)
		{
			if (!Broodwar->isVisible(_buildingPos.x() + m, _buildingPos.y() + n))
			{
				visible = false;
				break;
			}
		}

		if (!visible) break;
	}

	if (!visible)
	{
		if (unit->getOrder() == Orders::Move)
		{
			if (tPos == _targetPosition) return true;
		}

		bool success = unit->move(bPos);
		_targetPosition = unit->getOrderTargetPosition();

		if (!success) _status = CANCELLED;
		return success;
	}

	bool success = unit->build(_buildingPos, _buildingType);
	_targetPosition = unit->getOrderTargetPosition();

	if (!success) _status = CANCELLED;
	return success;
}
