#include "AttackUnitTask.h"
#include "Environment.h"
#include "Agent.h"

#define TASK_PRIORITY_FACTOR				1
#define FLYING_FACTOR						2
#define SPEED_FACTOR						0.25
#define WORKER_FACTOR						0.05
#define MIN_APTITUDE						0.4
#define ATTACK_UNIT_TASK_VALIDATION_FREQ	10

void AttackUnitTask::generateNecessaryTasks(Environment* env)
{
	UnitSet allUnits = Broodwar->getAllUnits();

	for(UnitSet::iterator i = allUnits.begin(); i != allUnits.end(); i++)
	{
		Unit* unit = *i;

		if (!env->isEnemyUnit(unit)) continue;

		if (!unit->isDetected()) continue;

		if (unit->isInvincible()) continue;
		
		if (env->isTaskTarget(unit, ATTACK_UNIT_TASK)) continue;

		env->addTask(new AttackUnitTask(unit, env));
	}
}

AttackUnitTask::AttackUnitTask(Unit* target, Environment* env) : Task(env, "Attack enemy")
{
	_types.insert(ATTACK_UNIT_TASK);
	
	_target = target;
	_minAptitude = MIN_APTITUDE;

	_execValidationFreq = ATTACK_UNIT_TASK_VALIDATION_FREQ;

	_env->addTaskTarget(_target, ATTACK_UNIT_TASK);
}

AttackUnitTask::~AttackUnitTask(void)
{
	_env->removeTaskTarget(_target, ATTACK_UNIT_TASK);
}

Unit* AttackUnitTask::getTarget()
{
	return _target;
}

void AttackUnitTask::addAgent(Agent* agent)
{
	_agents.insert(agent);
	agent->setDisposition(BRAVE);
}

void AttackUnitTask::removeAgent(Agent* agent)
{
	agent->setDisposition(COWARD);
	_agents.erase(agent);
}

double AttackUnitTask::evaluateAptitude(Agent* agent)
{
	if (!agent->isType(ATTACKER_AGENT)) return NO_APTITUDE;

	UnitType targetType = _target->getType();
	bool targetIsFlier = targetType.isFlyer();
	bool targetIsLifted = _target->isLifted();

	if (targetIsFlier || targetIsLifted)
	{
		if (!agent->canAttackAir()) return NO_APTITUDE;
	}

	if (!targetIsFlier && !targetIsLifted)
	{
		if (!agent->canAttackGround()) return NO_APTITUDE;
	}

	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	double dist = unit->getDistance(_target);
	dist = max(0.0, dist - type.seekRange());

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	if (type.isWorker()) aptitude *= WORKER_FACTOR;
	aptitude *= SPEED_FACTOR*type.topSpeed();

	if (type.isFlyer()) aptitude *= FLYING_FACTOR;
	else if (!unit->hasPath(_target)) return NO_APTITUDE;

	if (aptitude < _minAptitude) return NO_APTITUDE;

	return aptitude;
}

void AttackUnitTask::evaluateStatus()
{
	if (!_target->exists())
	{
		if (Broodwar->isVisible(TilePosition(_target->getPosition())))
		{
			_statusMessage = "Target doesn't exist.";
			_status = CANCELLED;
			return;
		}
	}

	if (_target->isInvincible())
	{
		_statusMessage = "Target is invincible.";
		_status = CANCELLED;
		return;
	}

	if (_target->isLoaded())
	{
		_statusMessage = "Target is loaded.";
		_status = CANCELLED;
		return;
	}

	if (!_target->isDetected())
	{
		_statusMessage = "Target is undectected.";
		_status = CANCELLED;
		return;
	}

	_canExecute = (_agents.size() >= 1);

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		_status = HALTED;
		return;
	}

	UnitType targetType = _target->getType();

	bool canBeAttacked = false;
	AgentList agentsToRemove;
	for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Agent* agent = *i;

		if (_target->isLifted() && !agent->canAttackAir())
		{
			agentsToRemove.push_back(agent);
			continue;
		}

		if (!_target->isLifted() && !agent->canAttackGround())
		{
			agentsToRemove.push_back(agent);
			continue;
		}

		canBeAttacked = true;
	}

	for(AgentList::iterator i = agentsToRemove.begin(); i != agentsToRemove.end(); i++)
	{
		_agents.erase(*i);
	}

	if (!canBeAttacked)
	{
		_statusMessage = "No unit can attack target.";
		_status = HALTED;
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void AttackUnitTask::evaluateNeededUnits()
{
	_needsMoreUnits = true;
}

void AttackUnitTask::calculatePriority()
{
	UnitType tType = _target->getType();
	double tValue = Environment::getUnitCombatValue(_target);
	//double tValue = Environment::getUnitHealthValue(_target)/Environment::getUnitTypeValue(tType);
	double wValue = Environment::getUnitAirAttackValue(tType) + Environment::getUnitGroundAttackValue(tType);

	double prePriority = TASK_PRIORITY_FACTOR*tValue*(1 + wValue);

	_priority = scalePriority(prePriority, MAX_ATTACK_PRIORITY, MIN_ATTACK_PRIORITY);
}

bool AttackUnitTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	Unit* bestTarget = NULL;
	double highestValue = 0;

	UnitSet airTargets = agent->getEnemiesInRangeAirAttack();
	UnitSet groundTargets = agent->getEnemiesInRangeGroundAttack();

	for (UnitSet::iterator i = airTargets.begin(); i != airTargets.end(); i++)
	{
		Unit* tUnit = (*i);

		if (tUnit == _target)
		{
			bestTarget = _target;
			break;
		}

		//UnitType tType = tUnit->getType();
		//double wValue = Environment::getUnitAirAttackValue(tType) + Environment::getUnitGroundAttackValue(tType);

		//if (wValue == 0) continue;

		if (!Environment::isUnitCombatValuable(tUnit)) continue;

		double value = Environment::getUnitCombatValue(tUnit);
		//double value = Environment::getUnitHealthValue(tUnit)/Environment::getUnitTypeValue(tType);

		if (highestValue < value)
		{
			bestTarget = tUnit;
			highestValue = value;
		}
	}

	if (bestTarget != _target)
	{
		for (UnitSet::iterator i = groundTargets.begin(); i != groundTargets.end(); i++)
		{
			Unit* tUnit = (*i);

			if (tUnit == _target)
			{
				bestTarget = _target;
				break;
			}

			//UnitType tType = tUnit->getType();
			//double wValue = Environment::getUnitAirAttackValue(tType) + Environment::getUnitGroundAttackValue(tType);

			//if (wValue == 0) continue;

			if (!Environment::isUnitCombatValuable(tUnit)) continue;

			double value = Environment::getUnitCombatValue(tUnit);
			//double value = Environment::getUnitHealthValue(tUnit)/Environment::getUnitTypeValue(tType);

			if (highestValue < value)
			{
				bestTarget = tUnit;
				highestValue = value;
			}
		}
	}

	if (bestTarget == NULL) bestTarget = _target;

	if ((unit->getOrderTarget() != bestTarget) && (unit->getTarget() != bestTarget))
		return unit->attack(bestTarget);

	return true;
}

