#include "AssaultSectorTask.h"
#include "Environment.h"
#include "Agent.h"
#include "MapSector.h"

#define TASK_PRIORITY_FACTOR					1
#define FLYING_FACTOR							2
#define SPEED_FACTOR							1
#define ATTACK_RANGE_FACTOR						1
#define ATTACK_DAMAGE_FACTOR					1
#define UNIT_VALUE_FACTOR						1
#define UNIT_COUNT_FACTOR						1
#define WORKER_FACTOR							0.1
#define MIN_ENEMY_VALUE							0
#define WEAPON_FACTOR							5
#define DIFFICULTY_FACTOR						40

#define VIEW_RANGE_FACTOR						1.5
#define BASE_RADIUS								10
#define MINIMUM_RANGE_FACTOR					12
#define MINIMUM_RANGE_INCREASE					0.25
#define MINIMUM_RANGE_BONUS_INCREASE			10
#define MINIMUM_RANGE_BONUS_DECREASE			20
#define MINIMUM_RANGE_COMBAT_FACTOR				1.5
#define PERCENTAGE_READY_INCREASE				0.10
#define PERCENTAGE_READY_DECREASE				0.05
#define ASSAULT_PREP_TASK_VALIDATION_FREQ		50
#define ASSAULT_READY_TASK_VALIDATION_FREQ		10
#define MAX_ASSAULT_COUNT						25
#define ASSAULT_TASK_FACTOR						1
#define MIN_UNITS_PER_ASSAULT					15
#define MAX_UNITS_PER_ASSAULT					25
#define MAX_ASSAULTS_PER_SECTOR					2
#define CHANGE_CENTER_AGENT_FREQ				200

#define ENEMY_STRENGTH_FACTOR					1.5

#define MARINE_MEDIC_RATIO						8

#define MAX_SECTOR_ASSAULT_RANGE				400			//in Pixels

void AssaultSectorTask::generateNecessaryTasks(Environment* env)
{
	int assaultSectorTaskCount = env->getFilteredTasks(ASSAULT_SECTOR_TASK).size();
	if (MAX_ASSAULT_COUNT <= assaultSectorTaskCount) return;

	SectorList mapSectors = prioritizeSectors(env);

	for(SectorList::iterator i = mapSectors.begin(); i != mapSectors.end(); i++)
	{
		MapSector* sector = (*i);

		//Broodwar->printf("Generating AssaultTask task.");
		env->addTask(new AssaultSectorTask(sector, env));
		assaultSectorTaskCount++;

		if (MAX_ASSAULT_COUNT <= assaultSectorTaskCount) break;
	}
}

bool compareAssaultSectors(MapSector* first, MapSector* second)
{
	int firstCount = first->getAssaultSectorTaskCount();
	int secondCount = second->getAssaultSectorTaskCount();

	if (firstCount < secondCount) return true;

	double firstVal = first->getEnemyValue();
	double secondVal = second->getEnemyValue();

	if (firstVal > secondVal) return true;

	return false;
}

SectorList AssaultSectorTask::prioritizeSectors(Environment* env)
{
	SectorList sectorList;

	MapSectorMap* mapSectors = env->getMapSectors();

	AgentMap agents = env->getFilteredAgents(ATTACKER_AGENT);

	for(MapSectorMap::iterator i = mapSectors->begin(); i != mapSectors->end(); i++)
	{
		MapSector* sector = (*i).second;

		TilePosition sectorCenter = sector->getCenterPosition();

		if (sectorCenter == TilePositions::None) continue;

		if (sector->getAssaultSectorTaskCount() >= MAX_ASSAULTS_PER_SECTOR) continue;

		if (sector->getEnemyValue() <= MIN_ENEMY_VALUE) continue;

		bool canAssault = false;
		for(AgentMap::iterator i = agents.begin(); i != agents.end(); i++)
		{
			Agent* agent = (*i).second;

			if (evaluateAptitudeForTask(sector, agent) > NO_APTITUDE)
			{
				canAssault = true;
				break;
			}
		}
		
		if (canAssault) sectorList.push_back(sector);
	}

	sectorList.sort(compareAssaultSectors);

	return sectorList;
}

AssaultSectorTask::AssaultSectorTask(MapSector* sector, Environment* env) : Task(env, "Assault ")
{
	_types.insert(ASSAULT_SECTOR_TASK);

	TilePosition tPos = sector->getPosition();

	char coordinates[20];

	sprintf_s(coordinates, 20, "%d,%d", tPos.x()/SECTOR_LENGTH, tPos.y()/SECTOR_LENGTH);
	
	_name.append(coordinates);
	
	_assaultCount = sector->getAssaultSectorTaskCount();
	sector->incAssaultSectorTaskCount();
	_sector = sector;

	_execValidationFreq = ASSAULT_PREP_TASK_VALIDATION_FREQ;

	_targetEnemy = NULL;

	_groupPosition = Positions::None;
	_readyRange = 0;
	_centerAgent = NULL;
}

AssaultSectorTask::~AssaultSectorTask(void)
{
	_sector->decAssaultSectorTaskCount();
}

void AssaultSectorTask::addAgent(Agent* agent)
{
	_agents.insert(agent);

	UnitType type = agent->getUnitType();

	_unitTypeCounts[type]++;

	agent->setDisposition(BRAVE);
}

void AssaultSectorTask::removeAgent(Agent* agent)
{
	agent->setDisposition(COWARD);
	Unit* unit = agent->getUnit();

	if ((unit->getHitPoints() < 1) ||
		(!unit->exists()))
	{
		_sector->incDeathCount();
	}

	if (agent == _centerAgent) _centerAgent = NULL;

	_unitTypeCounts[unit->getType()]--;

	_agents.erase(agent);
}

MapSector* AssaultSectorTask::getSector()
{
	return _sector;
}

int AssaultSectorTask::getReadyRange()
{
	return _readyRange;
}

Position AssaultSectorTask::getGroupPosition()
{
	return _groupPosition;
}

bool AssaultSectorTask::isAssaultReady()
{
	return _isAssaultReady;
}

double AssaultSectorTask::getAirStrength()
{
	return _airStrength;
}

double AssaultSectorTask::getGroundStrength()
{
	return _groundStrength;
}

double AssaultSectorTask::getMinimumAttackAirStrength()
{
	return _minimumAttackAirStrength;
}

double AssaultSectorTask::getMinimumAttackGroundStrength()
{
	return _minimumAttackGroundStrength;
}

double AssaultSectorTask::getEnemyAirStrength()
{
	return _enemyAirStrength;
}

double AssaultSectorTask::getEnemyGroundStrength()
{
	return _enemyGroundStrength;
}

double AssaultSectorTask::getMinimumDefenseAirStrength()
{
	return _minimumDefenseAirStrength;
}

double AssaultSectorTask::getMinimumDefenseGroundStrength()
{
	return _minimumDefenseGroundStrength;
}

Unit* AssaultSectorTask::getTargetEnemy()
{
	return _targetEnemy;
}

double AssaultSectorTask::evaluateAptitudeForTask(MapSector* sector, Agent* agent, Position groupPosition)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (type.isBuilding()) return NO_APTITUDE;
	if (type.isWorker() && sector->getEnvironment()->hasEconomy()) return NO_APTITUDE;

	TilePosition tPos = sector->getCenterPosition();
	Position pos = groupPosition;

	if (pos == Positions::None) pos = Position(tPos);

	if (pos == Positions::None) return NO_APTITUDE;

	double dist = unit->getDistance(pos);

	double aptitude = DIST_APTITUDE_FACTOR/(1.0 + dist);

	if (type.isFlyer())
	{
		aptitude *= FLYING_FACTOR;
	}
	else
	{
		if (!pos.hasPath(unit->getPosition())) return NO_APTITUDE;
	}

	double airPercentage = sector->getAirValuePercentage();

	aptitude *= ATTACK_RANGE_FACTOR*(agent->getAirWeaponRange()*airPercentage + agent->getGroundWeaponRange()*(1.0 - airPercentage));
	aptitude *= ATTACK_DAMAGE_FACTOR*(agent->getAirWeaponDamage()*airPercentage + agent->getGroundWeaponDamage()*(1.0 - airPercentage));
	aptitude *= UNIT_VALUE_FACTOR*Environment::getUnitHealthValue(unit)*Environment::getUnitTypeValue(type);
	aptitude *= SPEED_FACTOR*type.topSpeed();

	return aptitude;
}

double AssaultSectorTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (type == UnitTypes::Terran_Medic)
	{
		if (_unitTypeCounts[type]*MARINE_MEDIC_RATIO >= _unitTypeCounts[UnitTypes::Terran_Marine]) return NO_APTITUDE;
	}

	return evaluateAptitudeForTask(_sector, agent, _groupPosition);
}

void AssaultSectorTask::calculatePriority()
{
	double enemyValue = _sector->getEnemyValue();

	double difficultyFactor = DIFFICULTY_FACTOR/(double)(DIFFICULTY_FACTOR + _sector->getDeathCount());

	double prePriority = TASK_PRIORITY_FACTOR*enemyValue*difficultyFactor/(double)(1.0 + _assaultCount);

	int max_assault_priority = MAX_INACTIVE_ASSAULT_PRIORITY;
	int min_assault_priority = MIN_INACTIVE_ASSAULT_PRIORITY;

	if (_targetEnemy != NULL)
	{
		max_assault_priority = MAX_ACTIVE_ASSAULT_PRIORITY;
		min_assault_priority = MIN_ACTIVE_ASSAULT_PRIORITY;
	}

	_priority = scalePriority(prePriority, max_assault_priority, min_assault_priority);
}

void AssaultSectorTask::evaluateAssaultReadyness()
{
	_syncUp = false;
	_isAssaultReady = true;
	_execValidationFreq = ASSAULT_PREP_TASK_VALIDATION_FREQ;
	_readyRange = 0;
	_groupPosition = Positions::None;
	
	int agentCount = _agents.size();
	_unitsNeeded = agentCount;

	if (agentCount == 0)
	{
		_unitsNeeded = agentCount + 1;
		 _isAssaultReady = false;
		 return;
	}

	int sumX = 0;
	int sumY = 0;
	double rangeFactor = MINIMUM_RANGE_FACTOR + MINIMUM_RANGE_INCREASE*agentCount;
	
	double agentAntiAirValue = 0.0;
	double agentAntiGroundValue = 0.0;
	double agentAirHealth = 0.0;
	double agentGroundHealth = 0.0;
	int agentAirUnits = 0;

	int bestSightRange = 0;
	bool hasAttackers = false;
	for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Agent* agent = (*i);
		Unit* unit = agent->getUnit();
		UnitType type = unit->getType();
		Position pos = unit->getPosition();

		sumX += pos.x();
		sumY += pos.y();

		double antiAirValue = Environment::getUnitAirAttackValue(type);
		double antiGroundValue = Environment::getUnitGroundAttackValue(type);
		double healthValue = (unit->getHitPoints() + unit->getShields());

		if (type.canAttack()) hasAttackers = true;

		agentAntiAirValue += antiAirValue;
		agentAntiGroundValue += antiGroundValue;

		if (type.isFlyer() || unit->isLifted())
		{
			agentAirHealth += healthValue;
			agentAirUnits++;
		}
		else
			agentGroundHealth += healthValue;

		int sightRange = type.sightRange();

		if (sightRange > bestSightRange) bestSightRange = sightRange;
	}

	double agentAirProportion = agentAirUnits/(double)agentCount;

	Position centroid = Position(sumX/agentCount, sumY/agentCount);
	_groupPosition = centroid;

	Position bestPosition = Positions::None;

	if ((_centerAgent == NULL) ||
		((Broodwar->getFrameCount()%CHANGE_CENTER_AGENT_FREQ == 0) &&
		(_targetEnemy == NULL)))
	{
		double closestDistance = MAX_POS_DISTANCE;
		double slowestSpeed = MAX_POS_DISTANCE;
		int healthiest = MAX_POS_DISTANCE;
		for(AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
		{
			Agent* agent = *i;
			Unit* unit = agent->getUnit();
			Position pos = unit->getPosition();
			double distance = pos.getDistance(centroid);
			UnitType type = unit->getType();

			if (type.isFlyer()) continue;
			if (!type.canAttack()) continue;

			int health = unit->getHitPoints() + unit->getShields();
			double speed = type.topSpeed();

			if (health > healthiest) continue;
			if ((health == healthiest) && (slowestSpeed < speed)) continue;
			if ((health == healthiest) && (slowestSpeed == speed) && (closestDistance < distance)) continue;

			healthiest = health;
			slowestSpeed = speed;
			_centerAgent = agent;
			closestDistance = distance;
			bestPosition = pos;
		}
	}
	else
	{
		bestPosition = _centerAgent->getUnitPosition();
	}
		
	if (bestPosition != Positions::None) _groupPosition = bestPosition;

	UnitSet targetsInRange = Broodwar->getUnitsInRadius(_groupPosition, (int)(bestSightRange*VIEW_RANGE_FACTOR));
	
	_targetEnemy = NULL;
	bool targetIsCombatValuable = false;
	double highestTargetValue = 0.0;
	for (UnitSet::iterator i = targetsInRange.begin(); i != targetsInRange.end(); i++)
	{
		Unit* target = (*i);

		if (!_env->isEnemyUnit(target)) continue;

		if (!target->isDetected()) continue;

		UnitType tType = target->getType();

		double value = Environment::getUnitCombatValue(target);
		//double value = Environment::getUnitHealthValue(target)/Environment::getUnitTypeValue(tType);

		if (targetIsCombatValuable && !Environment::isUnitCombatValuable(target)) continue;

		if (!targetIsCombatValuable && Environment::isUnitCombatValuable(target))
		{
			_targetEnemy = target;
			highestTargetValue = value;
			targetIsCombatValuable = true;
			continue;
		}

		if (highestTargetValue < value)
		{
			_targetEnemy = target;
			highestTargetValue = value;
		}
	}

	UnitMemorySet quadrantEnemies;
	Quadrant* quadrant = _env->getQuadrant(_sector);
	
	_readyRange = (int)(BASE_RADIUS * rangeFactor);

	unsigned int maxUnits = MAX_UNITS_PER_ASSAULT;

	bool forceAssault = (
		(!_env->hasEconomy()) ||
		(_targetEnemy != NULL) ||
		(_unitsNeeded >= maxUnits));

	forceAssault = forceAssault && hasAttackers;

	if (!forceAssault)
	{
		if (MIN_UNITS_PER_ASSAULT > _unitsNeeded)
		{
			_isAssaultReady = false;
			bestPosition = _env->getBestOpenSpace(_groupPosition);
			
			if (bestPosition != Positions::None) _groupPosition = bestPosition;

			_unitsNeeded = agentCount + 1;
			return;
		}

		MapSectorSet quadrantSectors = _env->getMapSectorsInQuadrant(quadrant);

		for(MapSectorSet::iterator i = quadrantSectors.begin(); i != quadrantSectors.end(); i++)
		{
			UnitMemorySet* enemies = (*i)->getEnemies();
			quadrantEnemies.insert(enemies->begin(), enemies->end());
		}

		//UnitMemorySet allEnemies = _env->getEnemyMemories();

		double enemyAntiAirValue = 0.0;
		double enemyAntiGroundValue = 0.0;
		double enemyAirHealth = 0.0;
		double enemyGroundHealth = 0.0;
		int enemyAirUnits = 0;
		int enemyCount = quadrantEnemies.size();
		//int enemyCount = allEnemies.size();

		for(UnitMemorySet::iterator i = quadrantEnemies.begin(); i != quadrantEnemies.end(); i++)
		//for(UnitMemorySet::iterator i = allEnemies.begin(); i != allEnemies.end(); i++)
		{
			UnitMemory* memory = *i;
			UnitType type = memory->type;

			double antiAirValue = Environment::getUnitAirAttackValue(type);
			double antiGroundValue = Environment::getUnitGroundAttackValue(type);
			double healthValue = (memory->knownHitPoints + memory->knownShields);

			if (type == UnitTypes::Terran_Bunker)
			{
				antiAirValue += 4*Environment::getUnitAirAttackValue(UnitTypes::Terran_Marine);
				antiGroundValue += 4*Environment::getUnitGroundAttackValue(UnitTypes::Terran_Marine);
			}

			if (type.isWorker())
			{
				antiAirValue *= WORKER_FACTOR;
				antiGroundValue *= WORKER_FACTOR;
			}

			enemyAntiAirValue += antiAirValue;
			enemyAntiGroundValue += antiGroundValue;

			if ((antiAirValue + antiGroundValue) > 0)
			{
				if (type.isFlyer() || memory->wasLifted)
				{
					enemyAirHealth += healthValue;
					enemyAirUnits++;
				}
				else
					enemyGroundHealth += healthValue;
			}
		}

		double enemyAirProportion = 0.0;
		if (enemyCount > 0) enemyAirProportion = enemyAirUnits/(double)enemyCount;

		_airStrength = agentAntiAirValue*WEAPON_FACTOR*enemyAirProportion;
		_groundStrength = agentAntiGroundValue*WEAPON_FACTOR*(1.0 - enemyAirProportion);
		_minimumAttackAirStrength = enemyAirHealth;
		_minimumAttackGroundStrength = enemyGroundHealth;
		
		_enemyAirStrength = enemyAntiAirValue*WEAPON_FACTOR*agentAirProportion;
		_enemyGroundStrength = enemyAntiGroundValue*WEAPON_FACTOR*(1.0 - agentAirProportion);
		_minimumDefenseAirStrength = agentAirHealth;
		_minimumDefenseGroundStrength = agentGroundHealth;

		bool superiorAirAttack = _minimumAttackAirStrength*ENEMY_STRENGTH_FACTOR <= _airStrength;
		bool superiorGroundAttack = _minimumAttackGroundStrength*ENEMY_STRENGTH_FACTOR <= _groundStrength;
		bool superiorAirDefense = _minimumDefenseAirStrength >= _enemyAirStrength*ENEMY_STRENGTH_FACTOR;
		bool superiorGroundDefense = _minimumDefenseGroundStrength >= _enemyGroundStrength*ENEMY_STRENGTH_FACTOR;

		bool superiorAttack = superiorAirAttack && superiorGroundAttack;
		bool superiorDefense = superiorAirDefense && superiorGroundDefense;
	
		if (!superiorAttack || !superiorDefense)
		{
			_isAssaultReady = false;
			bestPosition = _env->getBestOpenSpace(_groupPosition);
			
			if (bestPosition != Positions::None) _groupPosition = bestPosition;

			_unitsNeeded = agentCount + 1;
			return;
		}

		AgentMap attackers = _env->getFilteredAgents(ATTACKER_AGENT);
		int attackerCount = attackers.size() - Agent::filterMap(&attackers, WORKER_AGENT).size();

		// Don't attack unless you have a minimum reserve of units already
		if (attackerCount < (MIN_UNITS_PER_ASSAULT + agentCount))
		{
			_isAssaultReady = false;
			bestPosition = _env->getBestOpenSpace(_groupPosition);
			
			if (bestPosition != Positions::None) _groupPosition = bestPosition;
			return;
		}
	}

	if (_targetEnemy != NULL) _readyRange = (int)(_readyRange*MINIMUM_RANGE_COMBAT_FACTOR);

	_unitsNeeded = 0;

	_execValidationFreq = ASSAULT_READY_TASK_VALIDATION_FREQ;

	int frameSpan = Broodwar->getFrameCount() - _creationFrame;
	bool sync = (frameSpan%_execValidationFreq) == 0;

	if (sync) _syncUp = true;
}

void AssaultSectorTask::evaluateStatus()
{
	evaluateAssaultReadyness();

	if (_sector->getEnemyValue() <= MIN_ENEMY_VALUE)
	{
		_statusMessage = "Completed.";
		_status = FINISHED;
		return;
	}
	
	_canExecute = true;

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		_status = HALTED;
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void AssaultSectorTask::evaluateNeededUnits()
{
	TaskSet tasks = _env->getFilteredTasksByPriority(ASSAULT_SECTOR_TASK, _priority);

	for (TaskSet::iterator i = tasks.begin(); i != tasks.end(); i++)
	{
		if (!((AssaultSectorTask*)(*i))->isAssaultReady())
		{
			_needsMoreUnits = false;
			return;
		}
	}

	_needsMoreUnits = _agents.size() < min((unsigned int)MAX_UNITS_PER_ASSAULT, _unitsNeeded);
}

bool AssaultSectorTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (type == UnitTypes::Terran_Marine) return marineExecute(agent);
	if (type == UnitTypes::Terran_Medic) return medicExecute(agent);
	if (type == UnitTypes::Terran_Wraith) return wraithExecute(agent);
	if (type == UnitTypes::Terran_Siege_Tank_Siege_Mode) return siegeExecute(agent);
	if (type == UnitTypes::Terran_Siege_Tank_Tank_Mode) return tankExecute(agent);

	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool AssaultSectorTask::marineExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	double distance = unit->getDistance(_groupPosition);

	double readyRange = _readyRange;
	
	if (distance <= readyRange)
	{
		if (_targetEnemy != NULL)
		{
			Unit* target = agent->getBestTargetInAttackRange();

			if (target == NULL) target = _targetEnemy;

			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
			
			return unit->attack(target);
		}

		if (!_isAssaultReady) return true;

		TilePosition cTPos = _sector->getCenterPosition();
		Position cPos = Position(cTPos);
		Position tPos = unit->getOrderTargetPosition();

		if (_sector->isInSector(tPos)) return true;

		return unit->move(cPos);
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool AssaultSectorTask::medicExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	double distance = unit->getDistance(_groupPosition);

	double readyRange = _readyRange*0.30;
	
	if (distance <= readyRange)
	{
		if (_targetEnemy != NULL)
		{
			Order order = unit->getOrder();

			if (order != Orders::HealMove) return unit->stop();
		}

		if (!_isAssaultReady) return true;

		TilePosition cTPos = _sector->getCenterPosition();
		Position cPos = Position(cTPos);
		Position tPos = unit->getOrderTargetPosition();

		if (_sector->isInSector(tPos)) return true;

		return unit->move(cPos);
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool AssaultSectorTask::wraithExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (_targetEnemy != NULL)
	{
		if (!unit->isCloaked() && unit->canIssueCommand(UnitCommand(unit, UnitCommandTypes::Cloak, NULL, 0, 0, 0)))
		{
			return unit->cloak();
		}

		Unit* target = _targetEnemy;

		if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
		
		return unit->attack(target);
	}

	if (unit->isCloaked())
	{
		return unit->decloak();
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool AssaultSectorTask::tankExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (unit->isSieged()) return true;

	if (!unit->canIssueCommand(UnitCommand(unit, UnitCommandTypes::Siege, NULL, 0, 0, 0)))
	{
		return marineExecute(agent);
	}

	double distance = unit->getDistance(_groupPosition);

	double readyRange = _readyRange;
	
	if (distance <= readyRange)
	{
		if (_targetEnemy != NULL)
		{
			Unit* target = agent->getBestTargetInAttackRange(true, true);

			if (target != NULL)
			{
				return unit->siege();
			}

			target = agent->getBestTargetInAttackRange(true, false);

			if (target != NULL)
			{
				if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
				
				return unit->attack(target);
			}
			
			target = agent->getBestTargetInAttackRange(false, true);

			if (target != NULL)
			{
				return unit->siege();
			}

			target = agent->getBestTargetInAttackRange(false, false);

			if (target != NULL)
			{
				if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
				
				return unit->attack(target);
			}

			target = _targetEnemy;

			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;
			
			return unit->attack(target);
		}

		if (!_isAssaultReady) return unit->siege();

		TilePosition cTPos = _sector->getCenterPosition();
		Position cPos = Position(cTPos);
		Position tPos = unit->getOrderTargetPosition();

		if (_sector->isInSector(tPos)) return true;

		return unit->move(cPos);
	}
	
	//If not in range, move to range
	return unit->move(_groupPosition);
}

bool AssaultSectorTask::siegeExecute(Agent* agent)
{
	Unit* unit = agent->getUnit();
	UnitType type = unit->getType();

	if (!unit->isSieged()) return true;

	double distance = unit->getDistance(_groupPosition);

	double readyRange = _readyRange*2;
	
	if (distance <= readyRange)
	{
		Unit* target = agent->getBestTargetInAttackRange(true, true);

		if (target != NULL)
		{
			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;

			return unit->attack(target);
		}

		target = agent->getBestTargetInAttackRange(true, false);

		if (target != NULL) return unit->unsiege();

		target = agent->getBestTargetInAttackRange(false, true);

		if (target != NULL)
		{
			if ((unit->getOrderTarget() == target) || (unit->getTarget() == target)) return true;

			return unit->attack(target);
		}

		target = agent->getBestTargetInAttackRange(false, false);

		if (target != NULL) return unit->unsiege();

		if (!_isAssaultReady) return true;
	}

	return unit->unsiege();
}

