#include "AddonBuildTask.h"
#include "Environment.h"
#include "BaseAgent.h"
#include "Agent.h"

#define BASE_APTITUDE	1.0
#define RESOURCE_BONUS	8

AddonBuildTask::AddonBuildTask(UnitType buildingType, Environment* env) : Task(env, "Build Addon ")
{
	_types.insert(ADDON_BUILD_TASK);

	_buildingType = buildingType;
	_sourceBuildingType = _buildingType.whatBuilds().first;

	_building = NULL;

	initialize();
}

AddonBuildTask::~AddonBuildTask(void)
{
	_env->decRequiredMinerals(_buildingType.mineralPrice());
	_env->decRequiredGas(_buildingType.gasPrice());
}

void AddonBuildTask::initialize()
{
	_name.append(_buildingType.getName());

	_env->incRequiredMinerals(_buildingType.mineralPrice());
	_env->incRequiredGas(_buildingType.gasPrice());

	_requiresUnits = false;
}

void AddonBuildTask::addAgent(Agent* agent)
{
	_agents.insert(agent);

	_env->addTaskTarget(agent->getUnit(), ADDON_BUILD_TASK);
}

void AddonBuildTask::removeAgent(Agent* agent)
{
	_env->removeTaskTarget(agent->getUnit(), ADDON_BUILD_TASK);

	_agents.erase(agent);
}

double AddonBuildTask::evaluateAptitude(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (unit->getType() != _sourceBuildingType) return NO_APTITUDE;

	if (unit->isLifted()) return NO_APTITUDE;

	if (!unit->isIdle()) return NO_APTITUDE;

	if (unit->getAddon() != NULL) return NO_APTITUDE;

	if (_env->isTaskTarget(unit, ADDON_BUILD_TASK)) return NO_APTITUDE;

	double aptitude = BASE_APTITUDE;

	return aptitude;
}

void AddonBuildTask::evaluateStatus()
{
	_canExecute = true;
	_statusMessage = "None specified.";
	_requiresUnits = false;

	bool hasUnits = _agents.size() >= 1;
	
	Player* player = _env->getPlayer();

	if (hasUnits)
	{
		Unit* sourceBuilding = (*_agents.begin())->getUnit();

		if (_building == NULL) _building = sourceBuilding->getAddon();
	}

	if (_building == NULL)
	{
		UnitTypeCountMap requiredUnits = _buildingType.requiredUnits();

		for (UnitTypeCountMap::iterator i = requiredUnits.begin(); i != requiredUnits.end(); i++)
		{
			if (_env->getFilteredAgents((*i).first).size() == 0)
			{
				_canExecute = false;
				break;
			}
		}

		if (!_canExecute)
		{
			_statusMessage = "Required building hasn't been created.";
			return;
		}

		int currentMinerals = player->minerals();
		int reservedMinerals = _env->getReservedMinerals();
		int mineralCost = _buildingType.mineralPrice();
		
		int currentGas = player->gas();
		int reservedGas = _env->getReservedGas();
		int gasCost = _buildingType.gasPrice();
		
		int excessMinerals = max(0, currentMinerals - reservedMinerals);
		int excessGas = max(0, currentGas - reservedGas);

		_canExecute = ((excessMinerals >= mineralCost) && (excessGas >= gasCost));

		double percentCollected = 1.0;

		if ((mineralCost > 0) && (gasCost > 0))
		{
			percentCollected = min(excessMinerals/(double)mineralCost, excessGas/(double)gasCost);
			percentCollected = min(percentCollected, 1.0);
		}

		_env->reserveMinerals((int)(mineralCost*percentCollected + RESOURCE_BONUS));
		_env->reserveGas((int)(gasCost*percentCollected + RESOURCE_BONUS));

		if (!_canExecute)
		{
			_statusMessage = "Not enough resources.";
			return;
		}

		_requiresUnits = true;
	}
	else
	{
		_requiresUnits = true;

		if (_building->getRemainingBuildTime() <= 0)
		{
			_statusMessage = "Finished.";
			_status = FINISHED;
			return;
		}

		if (_building->isBeingConstructed())
			_canBePreempted = false;
		else
			_canBePreempted = true;
	}

	_canExecute = hasUnits;

	if (!_canExecute)
	{
		_statusMessage = "No unit assigned.";
		return;
	}

	_statusMessage = "No issues.";
	_status = IN_PROGRESS;
}

void AddonBuildTask::evaluateNeededUnits()
{
	if (_agents.size() < 1)	_needsMoreUnits = _requiresUnits;
	else					_needsMoreUnits = false;
}

UnitType AddonBuildTask::getBuildingType()
{
	return _buildingType;
}

Unit* AddonBuildTask::getBuilding()
{
	return _building;
}

bool AddonBuildTask::execute(Agent* agent)
{
	Unit* unit = agent->getUnit();

	if (unit->isConstructing()) return true;

	if (_building != NULL) return true;

	bool success = unit->buildAddon(_buildingType);

	if (!success) _status = CANCELLED;
	return success;
}
