#include "Agent.h"
#include "MapSector.h"
#include "Environment.h"

#define POSSIBLE_ENEMY_VALUE_INC	50

MapSector::MapSector(TilePosition position, Environment* env)
{
	_position = position;
	_posX = position.x();
	_posY = position.y();

	_env = env;
}

MapSector::MapSector(int x, int y, Environment* env)
{
	_position = TilePosition(x, y);
	_posX = x;
	_posY = y;

	_env = env;
}

MapSector::~MapSector()
{
}

void MapSector::initialize()
{
	_exploreSectorTask = NULL;
	_assaultSectorTaskCount = 0;
	
	_resourceValue = 0.0;
	_enemyValue = 0.0;
	_possibleEnemyValue = 0.0;

	_lastVisible = -1;

	_deathCount = 0;

	int mapWidth = Broodwar->mapWidth();
	int mapHeight = Broodwar->mapHeight();

	_sectorWidth = min(mapWidth, _posX + SECTOR_LENGTH) - _posX;
	_sectorHeight = min(mapHeight, _posY + SECTOR_LENGTH) - _posY;

	int centerX = _posX + _sectorWidth/2;
	int centerY = _posY + _sectorHeight/2;

	_centerPosition = TilePosition(centerX, centerY);
	TilePosition closestPosition = TilePositions::None;
	double closestDistance = MAX_POS_DISTANCE;

	for (int j = 0; j < _sectorHeight; j++)
	{
		for (int i = 0; i < _sectorWidth; i++)
		{
			int x = i + _posX;
			int y = j + _posY;

			TilePosition tPos = TilePosition(x, y);
			unsigned long index = ((unsigned long)x<<16) + (unsigned long)y;

			_positions[index] = tPos;

			if (!Broodwar->isWalkable(x*4, y*4)) continue;

			//if (!Broodwar->isBuildable(x, y)) continue;

			//if (Broodwar->getUnitsOnTile(x, y).size() > 0) continue;

			_walkablePositions[index] = tPos;

			double distance = tPos.getDistance(_centerPosition);

			if (distance >= closestDistance) continue;

			closestPosition = tPos;
			closestDistance = distance;
		}
	}

	_centerPosition = closestPosition;

	if (_centerPosition != TilePositions::None)
	{
		_walkablePercentage = (double)_walkablePositions.size()/(double)_positions.size();
	}

	refresh();
}

void MapSector::refreshResources()
{
	_minerals.clear();
	_geysers.clear();
	
	for (UnitMemorySet::const_iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		UnitMemory* memory = (*i);
		UnitType type = memory->type;

		if (type.isMineralField())
			_minerals.insert(memory);

		if (type == UnitTypes::Resource_Vespene_Geyser)
			_geysers.insert(memory);
	}
}

void MapSector::refreshEnemies()
{
	_enemies.clear();
	hasUndetecteds = false;

	UnitMemorySet memoriesToRemove;
	
	for (UnitMemorySet::const_iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		UnitMemory* memory = (*i);
		Player* knownPlayer = memory->knownPlayer;
		Position pos = memory->knownPosition;

		if (Broodwar->isVisible(TilePosition(pos)) && !memory->unit->isVisible())
		{
			memoriesToRemove.insert(memory);

			continue;
		}

		if (!knownPlayer->isEnemy(_env->getPlayer())) continue;

		if (!memory->wasDetected)
			hasUndetecteds = true;

		_enemies.insert(memory);
	}
	
	for (UnitMemorySet::const_iterator i = memoriesToRemove.begin(); i != memoriesToRemove.end(); i++)
	{
		removeUnitMemory(*i);
	}
}

void MapSector::refreshBuildingCount()
{
	_buildingCount = 0;
	
	for (AgentSet::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Agent* agent = (*i);
		UnitType type = agent->getUnitType();

		if (!type.isBuilding()) continue;

		_buildingCount++;
	}

	_buildingCount += _minerals.size();
	_buildingCount += _geysers.size();
}

Environment* MapSector::getEnvironment()
{
	return _env;
}

int MapSector::getBuildingCount()
{
	return _buildingCount;
}

UnitMemorySet* MapSector::getMinerals()
{
	return &_minerals;
}

void MapSector::refresh()
{
	refreshEnemies();
	refreshResources();

	calculateEnemyValue();
	calculateResourceValue();

	refreshBuildingCount();

	if (_centerPosition == TilePositions::None) return;

	int currentFrame = Broodwar->getFrameCount();

	if ((_lastVisible + SECTOR_FRAMES_PER_REFRESH) >= currentFrame) return;

	if (!Broodwar->isVisible(_centerPosition)) return;
	
	_possibleEnemyValue = 0;

	_lastVisible = currentFrame;

	_notVisibleWalkablePositions.clear();

	int visibleCount = 0;
	int walkablePosCount = _walkablePositions.size();

	if (walkablePosCount <= 0)
	{
		_visibilityPercentage = 1.0;
		return;
	}
	
	for (TilePositionMap::const_iterator i = _walkablePositions.begin(); i != _walkablePositions.end(); i++)
	{
		unsigned long index = (*i).first;
		TilePosition tPos = (*i).second;

		if (!Broodwar->isVisible(tPos))
		{
			_notVisibleWalkablePositions[index] = tPos;
		}
		else
		{
			visibleCount++;
		}
	}

	_visibilityPercentage = (double)visibleCount/(double)walkablePosCount;
}

void MapSector::calculateResourceValue()
{
	double value = 0.0;

	for(UnitMemorySet::iterator i = _minerals.begin(); i != _minerals.end(); i++)
	{
		UnitMemory* memory = (*i);

		if (_env->isTaskTarget(memory->unit, RECOLLECT_MIN_TASK)) continue;

		if (memory->knownResources <= 0) continue;

		value += MINERAL_SECTOR_VALUE;
	}

	value += _geysers.size()*GEYSER_SECTOR_VALUE;

	_resourceValue = value;
}

void MapSector::incPossibleEnemyValue()
{
	_possibleEnemyValue += POSSIBLE_ENEMY_VALUE_INC;
}

void MapSector::resetPossibleEnemyValue()
{
	_possibleEnemyValue = 0;
}

double MapSector::getResourceValue()
{
	return _resourceValue;
}

void MapSector::calculateEnemyValue()
{
	double value = 0.0;
	double airValue = 0.0;
	double undValue = 0.0;

	for (UnitMemorySet::const_iterator i = _enemies.begin(); i != _enemies.end(); i++)
	{
		UnitMemory* enemy = *i;
		UnitType eType = enemy->type;

		//if (!eType.isBuilding()) continue;

		Position knownPosition = enemy->knownPosition;
		
		if(Broodwar->isVisible(TilePosition(knownPosition)))
		{
			Unit* unit = enemy->unit;

			//Ignore units which should be visible but aren't
			if (!unit->isVisible()) continue;
		}

		double eValue = Environment::getUnitHealthValue(enemy)*Environment::getUnitTypeValue(eType);

		if (!enemy->wasDetected)
		{
			undValue += eValue;
		}

		UnitType type = enemy->type;

		if (type.isFlyer() || enemy->wasLifted)
		{
			airValue += eValue;
		}

		value += eValue;
	}

	_airValuePercentage = 0.0;
	_undetectedValuePercentage = 0.0;

	if (value > 0)
	{
		_airValuePercentage = airValue/value;
		_undetectedValuePercentage = undValue/value;
	}

	_enemyValue = value + _possibleEnemyValue;
}

double MapSector::getEnemyValue()
{
	return _enemyValue;
}

void MapSector::addUnitMemory(UnitMemory* memory)
{
	MapSector* knownSector = memory->knownSector;

	if (knownSector == this) return;

	if (knownSector != NULL)
		knownSector->removeUnitMemory(memory);

	_unitMemories.insert(memory);

	memory->knownSector = this;
}

void MapSector::removeUnitMemory(UnitMemory* memory)
{
	_unitMemories.erase(memory);
	_enemies.erase(memory);

	memory->knownSector = NULL;
}

UnitMemorySet* MapSector::getEnemies()
{
	return &_enemies;
}

void MapSector::addAgent(Agent* agent)
{
	_agents.insert(agent);
}

void MapSector::removeAgent(Agent* agent)
{
	_agents.erase(agent);
}

int MapSector::getLastVisible()
{
	return _lastVisible;
}

double MapSector::getAirValuePercentage()
{
	return _airValuePercentage;
}

double MapSector::getUndetectedValuePercentage()
{
	return _undetectedValuePercentage;
}

void MapSector::markExplored()
{
	_lastVisible = Broodwar->getFrameCount();
}

void MapSector::incDeathCount()
{
	_deathCount++;
}

int MapSector::getDeathCount()
{
	return _deathCount;
}

int MapSector::getTimeNotVisible()
{
	int currentFrame = Broodwar->getFrameCount();

	return currentFrame - _lastVisible;
}

TilePositionMap* MapSector::getPositions()
{
	return &_positions;
}

TilePositionMap* MapSector::getWalkablePositions()
{
	return &_walkablePositions;
}

TilePositionMap* MapSector::getNotVisibleWalkablePositions()
{
	return &_notVisibleWalkablePositions;
}

double MapSector::getWalkablePercentage()
{
	return _walkablePercentage;
}

double MapSector::getVisibilityPercentage()
{
	return _visibilityPercentage;
}

TilePosition MapSector::getCenterPosition()
{
	return _centerPosition;
}

TilePosition MapSector::getPosition()
{
	return _position;
}

bool MapSector::isWalkableFromStart()
{
	TilePosition startPosition = Broodwar->self()->getStartLocation();

	return startPosition.hasPath(_centerPosition);
}

double MapSector::getDistanceFromStart()
{
	TilePosition startPosition = Broodwar->self()->getStartLocation();

	return startPosition.getDistance(_centerPosition);
}

double MapSector::getCornerDistance()
{
	int mapWidth = Broodwar->mapWidth();
	int mapHeight = Broodwar->mapHeight();

	double cornerDist = _centerPosition.getDistance(TilePosition(0,0));
	cornerDist = min(cornerDist, _centerPosition.getDistance(TilePosition(mapWidth,0)));
	cornerDist = min(cornerDist, _centerPosition.getDistance(TilePosition(0,mapHeight)));
	cornerDist = min(cornerDist, _centerPosition.getDistance(TilePosition(mapWidth,mapHeight)));

	return cornerDist;
}

double MapSector::getDistance(Position pos)
{
	return pos.getDistance(Position(_centerPosition));
}

double MapSector::getDistance(TilePosition pos)
{
	return pos.getDistance(_centerPosition);
}

void MapSector::setExploreSectorTask(ExploreSectorTask* task)
{
	_exploreSectorTask = task;
}

ExploreSectorTask* MapSector::getExploreSectorTask()
{
	return _exploreSectorTask;
}

void MapSector::incAssaultSectorTaskCount()
{
	_assaultSectorTaskCount++;
}

void MapSector::decAssaultSectorTaskCount()
{
	_assaultSectorTaskCount--;
}

int MapSector::getAssaultSectorTaskCount()
{
	return _assaultSectorTaskCount;
}

bool MapSector::isInSector(Unit* unit)
{
	TilePosition tPos = TilePosition(unit->getPosition());

	if ((tPos.x() >= _posX) && (tPos.y() >= _posY) &&
		(tPos.x() < (_posX + _sectorWidth)) && (tPos.y() < (_posY + _sectorHeight)))
		return true;

	return false;
}

bool MapSector::isInSector(Position pos)
{
	TilePosition tPos = TilePosition(pos);

	if ((tPos.x() >= _posX) && (tPos.y() >= _posY) &&
		(tPos.x() < (_posX + _sectorWidth)) && (tPos.y() < (_posY + _sectorHeight)))
		return true;

	return false;
}

bool MapSector::isInSector(TilePosition pos)
{
	if ((pos.x() >= _posX) && (pos.y() >= _posY) &&
		(pos.x() < (_posX + _sectorWidth)) && (pos.y() < (_posY + _sectorHeight)))
		return true;

	return false;
}

