#pragma once

#include "defines.h"

class Environment
{
	Player* _player;

	TilePositionMap _validPositions;
	MapSectorMap _mapSectors;
	QuadrantMap _quadrants;
	int _quadrantWidth;
	int _quadrantHeight;

	TaskSet _tasks;
	AgentMap _agents;
	UnitMemoryMap _unitMemories;

	int _reservedMinerals;
	int _reservedGas;
	int _reservedSupply;

	int _requiredMinerals;
	int _requiredGas;
	int _requiredSupply;

	int _requiredWorkers;
	int _requiredMineralWorkers;
	int _requiredGasWorkers;

	UnitTypeCountMap _unitTypeDeaths;
	int _totalDeaths;

	PlannedBuildingSet _plannedBuildings;

	TaskTargetMap _taskTargets;

	int _nextTaskId;

	bool _hasEconomy;
	bool _hasAttackers;

public:
	Environment();
	virtual ~Environment(void);

	int getNextTaskId();

	Player* getPlayer();

	MapSectorMap* getMapSectors();
	MapSectorSet getMapSectorsInQuadrant(Quadrant* quadrant);

	QuadrantMap* getQuadrants();
	Quadrant* getQuadrant(int i, int j);
	Quadrant* getQuadrant(TilePosition pos);
	Quadrant* getQuadrant(MapSector* sector);
	void resetQuadrants();

	void addTask(Task *task);
	void removeTask(Task *task);
	TaskSet* getTasks();

	void incUnitTypeDeaths(UnitType type);
	int getUnitTypeDeaths(UnitType type);
	int getTotalDeaths();

	bool containsAgent(Unit *unit);
	void addAgent(Unit *unit);
	void removeAgent(Unit *unit);
	
	void addUnitMemory(Unit *unit);
	void removeUnitMemory(Unit *unit);
	UnitMemory* getUnitMemory(Unit *unit);

	UnitMemorySet getEnemyMemories();

	AgentMap* getAgents();

	void initialize();
	void refresh();

	void refreshMemoryUnits();

	UnitMemorySet getMemoriesInRadius(Position pos, int radius);

	TilePositionMap getValidPositions();
	void generateValidPositions();
	TilePosition getBestPosition(UnitType type, Position idealPos);
	TilePosition getBestPosition(TileBox box, Position idealPos, UnitType type = UnitTypes::None);
	TilePosition getBestCommandPosition();
	
	Position getBestOpenSpace(Position idealPos);

	//bool areTilesConnected(TilePosition source, TilePosition destination);

	void reserveMinerals(int quantity);
	void reserveGas(int quantity);
	void reserveSupply(int quantity);

	int getReservedMinerals();
	int getReservedGas();
	int getReservedSupply();

	void resetReserves();

	void incRequiredMinerals(int quantity);
	void incRequiredGas(int quantity);
	void incRequiredSupply(int quantity);

	void decRequiredMinerals(int quantity);
	void decRequiredGas(int quantity);
	void decRequiredSupply(int quantity);

	int getRequiredMinerals();
	int getRequiredGas();
	int getRequiredSupply();

	int getCurrentMinerals();
	int getCurrentGas();
	int getCurrentSupply();
	int getTotalSupply();

	void purgeTasks();
	void purgeAgents();
	void purgeMapSectors();
	void purgeQuadrants();
	void purgeUnitMemories();

	bool isEnemyUnit(Unit* unit);
	
	void addTaskTarget(Unit* unit, TaskType type);
	void removeTaskTargets(Unit* unit);
	void removeTaskTarget(Unit* unit, TaskType type);
	bool isTaskTarget(Unit* unit, TaskType type);

	TaskSet getFilteredTasks(TaskType type);
	TaskSet getFilteredTasks(TaskType type, int maxCreationFrame);
	TaskSet getFilteredTasksByPriority(TaskType type, double priority);

	AgentMap getFilteredAgents(AgentType type, bool finished = true);
	AgentMap getFilteredAgents(UnitType type, bool finished = true);
	AgentMap getFilteredAgents(TaskType type, bool finished = true);

	Agent* getClosestAgent(AgentType type, Position pos);
	Agent* getClosestAgent(UnitType type, Position pos);
	Agent* getClosestAgent(TaskType type, Position pos);

	AgentSet getAgentsInRadius(Position pos, int radius);

	void addPlannedBuilding(BuildingSpaceInfo* pBuilding);
	void removePlannedBuilding(BuildingSpaceInfo* pBuilding);
	PlannedBuildingSet* getPlannedBuildings();

	void incRequiredWorkers(int count);
	void decRequiredWorkers(int count);
	int getRequiredWorkers();
	int calculateNeededExtraWorkers();

	void incRequiredMineralWorkers(int count);
	void decRequiredMineralWorkers(int count);
	int getRequiredMineralWorkers();

	void incRequiredGasWorkers(int count);
	void decRequiredGasWorkers(int count);
	int getRequiredGasWorkers();

	bool hasEconomy();
	bool hasAttackers();

	UnitSet getThreatenedUnits(Unit* enemyUnit);
	
	static bool isUnitCombatValuable(Unit* unit);
	static double getUnitCombatValue(Unit* unit);
	static double getUnitAirAttackValue(UnitType type);
	static double getUnitGroundAttackValue(UnitType type);
	static double getUnitTypeValue(UnitType type);
	static double getUnitHealthValue(Unit* unit);
	static double getUnitHealthValue(UnitMemory* memory);
};

struct Quadrant
{
	Quadrant(TilePosition c) : visited(false), dangerous(false), beingExplored(false), center(c) {}

	bool visited;
	bool dangerous;
	bool beingExplored;
	TilePosition center;
};

struct BuildingSpaceInfo
{
	BuildingSpaceInfo(TilePosition p, UnitType t) : tilePosition(p), type(t) {}

	TilePosition tilePosition;
	UnitType type;
};

struct TileBox
{
	TileBox(UnitType type);
	TileBox(int width, int height);

	int tileWidth;
	int tileHeight;
};

struct UnitMemory
{
	UnitMemory(Unit* u);

	bool refresh();

	Unit* unit;
	UnitType type;
	int knownHitPoints;
	int knownShields;
	int knownResources;
	bool wasLifted;
	bool wasDetected;
	Position knownPosition;
	MapSector* knownSector;
	Player* knownPlayer;
	bool wasEnemy;
};

