#include "Environment.h"
#include "MapSector.h"
#include "Task.h"
#include "BuildRefineryTask.h"
#include "RecollectMinTask.h"
#include "RecollectGasTask.h"
#include "Agent.h"
#include "BaseAgent.h"
#include "WorkerAgent.h"

#define ATTACK_RANGE_FACTOR		0.0
#define QUADRANTS_PER_AXIS		3

Environment::Environment()
{
	_player = Broodwar->self();

	_reservedMinerals = 0;
	_reservedGas = 0;
	_reservedSupply = 0;
	
	_requiredMinerals = 0;
	_requiredGas = 0;
	_requiredSupply = 0;
	
	_requiredWorkers = 0;
	_requiredGasWorkers = 0;
	_requiredMineralWorkers = 0;

	_nextTaskId = 0;
}

Environment::~Environment(void)
{
	purgeTasks();
	purgeAgents();
	purgeMapSectors();
	purgeQuadrants();
	purgeUnitMemories();
}

int Environment::getNextTaskId()
{
	int currentId = _nextTaskId;
	_nextTaskId++;

	return currentId;
}

Player* Environment::getPlayer()
{
	return _player;
}

MapSectorMap* Environment::getMapSectors()
{
	return &_mapSectors;
}

MapSectorSet Environment::getMapSectorsInQuadrant(Quadrant* quadrant)
{
	MapSectorSet mapSectors;

	for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
	{
		MapSector* sector = (*i).second;

		if (getQuadrant(sector) != quadrant) continue;

		mapSectors.insert(sector);
	}

	return mapSectors;
}

QuadrantMap* Environment::getQuadrants()
{
	return &_quadrants;
}

Quadrant* Environment::getQuadrant(int i, int j)
{
	int ci = i / _quadrantWidth;
	int cj = j / _quadrantHeight;

	unsigned long cIndex = ((unsigned long)ci<<16) + (unsigned long)cj;

	if (_quadrants.find(cIndex) == _quadrants.end()) return NULL;
	
	return _quadrants[cIndex];
}

Quadrant* Environment::getQuadrant(TilePosition pos)
{
	return getQuadrant(pos.x(), pos.y());
}

Quadrant* Environment::getQuadrant(MapSector* sector)
{
	return getQuadrant(sector->getPosition());
}

void Environment::resetQuadrants()
{
	for(QuadrantMap::iterator i = _quadrants.begin(); i != _quadrants.end(); i++)
	{
		Quadrant* quadrant = (*i).second;
		quadrant->visited = false;
		//quadrant->dangerous = false;
	}
}

void Environment::addTask(Task *task)
{
	_tasks.insert(task);
}

void Environment::removeTask(Task *task)
{
	_tasks.erase(task);

	task->cleanup();
	delete task;
}

TaskSet* Environment::getTasks()
{
	return &_tasks;
}

void Environment::purgeTasks()
{
	for(TaskSet::const_iterator i = _tasks.begin(); i != _tasks.end(); i++)
	{
		(*i)->cleanup();
		delete *i;
	}

	_tasks.clear();
}

void Environment::purgeMapSectors()
{
	for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
	{
		delete (*i).second;
	}

	_mapSectors.clear();
}

void Environment::purgeQuadrants()
{
	for(QuadrantMap::iterator i = _quadrants.begin(); i != _quadrants.end(); i++)
	{
		delete (*i).second;
	}

	_quadrants.clear();
}

void Environment::purgeUnitMemories()
{
	for(UnitMemoryMap::iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		delete (*i).second;
	}

	_unitMemories.clear();
}

bool Environment::containsAgent(Unit *unit)
{
	if (_agents.find(unit) != _agents.end()) return true;

	return false;
}

void Environment::addAgent(Unit *unit)
{
	if (_agents.find(unit) != _agents.end()) return;

	Agent* agent = Agent::createAgent(unit, this);

	_agents[unit] = agent;

	agent->initialize();
}

void Environment::removeAgent(Unit *unit)
{
	//This shouldn't happen
	if (_agents.find(unit) == _agents.end()) return;

	Agent* agent = _agents[unit];

	_agents.erase(unit);

	agent->cleanup();
	delete agent;
}

void Environment::addUnitMemory(Unit *unit)
{
	if (_unitMemories.find(unit) != _unitMemories.end()) return;

	UnitMemory* uMemory = new UnitMemory(unit);

	_unitMemories[unit] = uMemory;
}

void Environment::removeUnitMemory(Unit *unit)
{
	//This shouldn't happen
	if (_unitMemories.find(unit) == _unitMemories.end()) return;

	UnitMemory* uMemory = _unitMemories[unit];
	MapSector* knownSector = uMemory->knownSector;

	if (knownSector != NULL) knownSector->removeUnitMemory(uMemory);

	_unitMemories.erase(unit);

	delete uMemory;
}

UnitMemory* Environment::getUnitMemory(Unit *unit)
{
	if (_unitMemories.find(unit) == _unitMemories.end()) return NULL;

	return _unitMemories[unit];
}

UnitMemorySet Environment::getEnemyMemories()
{
	UnitMemorySet enemies;

	for(UnitMemoryMap::iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		UnitMemory* memory = (*i).second;

		if (memory->wasEnemy) enemies.insert(memory);
	}

	return enemies;
}

AgentMap* Environment::getAgents()
{
	return &_agents;
}

void Environment::purgeAgents()
{
	for(AgentMap::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		delete (*i).second;
	}

	_agents.clear();
}

void Environment::initialize()
{
	_hasEconomy = true;
	_hasAttackers = true;

	UnitSet allUnits = Broodwar->getAllUnits();

	//Obtain all units belonging to the player
	for(UnitSet::const_iterator i = allUnits.begin(); i != allUnits.end(); i++)
	{
		Unit* unit = *i;

		if (unit->getPlayer() == _player) addAgent(unit);
		
		addUnitMemory(unit);
	}

	int mapWidth = Broodwar->mapWidth();
	int mapHeight = Broodwar->mapHeight();

	_quadrantWidth = mapWidth/QUADRANTS_PER_AXIS + 1;
	_quadrantHeight = mapHeight/QUADRANTS_PER_AXIS + 1;

	for (int j = 0; j < mapHeight; j += SECTOR_LENGTH)
	{
		for (int i = 0; i < mapWidth; i += SECTOR_LENGTH)
		{
			unsigned long index = ((unsigned long)i<<16) + (unsigned long)j;

			MapSector* sector = new MapSector(i, j, this);
			_mapSectors[index] = sector;
			sector->initialize();

			int ci = i / _quadrantWidth;
			int cj = j / _quadrantHeight;

			unsigned long cIndex = ((unsigned long)ci<<16) + (unsigned long)cj;

			if (_quadrants.find(cIndex) == _quadrants.end())
			{
				TilePosition center(i + _quadrantWidth/2, j + _quadrantHeight/2);
				_quadrants[cIndex] = new Quadrant(center);
			}
		}
	}

	generateValidPositions();
}

void Environment::refreshMemoryUnits()
{
	UnitSet allUnits = Broodwar->getAllUnits();

	for(UnitSet::iterator i = allUnits.begin(); i != allUnits.end(); i++)
	{
		addUnitMemory(*i);
	}
}

void Environment::refresh()
{
	_hasEconomy = true;
	_hasAttackers = true;

	if (Agent::filterMap(&_agents, BASE_AGENT).size() <= 0)
	{
		if ((Agent::filterMap(&_agents, WORKER_AGENT).size() <= 0) ||
			(_player->minerals() < 400))
		{
			if (_player->minerals() < 50)
				_hasEconomy = false;
			else if ((Agent::filterMap(&_agents, UnitTypes::Terran_Barracks).size() <= 0) &&
				(Agent::filterMap(&_agents, UnitTypes::Terran_Factory).size() <= 0) &&
				(Agent::filterMap(&_agents, UnitTypes::Terran_Starport).size() <= 0))
				_hasEconomy = false;
		}
	}
	
	if (Agent::filterMap(&_agents, WORKER_AGENT).size() <= 0)
	{
		if (_player->minerals() < 50)
			_hasEconomy = false;
	}

	if (Agent::filterMap(&_agents, ATTACKER_AGENT).size() <= 0)
		_hasAttackers = false;

	// get current game frame
	int currentFrame = Broodwar->getFrameCount();

	resetReserves();
	refreshMemoryUnits();

	for(AgentMap::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Agent* agent = (*i).second;

		agent->refresh();

		Position aPosition = agent->getUnitPosition();
		MapSector* aSector = agent->getSector();

		if ((aSector != NULL) && aSector->isInSector(aPosition)) continue;

		for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
		{
			MapSector* sector = (*i).second;

			if (sector->isInSector(aPosition))
			{
				agent->setSector(sector);
				break;
			}
		}
	}

	for(UnitMemoryMap::iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		UnitMemory* memory = (*i).second;

		if (!memory->refresh()) continue;

		Position knownPosition = memory->knownPosition;
		MapSector* knownSector = memory->knownSector;

		if ((knownSector != NULL) && knownSector->isInSector(knownPosition)) continue;

		for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
		{
			MapSector* sector = (*i).second;

			if (sector->isInSector(knownPosition))
			{
				sector->addUnitMemory(memory);
				break;
			}
		}
	}

	for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
	{
		MapSector* sector = (*i).second;

		sector->refresh();
	}
	
	for(QuadrantMap::iterator i = _quadrants.begin(); i != _quadrants.end(); i++)
	{
		Quadrant* quadrant = (*i).second;

		MapSectorSet sectors = getMapSectorsInQuadrant(quadrant);

		quadrant->dangerous = false;
		for(MapSectorSet::iterator i = sectors.begin(); i != sectors.end(); i++)
		{
			MapSector* sector = (*i);

			if (sector->getEnemyValue() > 0)
			{
				quadrant->dangerous = true;
				break;
			}
		}
	}
}

void Environment::generateValidPositions()
{
	_validPositions.clear();
	_validPositions = getValidPositions();
}

TilePosition Environment::getBestPosition(UnitType type, Position idealPos)
{
	TilePosition bestPosition = TilePositions::None;

	//
	//Special case for refineries. We which to return positions that fall in geysers only.
	//
	if (type.isRefinery())
	{
		//UnitSet geysers = _baseLocation->getGeysers();
		UnitSet geysers = Broodwar->getGeysers();

		for (UnitSet::iterator i = geysers.begin(); i != geysers.end(); i++)
		{
			Unit* geyser = (*i);

			if ((geyser->getDistance(idealPos) < MAX_RESOURCE_RANGE) && (geyser->getType() == UnitTypes::Resource_Vespene_Geyser))
			{
				bestPosition = geyser->getTilePosition();
				break;
			}
		}

		return bestPosition;
	}

	//All other types
	return Environment::getBestPosition(TileBox(type), idealPos, type);
}

TilePosition Environment::getBestPosition(TileBox box, Position idealPos, UnitType type)
{
	TilePosition bestPosition = TilePositions::None;

	int width = box.tileWidth;
	int height = box.tileHeight;

	//Evaluate each tile position in _validPositions
	double bestValue = 0.0;
	for (TilePositionMap::iterator i = _validPositions.begin(); i != _validPositions.end(); i++)
	{
		TilePosition tPos = (*i).second;

		Quadrant* quadrant = getQuadrant(tPos);
		if (quadrant->dangerous) continue;

		if (!tPos.hasPath(TilePosition(idealPos))) continue;

		if ((type != UnitTypes::None) && type.isBuilding() && !Broodwar->canBuildHere(NULL, tPos, type, true)) continue;

		//Check first if there's enough space to hold the unit
		bool canPlace = true;
		for (int w = 0; w < width; w++)
		{
			for (int h = 0; h < height; h++)
			{
				int mTPosX = tPos.x() + w;
				int mTPosY = tPos.y() + h;

				unsigned long index = ((unsigned long)mTPosX<<16) + (unsigned long)mTPosY;

				if (_validPositions.find(index) == _validPositions.end())
				{
					canPlace = false;
					break;
				}
			}

			if (!canPlace) break;
		}

		if (!canPlace) continue;

		double value = 100.0/(1.0 + idealPos.getDistance(Position(tPos)));

		if (value > bestValue)
		{
			bestPosition = tPos;
			bestValue = value;
		}
	}

	return bestPosition;
}

UnitMemorySet Environment::getMemoriesInRadius(Position pos, int radius)
{
	UnitMemorySet memoriesInRadius;

	for (UnitMemoryMap::iterator i = _unitMemories.begin(); i != _unitMemories.end(); i++)
	{
		UnitMemory* memory = (*i).second;
		double distance = pos.getDistance(memory->knownPosition);

		if (distance <= (double)radius) memoriesInRadius.insert(memory);
	}

	return memoriesInRadius;
}

TilePosition Environment::getBestCommandPosition()
{
	UnitType type = UnitTypes::Terran_Command_Center;
	TilePosition bestPosition = TilePositions::None;
	TileBox box(type);

	TilePosition startPos = _player->getStartLocation();

	AgentMap commands = getFilteredAgents(BASE_AGENT);

	int width = box.tileWidth;
	int height = box.tileHeight;

	//Evaluate each tile position in _validPositions
	double bestValue = 0.0;
	double closestDistance = MAX_POS_DISTANCE;
	for (TilePositionMap::iterator i = _validPositions.begin(); i != _validPositions.end(); i++)
	{
		TilePosition tPos = (*i).second;

		if (!Broodwar->canBuildHere(NULL, tPos, type, true)) continue;

		Quadrant* quadrant = getQuadrant(tPos);
		if (quadrant->dangerous) continue;
		
		double distanceFromStart = quadrant->center.getDistance(startPos)*32;

		if (closestDistance < distanceFromStart) continue;

		bool commandTooClose = false;

		//Don't put it too close to planned command centers
		for (PlannedBuildingSet::iterator i = _plannedBuildings.begin(); i != _plannedBuildings.end(); i++)
		{
			BuildingSpaceInfo* building = (*i);

			if (building->type != UnitTypes::Terran_Command_Center) continue;

			TilePosition position = building->tilePosition;

			double distance = tPos.getDistance(position)*32;
			if (distance <= MAX_RESOURCE_RANGE*1.2)
			{
				commandTooClose = true;
				break;
			}
		}

		//Don't put it too close to existing command centers
		for(AgentMap::iterator i = commands.begin(); i != commands.end(); i++)
		{
			Agent* command = (*i).second;
			TilePosition position = TilePosition(command->getUnitPosition());

			double distance = tPos.getDistance(position)*32;
			if (distance <= MAX_RESOURCE_RANGE*1.2)
			{
				commandTooClose = true;
				break;
			}
		}

		if (commandTooClose) continue;

		double resourceScore = 0;
		for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
		{
			MapSector* sector = (*i).second;
			TilePosition centerPos = sector->getCenterPosition();

			double distance = tPos.getDistance(centerPos)*32;

			if (distance > (MAX_RESOURCE_RANGE/2.50)) continue;

			resourceScore += sector->getResourceValue()/(1.0 + distance);
		}

		if (resourceScore <= 0) continue;

		//if (!tPos.hasPath(TilePosition(idealPos))) continue;

		//Check first if there's enough space to hold the unit
		bool canPlace = true;
		for (int w = 0; w < width; w++)
		{
			for (int h = 0; h < height; h++)
			{
				int mTPosX = tPos.x() + w;
				int mTPosY = tPos.y() + h;

				unsigned long index = ((unsigned long)mTPosX<<16) + (unsigned long)mTPosY;

				if (_validPositions.find(index) == _validPositions.end())
				{
					canPlace = false;
					break;
				}
			}

			if (!canPlace) break;
		}

		if (!canPlace) continue;

		double value = (double)resourceScore;

		if (value > bestValue)
		{
			bestPosition = tPos;
			bestValue = value;
			closestDistance = distanceFromStart;
		}
	}

	return bestPosition;
}

Position Environment::getBestOpenSpace(Position idealPos)
{
	Position bestPosition = Positions::None;
	double closestDistance = MAX_POS_DISTANCE;
	int minBuildingCount = 100;
	double mostWalkablePercent = 0.0;

	for(MapSectorMap::iterator i = _mapSectors.begin(); i != _mapSectors.end(); i++)
	{
		MapSector* sector = (*i).second;

		Position centerPos(sector->getCenterPosition());
		double distance = idealPos.getDistance(centerPos);
		int buildingCount = sector->getBuildingCount();

		if (!idealPos.hasPath(centerPos)) continue;

		double walkablePercent = sector->getWalkablePercentage();
		walkablePercent = min(walkablePercent, 0.90);

		if (buildingCount > minBuildingCount) continue;
		if ((buildingCount == minBuildingCount) && (mostWalkablePercent > walkablePercent)) continue;
		if ((buildingCount == minBuildingCount) && (mostWalkablePercent == walkablePercent) && (distance >= closestDistance)) continue;

		mostWalkablePercent = walkablePercent;

		bestPosition = centerPos;
		closestDistance = distance;
		minBuildingCount = buildingCount;
	}

	return bestPosition;
}

//bool Environment::areTilesConnected(TilePosition source, TilePosition destination)
//{
//
//}

void Environment::reserveMinerals(int quantity)
{
	_reservedMinerals += quantity;
}

void Environment::reserveGas(int quantity)
{
	_reservedGas += quantity;
}

void Environment::reserveSupply(int quantity)
{
	_reservedSupply += quantity;
}

int Environment::getReservedMinerals()
{
	return _reservedMinerals;
}

int Environment::getReservedGas()
{
	return _reservedGas;
}

int Environment::getReservedSupply()
{
	return _reservedSupply;
}

void Environment::resetReserves()
{
	_reservedMinerals = 0;
	_reservedGas = 0;
	_reservedSupply = 0;
}

void Environment::incRequiredMinerals(int quantity)
{
	_requiredMinerals += quantity;

	if (_requiredMinerals < 0) _requiredMinerals = 0;
}

void Environment::incRequiredGas(int quantity)
{
	_requiredGas += quantity;

	if (_requiredGas < 0) _requiredGas = 0;
}

void Environment::incRequiredSupply(int quantity)
{
	_requiredSupply += quantity;

	if (_requiredSupply < 0) _requiredSupply = 0;
}

void Environment::decRequiredMinerals(int quantity)
{
	_requiredMinerals -= quantity;

	if (_requiredMinerals < 0) _requiredMinerals = 0;
}

void Environment::decRequiredGas(int quantity)
{
	_requiredGas -= quantity;

	if (_requiredGas < 0) _requiredGas = 0;
}

void Environment::decRequiredSupply(int quantity)
{
	_requiredSupply -= quantity;

	if (_requiredSupply < 0) _requiredSupply = 0;
}

int Environment::getRequiredMinerals()
{
	return _requiredMinerals;
}

int Environment::getRequiredGas()
{
	return _requiredGas;
}

int Environment::getRequiredSupply()
{
	return _requiredSupply;
}

int Environment::getCurrentMinerals()
{
	return _player->minerals();
}

int Environment::getCurrentGas()
{
	return _player->gas();
}

int Environment::getCurrentSupply()
{
	return _player->supplyTotal() - _player->supplyUsed();
}

int Environment::getTotalSupply()
{
	return _player->supplyTotal();
}

bool Environment::isEnemyUnit(Unit* unit)
{
	return unit->getPlayer()->isEnemy(_player);
}
	
void Environment::addTaskTarget(Unit* unit, TaskType type)
{
	if (_taskTargets.find(unit) == _taskTargets.end())
	{
		_taskTargets[unit] = TaskTypeSet();
	}

	_taskTargets[unit].insert(type);
}

void Environment::removeTaskTargets(Unit* unit)
{
	_taskTargets.erase(unit);
}

void Environment::removeTaskTarget(Unit* unit, TaskType type)
{
	if (_taskTargets.find(unit) == _taskTargets.end()) return;

	_taskTargets[unit].erase(type);
}

void Environment::incUnitTypeDeaths(UnitType type)
{
	int count = 1;

	if (_unitTypeDeaths.find(type) != _unitTypeDeaths.end())
	{
		count += _unitTypeDeaths[type];
	}

	_unitTypeDeaths[type] = count;
	_totalDeaths++;
}

int Environment::getUnitTypeDeaths(UnitType type)
{
	if (_unitTypeDeaths.find(type) == _unitTypeDeaths.end()) return 0;

	return _unitTypeDeaths[type];
}

int Environment::getTotalDeaths()
{
	return _totalDeaths;
}

bool Environment::isTaskTarget(Unit* unit, TaskType type)
{
	if (_taskTargets.find(unit) == _taskTargets.end()) return false;

	TaskTypeSet taskTypes = _taskTargets[unit];

	return (taskTypes.find(type) != taskTypes.end());
}

void Environment::addPlannedBuilding(BuildingSpaceInfo* pBuilding)
{
	_plannedBuildings.insert(pBuilding);
}

void Environment::removePlannedBuilding(BuildingSpaceInfo* pBuilding)
{
	_plannedBuildings.erase(pBuilding);
}

PlannedBuildingSet* Environment::getPlannedBuildings()
{
	return &_plannedBuildings;
}

TilePositionMap Environment::getValidPositions()
{
	TilePositionMap validPositions;

	int mapWidth = Broodwar->mapWidth();
	int mapHeight = Broodwar->mapHeight();

	for (int j = 0; j < mapHeight; j++)
	{
		for (int i = 0; i < mapWidth; i++)
		{
			TilePosition tPos = TilePosition(i, j);
			Position pos = Position(tPos);

			if (!Broodwar->isExplored(i, j)) continue;

			if (!Broodwar->isWalkable(i*4, j*4)) continue;

			if (Broodwar->getUnitsOnTile(i, j).size() > 0) continue;

			unsigned long index = ((unsigned long)i<<16) + (unsigned long)j;
			validPositions[index] = tPos;
		}
	}

	//Remove Tile Positions occupied by other buildings
	UnitSet allUnits = Broodwar->getAllUnits();
	for (UnitSet::iterator i = allUnits.begin(); i != allUnits.end(); i++)
	{
		Unit* unit = (*i);
		UnitType type = unit->getType();

		TilePosition tPos = unit->getTilePosition();
		Position pos = Position(tPos);

		TileBox box(type);

		for (int j = 0; j < box.tileHeight; j++)
		{
			for (int i = 0; i < box.tileWidth; i++)
			{
				unsigned long index = ((unsigned long)(tPos.x() + i)<<16) + (unsigned long)(tPos.y() + j);
				validPositions.erase(index);
			}
		}
	}

	//Remove Tile Positions occupied by planned buildings
	for (PlannedBuildingSet::iterator i = _plannedBuildings.begin(); i != _plannedBuildings.end(); i++)
	{
		BuildingSpaceInfo* building = (*i);

		TilePosition tPos = building->tilePosition;
		Position pos = Position(tPos);
		
		TileBox box(building->type);

		for (int j = 0; j < box.tileHeight; j++)
		{
			for (int i = 0; i < box.tileWidth; i++)
			{
				unsigned long index = ((unsigned long)(tPos.x() + i)<<16) + (unsigned long)(tPos.y() + j);
				validPositions.erase(index);
			}
		}
	}

	//Remove Tile Positions that lie between a resource depots and resource containers
	AgentMap bases = getFilteredAgents(BASE_AGENT);
	for (AgentMap::iterator i = bases.begin(); i != bases.end(); i++)
	{
		BaseAgent* base = (BaseAgent*)(*i).second;

		TilePosition tPos = base->getUnit()->getTilePosition();

		int leftX = tPos.x();
		int rightX = tPos.x() + 4;
		int topY = tPos.y();
		int bottomY = tPos.y() + 3;

		UnitSet* patches = base->getMineralPatches();
		for (UnitSet::iterator j = patches->begin(); j != patches->end(); j++)
		{
			Unit* patch = (*j);

			TilePosition pTPos = patch->getTilePosition();

			int pLeftX = pTPos.x();
			int pRightX = pTPos.x() + 2;
			int pTopY = pTPos.y();
			int pBottomY = pTPos.y() + 1;

			pLeftX = (pLeftX < leftX) ? pLeftX : leftX;
			pRightX = (pRightX > rightX) ? pRightX : rightX;
			pTopY = (pTopY < topY) ? pTopY : topY;
			pBottomY = (pBottomY > bottomY) ? pBottomY : bottomY;

			for (int j = pTopY; j < pBottomY; j++)
			{
				for (int i = pLeftX; i < pRightX; i++)
				{
					unsigned long index = ((unsigned long)i<<16) + (unsigned long)j;
					validPositions.erase(index);
				}
			}
		}

		UnitSet* geysers = base->getGeysers();
		for (UnitSet::iterator j = geysers->begin(); j != geysers->end(); j++)
		{
			Unit* geyser = (*j);

			TilePosition pTPos = geyser->getTilePosition();

			int pLeftX = pTPos.x();
			int pRightX = pTPos.x() + 2;
			int pTopY = pTPos.y();
			int pBottomY = pTPos.y() + 1;

			pLeftX = (pLeftX < leftX) ? pLeftX : leftX;
			pRightX = (pRightX > rightX) ? pRightX : rightX;
			pTopY = (pTopY < topY) ? pTopY : topY;
			pBottomY = (pBottomY > bottomY) ? pBottomY : bottomY;

			for (int j = pTopY; j < pBottomY; j++)
			{
				for (int i = pLeftX; i < pRightX; i++)
				{
					unsigned long index = ((unsigned long)i<<16) + (unsigned long)j;
					validPositions.erase(index);
				}
			}
		}

		UnitSet* refineries = base->getRefineries();
		for (UnitSet::iterator j = refineries->begin(); j != refineries->end(); j++)
		{
			Unit* refinery = (*j);

			TilePosition pTPos = refinery->getTilePosition();

			int pLeftX = pTPos.x();
			int pRightX = pTPos.x() + 2;
			int pTopY = pTPos.y();
			int pBottomY = pTPos.y() + 1;

			pLeftX = (pLeftX < leftX) ? pLeftX : leftX;
			pRightX = (pRightX > rightX) ? pRightX : rightX;
			pTopY = (pTopY < topY) ? pTopY : topY;
			pBottomY = (pBottomY > bottomY) ? pBottomY : bottomY;

			for (int j = pTopY; j < pBottomY; j++)
			{
				for (int i = pLeftX; i < pRightX; i++)
				{
					unsigned long index = ((unsigned long)i<<16) + (unsigned long)j;
					validPositions.erase(index);
				}
			}
		}
	}

	return validPositions;
}

void Environment::incRequiredWorkers(int count)
{
	_requiredWorkers += count;

	if (_requiredWorkers < 0) _requiredWorkers = 0;
}

void Environment::decRequiredWorkers(int count)
{
	_requiredWorkers -= count;

	if (_requiredWorkers < 0) _requiredWorkers = 0;
}

int Environment::getRequiredWorkers()
{
	return _requiredWorkers;
}

int Environment::calculateNeededExtraWorkers()
{
	AgentMap workerAgents = getFilteredAgents(WORKER_AGENT);

	int neededWorkers = _requiredWorkers - workerAgents.size();

	return max(neededWorkers, 0);
}

void Environment::incRequiredMineralWorkers(int count)
{
	_requiredMineralWorkers += count;
	incRequiredWorkers(count);

	if (_requiredMineralWorkers < 0) _requiredMineralWorkers = 0;
}

void Environment::decRequiredMineralWorkers(int count)
{
	_requiredMineralWorkers -= count;
	decRequiredWorkers(count);

	if (_requiredMineralWorkers < 0) _requiredMineralWorkers = 0;
}

int Environment::getRequiredMineralWorkers()
{
	return _requiredMineralWorkers;
}

void Environment::incRequiredGasWorkers(int count)
{
	_requiredGasWorkers += count;
	incRequiredWorkers(count);

	if (_requiredGasWorkers < 0) _requiredGasWorkers = 0;
}

void Environment::decRequiredGasWorkers(int count)
{
	_requiredGasWorkers -= count;
	decRequiredWorkers(count);

	if (_requiredGasWorkers < 0) _requiredGasWorkers = 0;
}

int Environment::getRequiredGasWorkers()
{
	return _requiredGasWorkers;
}

TaskSet Environment::getFilteredTasks(TaskType type)
{
	return Task::filterSet(&_tasks, type);
}

TaskSet Environment::getFilteredTasks(TaskType type, int maxCreationFrame)
{
	return Task::filterSet(&_tasks, type, maxCreationFrame);
}

TaskSet Environment::getFilteredTasksByPriority(TaskType type, double priority)
{
	return Task::filterSetByPriority(&_tasks, type, priority);
}

AgentMap Environment::getFilteredAgents(AgentType type, bool finished)
{
	return Agent::filterMap(&_agents, type, finished);
}

AgentMap Environment::getFilteredAgents(UnitType type, bool finished)
{
	return Agent::filterMap(&_agents, type, finished);
}

AgentMap Environment::getFilteredAgents(TaskType type, bool finished)
{
	return Agent::filterMap(&_agents, type, finished);
}

Agent* Environment::getClosestAgent(AgentType type, Position pos)
{
	return Agent::closestAgent(&_agents, type, pos);
}

Agent* Environment::getClosestAgent(UnitType type, Position pos)
{
	return Agent::closestAgent(&_agents, type, pos);
}

Agent* Environment::getClosestAgent(TaskType type, Position pos)
{
	return Agent::closestAgent(&_agents, type, pos);
}

AgentSet Environment::getAgentsInRadius(Position pos, int radius)
{
	AgentSet agentsInRadius;
	double dRadius = radius;

	for(AgentMap::iterator i = _agents.begin(); i != _agents.end(); i++)
	{
		Agent* agent = (*i).second;
		Position agentPos = agent->getUnitPosition();

		if (agentPos.getDistance(pos) < dRadius)
		{
			agentsInRadius.insert(agent);
		}
	}

	return agentsInRadius;
}

#define ENERGY_FACTOR		2;
#define INVISIBLE_FACTOR	100;
#define ATTACKER_FACTOR		1000;
#define HEALER_FACTOR		10000;
#define BUNKER_FACTOR		10000;

bool Environment::isUnitCombatValuable(Unit* unit)
{
	UnitType type = unit->getType();

	//if (type == UnitTypes::Zerg_Larva) return false;
	//if (type == UnitTypes::Zerg_Egg) return false;

	if (unit->getEnergy() > 0) return true;

	if (type.canAttack()) return true;
	//if (type == UnitTypes::Terran_Medic) return true;

	if (type.spaceProvided() > 0)
	{
		if (unit->getLoadedUnits().size() > 0) return true;
	}

	return false;
}

double Environment::getUnitCombatValue(Unit* unit)
{
	double value = 100.0;
	UnitType type = unit->getType();

	if (type == UnitTypes::Zerg_Larva) return 0.0;
	if (type == UnitTypes::Zerg_Egg) return 0.0;

	int health = unit->getHitPoints() + unit->getShields();
	int energy = unit->getEnergy();

	bool invisible = unit->isCloaked() || unit->isBurrowed();

	bool canAttack = type.canAttack();
	bool canHeal = (type == UnitTypes::Terran_Medic);
	bool isBunker = (type == UnitTypes::Terran_Bunker);

	bool carriesUnits = false;
	if (type.spaceProvided() > 0)
	{
		UnitSet carriedUnits = unit->getLoadedUnits();

		double extraValue = 0;
		for (UnitSet::iterator i = carriedUnits.begin(); i != carriedUnits.end(); i++)
		{
			extraValue += Environment::getUnitCombatValue(*i);
			carriesUnits = true;
		}

		value += extraValue;
	}

	value += energy*ENERGY_FACTOR;

	if (invisible)	value *= INVISIBLE_FACTOR;
	if (canAttack)	value *= ATTACKER_FACTOR;
	if (canHeal)	value *= HEALER_FACTOR;
	if (isBunker)	value *= BUNKER_FACTOR;

	if (type.isWorker()) value *= (unit->isCarryingGas() || unit->isCarryingMinerals()) ? 5.0 : 0.2;

	if (!carriesUnits) value /= (double)(1.0 + health);

	return value;
}

double Environment::getUnitAirAttackValue(UnitType type)
{
	WeaponType wType = type.airWeapon();

	if (wType == WeaponTypes::None) return 0.0;

	return (double)wType.damageAmount()*(1.0 + ATTACK_RANGE_FACTOR*(double)wType.maxRange());
}

double Environment::getUnitGroundAttackValue(UnitType type)
{
	WeaponType wType = type.groundWeapon();

	if (wType == WeaponTypes::None) return 0.0;

	return (double)wType.damageAmount()*(1.0 + ATTACK_RANGE_FACTOR*(double)wType.maxRange());
}

double Environment::getUnitTypeValue(UnitType type)
{
	double value = 1;

	if (type.gasPrice() > 0) value += type.gasPrice()*5;
	if (type.mineralPrice() > 0) value += type.mineralPrice();

	if (type.isResourceDepot()) value *= 200;
	if (type.isRefinery()) value *= 20;
	if (type.isBuilding()) value *= 5;
	if (type.isWorker()) value *= 20;

	return value;
}
double Environment::getUnitHealthValue(Unit* unit)
{
	UnitType type = unit->getType();

	double value = 1.0;
	if (type.maxHitPoints() > 0)
	{
		value = 10.0 - 9.0*(unit->getHitPoints() + unit->getShields())/(double)(type.maxHitPoints() + type.maxShields());
	}

	return value;
}

double Environment::getUnitHealthValue(UnitMemory* memory)
{
	UnitType type = memory->type;

	double value = 1.0;
	if (type.maxHitPoints() > 0)
	{
		value = 10.0 - 9.0*(memory->knownHitPoints + memory->knownShields)/(double)(type.maxHitPoints() + type.maxShields());
	}

	return value;
}

UnitSet Environment::getThreatenedUnits(Unit* enemyUnit)
{
	UnitSet threatenedUnits;

	UnitType eType = enemyUnit->getType();

	UnitSet nearUnits = enemyUnit->getUnitsInRadius(MAX_THREAT_RANGE);

	for(UnitSet::iterator i = nearUnits.begin(); i != nearUnits.end(); i++)
	{
		Unit* nearUnit = (*i);
		Player* unitPlayer = nearUnit->getPlayer();

		if ((unitPlayer == _player) || (unitPlayer->isAlly(_player)))
		{
			threatenedUnits.insert(nearUnit);
		}
	}

	return threatenedUnits;
}

bool Environment::hasEconomy()
{
	return _hasEconomy;
}

bool Environment::hasAttackers()
{
	return _hasAttackers;
}

//////////////////////////////////////////////////////////////////

TileBox::TileBox(UnitType type)
{
	if (!type.isBuilding())
	{
		tileWidth = 1;
		tileHeight = 1;

		return;
	}

	tileWidth = type.tileWidth();
	tileHeight = type.tileHeight();
	
	tileHeight += 2;
	tileWidth += 2;

	if (type == UnitTypes::Terran_Command_Center)
	{
		tileWidth += 2;
	}
	else if (type == UnitTypes::Terran_Starport)
	{
		tileWidth += 2;
	}
	else if (type == UnitTypes::Terran_Barracks)
	{
	}
	else if (type == UnitTypes::Terran_Factory)
	{
		tileWidth += 2;
		tileHeight += 1;
	}
	else if (type == UnitTypes::Terran_Science_Facility)
	{
		tileWidth += 2;
	}
}

TileBox::TileBox(int width, int height)
{
	tileWidth = width;
	tileHeight = height;
}

//////////////////////////////////////////////////////////////////

UnitMemory::UnitMemory(Unit* u) : unit(u)
{
	knownSector = NULL;
	refresh();
}

bool UnitMemory::refresh()
{
	if (!unit->isVisible()) return false;
	//if (!unit->isDetected()) return false;

	Position pos = unit->getPosition();

	if ((pos == Positions::Invalid) ||
		(pos == Positions::None) ||
		(pos == Positions::Unknown))
		return false;

	type = unit->getType();
	knownHitPoints = unit->getHitPoints();
	knownShields = unit->getShields();
	knownResources = unit->getResources();
	knownPosition = pos;
	wasLifted = unit->isLifted();
	wasDetected = unit->isDetected();
	knownPlayer = unit->getPlayer();
	wasEnemy = knownPlayer->isEnemy(Broodwar->self());

	return true;
}

