#include "Task.h"
#include "AssaultSectorTask.h"
#include "AttackUnitTask.h"
#include "ProtectUnitTask.h"
#include "ExploreSectorTask.h"
#include "ProtectUnitTask.h"
#include "MapSector.h"
#include "Environment.h"
#include "Agent.h"
#include "BaseAgent.h"
#include "WorkerAgent.h"
#include "BuildingAgent.h"
#include "ExplorerAgent.h"
#include "AttackerAgent.h"

Agent* Agent::createAgent(Unit *unit, Environment* env)
{
	Agent* agent;
	UnitType type = unit->getType();

	if (type.isResourceDepot())
	{
		agent = new BaseAgent(unit, env);
	}
	else if (type.isBuilding())
	{
		agent = new BuildingAgent(unit, env);
	}
	else if (type.isWorker())
	{
		agent = new WorkerAgent(unit, env);
	}
	else if (type.canAttack())
	{
		agent = new AttackerAgent(unit, env);
	}
	else if ((type != UnitTypes::Terran_Dropship) &&
		(type != UnitTypes::Terran_Medic))
	{
		agent = new ExplorerAgent(unit, env);
	}
	else
	{
		agent = new Agent(unit, env);
	}

	return agent;
}

AgentMap Agent::filterMap(AgentMap* agents, AgentType type, bool finished)
{
	AgentMap filteredMap;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (!agent->isType(type)) continue;

		if (finished && !agent->isFinished()) continue;
		
		filteredMap[(*i).first] = agent;
	}

	return filteredMap;
}

AgentMap Agent::filterMap(AgentMap* agents, UnitType type, bool finished)
{
	AgentMap filteredMap;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (agent->getUnitType() != type) continue;

		if (finished && !agent->isFinished()) continue;
		
		filteredMap[(*i).first] = agent;
	}

	return filteredMap;
}

AgentMap Agent::filterMap(AgentMap* agents, TaskType type, bool finished)
{
	AgentMap filteredMap;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (!agent->isCurrentTaskType(type)) continue;

		if (finished && !agent->isFinished()) continue;
		
		filteredMap[(*i).first] = agent;
	}

	return filteredMap;
}

Agent* Agent::closestAgent(AgentMap* agents, AgentType type, Position pos)
{
	Agent* closestAgent = NULL;
	double closestDistance = MAX_POS_DISTANCE;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (!agent->isType(type)) continue;

		if (!agent->isFinished()) continue;
		
		double distance = agent->getUnit()->getDistance(pos);

		if (distance >= closestDistance) continue;

		closestDistance = distance;
		closestAgent = agent;
	}

	return closestAgent;
}

Agent* Agent::closestAgent(AgentMap* agents, UnitType type, Position pos)
{
	Agent* closestAgent = NULL;
	double closestDistance = MAX_POS_DISTANCE;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (agent->getUnitType() != type) continue;

		if (!agent->isFinished()) continue;
		
		double distance = agent->getUnit()->getDistance(pos);

		if (distance >= closestDistance) continue;

		closestDistance = distance;
		closestAgent = agent;
	}

	return closestAgent;
}

Agent* Agent::closestAgent(AgentMap* agents, TaskType type, Position pos)
{
	Agent* closestAgent = NULL;
	double closestDistance = MAX_POS_DISTANCE;

	for (AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		if (!agent->isCurrentTaskType(type)) continue;

		if (!agent->isFinished()) continue;
		
		double distance = agent->getUnit()->getDistance(pos);

		if (distance >= closestDistance) continue;

		closestDistance = distance;
		closestAgent = agent;
	}

	return closestAgent;
}

Agent::Agent(Unit* unit, Environment* env)
{
	_unit = unit;
	_env = env;

	_taskHeldFrames = 0;
	_taskActionFrame = 0;

	_types.insert(AGENT);

	_disposition = COWARD;

	_currentTask = NULL;

	_canDoSomethingElse = true;

	_currentAptitude = 0.0;

	_protectUnitTask = NULL;

	_targetPosition = Positions::None;
	_lastPosition = Positions::None;

	_sector = NULL;
}

Agent::~Agent(void)
{
}

bool Agent::isType(AgentType type)
{
	for(AgentTypeSet::const_iterator i = _types.begin(); i != _types.end(); i++)
	{
		if (*i == type) return true;
	}

	return false;
}

void Agent::initialize()
{
}

void Agent::refresh()
{
}

Unit* Agent::getUnit()
{
	return _unit;
}

Position Agent::getUnitPosition()
{
	return _unit->getPosition();
}

Position Agent::getTargetPosition()
{
	return _targetPosition;
}

void Agent::setTargetPosition(Position pos)
{
	_targetPosition = pos;
}

//void Agent::removeDeletedTask()
//{
//	_currentTask = NULL;
//	_taskHeldFrames = 0;
//}

void Agent::assignTask(Task* task)
{
	if (task != _currentTask)
	{
		if (_currentTask != NULL) _currentTask->removeAgent(this);

		_taskHeldFrames = 0;
		_taskActionFrame = 0;
		_targetPosition = Positions::None;
	}

	_currentTask = task;

	if (_currentTask != NULL)
	{
		_currentTask->addAgent(this);

		_currentAptitude = _tempValue;
	}
}

Task* Agent::getCurrentTask()
{
	return _currentTask;
}

bool Agent::isCurrentTaskType(TaskType type)
{
	if (_currentTask == NULL) return false;

	return _currentTask->isType(type);
}

int Agent::getTaskHeldTime()
{
	return _taskHeldFrames;
}

bool Agent::isFinished()
{
	return !_unit->isMorphing() && _unit->isCompleted() && (_unit->getRemainingBuildTime() == 0);
}

bool Agent::canAttackAir()
{
	WeaponType wType = _unit->getType().airWeapon();

	return (wType != WeaponTypes::None);
}

bool Agent::canAttackGround()
{
	WeaponType wType = _unit->getType().groundWeapon();

	return (wType != WeaponTypes::None);
}

int Agent::getAirWeaponDamage()
{
	WeaponType wType = _unit->getType().airWeapon();

	if (wType == WeaponTypes::None) return 0;

	return wType.damageAmount();
}

int Agent::getGroundWeaponDamage()
{
	WeaponType wType = _unit->getType().groundWeapon();

	if (wType == WeaponTypes::None) return 0;

	return wType.damageAmount();
}

int Agent::getAirWeaponRange()
{
	WeaponType wType = _unit->getType().airWeapon();

	if (wType == WeaponTypes::None) return 0;

	return wType.maxRange();
}

int Agent::getGroundWeaponRange()
{
	WeaponType wType = _unit->getType().groundWeapon();

	if (wType == WeaponTypes::None) return 0;

	return wType.maxRange();
}

ProtectUnitTask* Agent::getProtectUnitTask()
{
	return _protectUnitTask;
}

void Agent::setProtectUnitTask(ProtectUnitTask* task)
{
	_protectUnitTask = task;
}

UnitSet Agent::getEnemiesInRange(int range)
{
	UnitType type = _unit->getType();
	UnitSet units = _unit->getUnitsInRadius(range);
	UnitSet enemies;

	for (UnitSet::iterator i = units.begin(); i != units.end(); i++)
	{
		Unit* eUnit = (*i);

		if (!_env->isEnemyUnit(eUnit)) continue;

		if (!eUnit->isDetected()) continue;

		enemies.insert(eUnit);
	}

	return enemies;
}

UnitSet Agent::getEnemiesInRangeAirAttack()
{
	WeaponType wType = _unit->getType().airWeapon();

	UnitSet units = _unit->getUnitsInWeaponRange(wType);
	UnitSet enemies;

	for (UnitSet::iterator i = units.begin(); i != units.end(); i++)
	{
		Unit* eUnit = (*i);

		if (!_env->isEnemyUnit(eUnit)) continue;

		if (!eUnit->isDetected()) continue;

		enemies.insert(eUnit);
	}

	return enemies;
}

UnitSet Agent::getEnemiesInRangeGroundAttack(bool sieged)
{
	UnitType type = _unit->getType();

	if (sieged && (type == UnitTypes::Terran_Siege_Tank_Tank_Mode))
	{
		type = UnitTypes::Terran_Siege_Tank_Siege_Mode;
	}
	else if (!sieged && (type == UnitTypes::Terran_Siege_Tank_Siege_Mode))
	{
		type = UnitTypes::Terran_Siege_Tank_Tank_Mode;
	}

	WeaponType wType = type.groundWeapon();

	UnitSet units = _unit->getUnitsInWeaponRange(wType);
	UnitSet enemies;

	for (UnitSet::iterator i = units.begin(); i != units.end(); i++)
	{
		Unit* eUnit = (*i);

		if (!_env->isEnemyUnit(eUnit)) continue;

		if (!eUnit->isDetected()) continue;

		enemies.insert(eUnit);
	}

	return enemies;
}

Unit* Agent::getBestTargetInRange(int range, bool ignoreUnarmed)
{
	UnitSet targets = getEnemiesInRange(range);

	Unit* bestTarget = NULL;
	double highestValue = 0.0;
	for (UnitSet::iterator i = targets.begin(); i != targets.end(); i++)
	{
		Unit* tUnit = (*i);

		UnitType tType = tUnit->getType();

		//if (!tType.canAttack() && ignoreUnarmed) continue;

		if (!Environment::isUnitCombatValuable(tUnit) && ignoreUnarmed) continue;

		double value = Environment::getUnitCombatValue(tUnit);
		//double value = Environment::getUnitHealthValue(tUnit)/Environment::getUnitTypeValue(tType);

		if (highestValue < value)
		{
			bestTarget = tUnit;
			highestValue = value;
		}
	}

	return bestTarget;
}

Unit* Agent::getBestTargetInAttackRange(bool ignoreUnarmed, bool sieged)
{
	Unit* bestTarget = NULL;
	double highestValue = 0.0;

	if (canAttackAir())
	{
		UnitSet airTargets = getEnemiesInRangeAirAttack();

		for (UnitSet::iterator i = airTargets.begin(); i != airTargets.end(); i++)
		{
			Unit* tUnit = (*i);

			UnitType tType = tUnit->getType();

			//if (!tType.canAttack() && ignoreUnarmed) continue;

			if (!Environment::isUnitCombatValuable(tUnit) && ignoreUnarmed) continue;

			if (!(tType.isFlyer() || tUnit->isLifted())) continue;

			double value = Environment::getUnitCombatValue(tUnit);
			//double value = Environment::getUnitHealthValue(tUnit)/Environment::getUnitTypeValue(tType);

			if (highestValue < value)
			{
				bestTarget = tUnit;
				highestValue = value;
			}
		}
	}

	if (canAttackGround())
	{
		UnitSet groundTargets = getEnemiesInRangeGroundAttack(sieged);

		for (UnitSet::iterator i = groundTargets.begin(); i != groundTargets.end(); i++)
		{
			Unit* tUnit = (*i);

			UnitType tType = tUnit->getType();

			//if (!tType.canAttack() && ignoreUnarmed) continue;

			if (!Environment::isUnitCombatValuable(tUnit) && ignoreUnarmed) continue;

			if (tType.isFlyer() || tUnit->isLifted()) continue;

			double value = Environment::getUnitCombatValue(tUnit);
			//double value = Environment::getUnitHealthValue(tUnit)/Environment::getUnitTypeValue(tType);

			if (highestValue < value)
			{
				bestTarget = tUnit;
				highestValue = value;
			}
		}
	}

	return bestTarget;
}

void Agent::step()
{
	_canDoSomethingElse = true;

	executeUnitBehavior();

	if (_currentTask != NULL)
	{
		int validationFreq = _currentTask->getExecValidationFreq();

		if (_currentTask->mustSyncUp())
			_taskActionFrame = _taskHeldFrames%validationFreq;

		if (_canDoSomethingElse && _currentTask->canExecute())
		{
			if ((_taskHeldFrames%validationFreq) == _taskActionFrame)
			{
				if (!_currentTask->execute(this))
					_taskActionFrame = (_taskActionFrame + 1)%validationFreq;
			}
		}
		else
		{
			if ((_taskHeldFrames%validationFreq) == _taskActionFrame)
				_taskActionFrame = (_taskActionFrame + 1)%validationFreq;
		}

		_taskHeldFrames++;
	}

	_lastPosition = _unit->getPosition();
}

void Agent::executeUnitBehavior()
{
}

void Agent::setDisposition(AgentDisposition disposition)
{
	_disposition = disposition;
}

AgentDisposition Agent::getDisposition()
{
	return _disposition;
}

void Agent::setCurrentAptitude(double aptitude)
{
	_currentAptitude = aptitude;
}

double Agent::getCurrentAptitude()
{
	return _currentAptitude;
}

void Agent::setTempValue(double value)
{
	_tempValue = value;
}

double Agent::getTempValue()
{
	return _tempValue;
}

void Agent::setSector(MapSector* sector)
{
	if (_sector == sector) return;

	if (_sector != NULL)
	{
		_sector->removeAgent(this);
	}

	_sector = sector;

	if (sector == NULL) return;

	sector->addAgent(this);
}

MapSector* Agent::getSector()
{
	return _sector;
}

UnitType Agent::getUnitType()
{
	return _unit->getType();
}

void Agent::cleanup()
{
	assignTask(NULL);

	if (_protectUnitTask != NULL)
	{
		_protectUnitTask->setStatus(CANCELLED);
	}

	_env->incUnitTypeDeaths(_unit->getType());
	_sector->incDeathCount();
	_sector->incPossibleEnemyValue();

	setSector(NULL);
}

void Agent::displayInfo(int &row)
{
	UnitType type = _unit->getType();

	if (_unit->isSelected())
	{
		if (_currentTask != NULL)
		{
			double aptitude = _currentTask->evaluateAptitude(this);

			//Print miscellaneous agent info.
			Broodwar->drawTextScreen(
				5,10 + (row*10),"Agent '%s': task = %s, aptitude = %0.3f, held time %d",
				type.getName().c_str(),
				_currentTask->getName().c_str(),
				aptitude,
				_taskHeldFrames);
		}

		row++;
	}

	//if (isCurrentTaskType(ATTACK_UNIT_TASK))
	//{
	//	AttackUnitTask* task = (AttackUnitTask*)_currentTask;

	//	Unit* target = task->getTarget();

	//	Position uPos = _unit->getPosition();
	//	int uRadius = _unit->getType().dimensionRight();
	//	
	//	Position tPos = target->getPosition();
	//	int tRadius = target->getType().dimensionRight();

	//	//draw circle around unit
	//	Broodwar->drawCircle(CoordinateType::Map, uPos.x(), uPos.y(), uRadius, Colors::Red, false);

	//	//draw circle around target
	//	Broodwar->drawCircle(CoordinateType::Map, tPos.x(), tPos.y(), tRadius, Colors::Red, false);

	//	//draw line to target position
	//	Broodwar->drawLine(CoordinateType::Map, uPos.x(), uPos.y(), tPos.x(), tPos.y(), Colors::Red);
	//}

	if (isCurrentTaskType(PROTECT_UNIT_TASK))
	{
		ProtectUnitTask* task = (ProtectUnitTask*)_currentTask;

		Agent* pAgent = task->getProtectedAgent();
		Unit* pUnit = pAgent->getUnit();

		Position aPos = pUnit->getPosition();
		Position uPos = _unit->getPosition();
		
		int aRadius = pUnit->getType().dimensionRight();
		int uRadius = _unit->getType().dimensionRight();

		int readyRange = task->getReadyRange();
		Position gcPos = task->getGroupPosition();

		if (gcPos == Positions::None) return;
		if (readyRange == 0) return;

		Color color = Colors::Purple;

		//draw circle around group centroid
		Broodwar->drawCircle(CoordinateType::Map, gcPos.x(), gcPos.y(), readyRange, color, false);

		//draw circle around unit
		Broodwar->drawCircle(CoordinateType::Map, uPos.x(), uPos.y(), uRadius, color, false);

		//draw circle around target
		Broodwar->drawCircle(CoordinateType::Map, aPos.x(), aPos.y(), aRadius, color, false);

		//draw line from centroid to assault position
		Broodwar->drawLine(CoordinateType::Map, gcPos.x(), gcPos.y(), aPos.x(), aPos.y(), color);

		//draw line from unit to centroid
		Broodwar->drawLine(CoordinateType::Map, uPos.x(), uPos.y(), gcPos.x(), gcPos.y(), color);

		Unit* target = task->getTargetEnemy();
		
		if (target != NULL)
		{
			Position tPos = target->getPosition();
			int tRadius = target->getType().dimensionRight();

			//draw circle around target
			Broodwar->drawCircle(CoordinateType::Map, tPos.x(), tPos.y(), tRadius, Colors::Red, false);

			//draw line to target position
			Broodwar->drawLine(CoordinateType::Map, uPos.x(), uPos.y(), tPos.x(), tPos.y(), Colors::Red);
		}
	}

	if (isCurrentTaskType(ASSAULT_SECTOR_TASK))
	{
		AssaultSectorTask* task = (AssaultSectorTask*)_currentTask;
		
		if (_unit->isSelected())
		{
			//Print assault task info.
			Broodwar->drawTextScreen(
				5,10 + (row*10),"Assault readiness: %d",
				task->isAssaultReady());
			row++;

			Broodwar->drawTextScreen(
				5,10 + (row*10),"Minimum Attack Ground Strength: %0.3f, Ground Strength: %0.3f",
				task->getMinimumAttackGroundStrength(),
				task->getGroundStrength());
			row++;

			Broodwar->drawTextScreen(
				5,10 + (row*10),"Minimum Attack Air Strength: %0.3f, Air Strength: %0.3f",
				task->getMinimumAttackAirStrength(),
				task->getAirStrength());
			row++;

			Broodwar->drawTextScreen(
				5,10 + (row*10),"Minimum Defense Ground Strength: %0.3f, Enemy Ground Strength: %0.3f",
				task->getMinimumDefenseGroundStrength(),
				task->getEnemyGroundStrength());
			row++;

			Broodwar->drawTextScreen(
				5,10 + (row*10),"Minimum Defense Air Strength: %0.3f, Enemy Air Strength: %0.3f",
				task->getMinimumDefenseAirStrength(),
				task->getEnemyAirStrength());
			row++;
		}

		MapSector* sector = task->getSector();
		int readyRange = task->getReadyRange();
		Position gcPos = task->getGroupPosition();

		if (gcPos == Positions::None) return;
		if (readyRange == 0) return;

		TilePosition ctPos = sector->getCenterPosition();
		Position bPos = Position(ctPos);
		Position coPos = Position(sector->getPosition());
		Position uPos = _unit->getPosition();
		
		int uRadius = type.dimensionRight();

		Color color = Colors::Orange;

		//draw circle around group centroid
		Broodwar->drawCircle(CoordinateType::Map, gcPos.x(), gcPos.y(), readyRange, color, false);

		//draw circle around unit
		Broodwar->drawCircle(CoordinateType::Map, uPos.x(), uPos.y(), uRadius, color, false);

		//draw line from centroid to assault position
		Broodwar->drawLine(CoordinateType::Map, gcPos.x(), gcPos.y(), bPos.x(), bPos.y(), color);

		//draw line from unit to centroid
		Broodwar->drawLine(CoordinateType::Map, uPos.x(), uPos.y(), gcPos.x(), gcPos.y(), color);
		
		TilePosition tPos = sector->getPosition();
		Position aPos(tPos);

		// Mark center of sector
		if (ctPos != TilePositions::None)
		{
			Position cPos(ctPos);

			Broodwar->drawBox(
				CoordinateType::Map,
				cPos.x() - 5, cPos.y() - 5,
				cPos.x() + 5, cPos.y() + 5,
				color, true);
		}

		// Draw outline of sector
		Broodwar->drawBox(
			CoordinateType::Map,
			coPos.x(), coPos.y(),
			coPos.x() + SECTOR_LENGTH_IN_PIXELS, coPos.y() + SECTOR_LENGTH_IN_PIXELS,
			color, false);

		Unit* target = task->getTargetEnemy();
		
		if (target != NULL)
		{
			Position tPos = target->getPosition();
			int tRadius = target->getType().dimensionRight();

			//draw circle around target
			Broodwar->drawCircle(CoordinateType::Map, tPos.x(), tPos.y(), tRadius, Colors::Red, false);

			//draw line to target position
			Broodwar->drawLine(CoordinateType::Map, uPos.x(), uPos.y(), tPos.x(), tPos.y(), Colors::Red);
		}
	}
}

