#include "BaseAgent.h"
#include "Agent.h"
#include "Task.h"
#include "RecollectMinTask.h"
#include "RecollectGasTask.h"
#include "TrainWorkerTask.h"
#include "MoveCommandTask.h"
#include "BuildSupplyTask.h"
#include "BuildBarracksTask.h"
#include "BuildFactoryTask.h"
#include "BuildStarportTask.h"
#include "BuildScienceFacilityTask.h"
#include "BuildRefineryTask.h"
#include "BuildCommandTask.h"
#include "BuildEngineeringTask.h"
#include "BuildAcademyTask.h"
#include "AddonBuildComSatTask.h"
#include "AddonBuildMachineShopTask.h"
#include "AddonBuildControlTowerTask.h"
#include "TrainMarineTask.h"
#include "TrainMedicTask.h"
#include "ExploreSectorTask.h"
#include "TrainSiegeTankTask.h"
#include "TrainWraithTask.h"
#include "AssaultSectorTask.h"
#include "AttackUnitTask.h"
#include "ProtectUnitTask.h"
#include "ScanTargetTask.h"
#include "ResearchTask.h"
#include "Environment.h"
#include "MapSector.h"
#include "Adjuntant.h"

#define MINIMAP_OFFSET_X				620
#define MINIMAP_OFFSET_Y				310
#define MINISECTOR_WIDTH				5

Adjuntant::Adjuntant(Environment* env, AiLogger* logger)
{
	_env = env;

	_logger = logger;
	_timerCollector = new TimerCollector(logger);

	_willBuildSupply = false;
}

Adjuntant::~Adjuntant(void)
{
	delete _timerCollector;
}

bool compareTasks(Task* first, Task* second)
{
	if (first->getPriority() > second->getPriority()) return true;

	return false;
}

TaskList Adjuntant::createSortedTaskList()
{
	TaskSet* tasks = _env->getTasks();
	TaskList taskList(tasks->begin(), tasks->end());

	taskList.sort(compareTasks);

	return taskList;
}

TaskList Adjuntant::prioritizeTasks()
{
	TaskSet* tasks = _env->getTasks();

	for(TaskSet::const_iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = *i;
		task->calculatePriority();
	}

	TaskList taskList = createSortedTaskList();

	int row = 0;
	for(TaskList::iterator i = taskList.begin(); i != taskList.end(); i++)
	{
		Task* task = *i;
	}

	return taskList;
}

void Adjuntant::displayPlannedBuildings()
{
	PlannedBuildingSet* plannedBuildings = _env->getPlannedBuildings();
	for(PlannedBuildingSet::iterator i = plannedBuildings->begin(); i != plannedBuildings->end(); i++)
	{
		BuildingSpaceInfo* plannedBuilding = *i;

		UnitType buildingType = plannedBuilding->type;

		int width = buildingType.tileWidth();
		int height = buildingType.tileHeight();

		Position bPos = Position(plannedBuilding->tilePosition);

		//draw outline of building to be placed
		Broodwar->drawBox(CoordinateType::Map, bPos.x(), bPos.y(), bPos.x() + width*32, bPos.y() + height*32, Colors::Yellow, false);
	}
}

void Adjuntant::displayMapSectors()
{
	int currentFrame = Broodwar->getFrameCount();

	int mapWidthInSectors = Broodwar->mapWidth()/SECTOR_LENGTH;
	int mapHeightInSectors = Broodwar->mapHeight()/SECTOR_LENGTH;

	int miniMapOffsetX = MINIMAP_OFFSET_X - MINISECTOR_WIDTH*mapWidthInSectors;
	int miniMapOffsetY = MINIMAP_OFFSET_Y - MINISECTOR_WIDTH*mapHeightInSectors;

	MapSectorMap* mapSectors = _env->getMapSectors();

	//draw sectors on custom minimap
	for(MapSectorMap::iterator i = mapSectors->begin(); i != mapSectors->end(); i++)
	{
		MapSector* sector = (*i).second;

		UnitMemorySet* enemies = sector->getEnemies();

		// Mark last known position of enemies on map
		for(UnitMemorySet::iterator i = enemies->begin(); i != enemies->end(); i++)
		{
			UnitMemory* memory = *i;
			UnitType type = memory->type;
			Position uPos = memory->knownPosition;
			int uRadius = type.dimensionRight();
		
			if(Broodwar->isVisible(TilePosition(uPos)))
			{
				Unit* unit = memory->unit;

				//Ignore units which should be visible but aren't
				if (!unit->isVisible()) continue;
			}

			//draw circle around enemy
			Broodwar->drawCircle(CoordinateType::Map, uPos.x(), uPos.y(), uRadius, Colors::Red, false);
		}

		TilePosition tPos = sector->getPosition();
		Position aPos(tPos);
		Color color;

		double eneValue = sector->getEnemyValue();
		double resValue = sector->getResourceValue();
		int lastVisible = sector->getLastVisible();
		
		if (sector->getExploreSectorTask() != NULL)
		{
			color = Colors::Yellow;
		}
		else if (lastVisible + SECTOR_FRAMES_PER_REFRESH >= currentFrame)
		{
			color = Colors::Green;
		}
		else if (lastVisible < 1)
		{
			color = Colors::Black;
		}
		else if (eneValue > 0.0)
		{
			color = Colors::Orange;
		}
		else if (resValue > 0.0)
		{
			color = Colors::Teal;
		}
		else
		{
			color = Colors::Grey;
		}

		double miniRatio = MINISECTOR_WIDTH/(double)SECTOR_LENGTH;
		int miniX = (int)(tPos.x()*miniRatio);
		int miniY = (int)(tPos.y()*miniRatio);

		Broodwar->drawBox(
			CoordinateType::Screen, 
			miniMapOffsetX + miniX, miniMapOffsetY + miniY,
			miniMapOffsetX + miniX + MINISECTOR_WIDTH, miniMapOffsetY + miniY + MINISECTOR_WIDTH, 
			color, true);

		// Draw sectors on gamemap
		TilePosition ctPos = sector->getCenterPosition();
		Position coPos = Position(sector->getPosition());

		// Mark center of sector
		if (ctPos != TilePositions::None)
		{
			Position cPos(ctPos);

			Broodwar->drawBox(
				CoordinateType::Map,
				cPos.x() - 5, cPos.y() - 5,
				cPos.x() + 5, cPos.y() + 5,
				color, true);
		}

		// Draw outline of sector
		Broodwar->drawBox(
			CoordinateType::Map,
			coPos.x(), coPos.y(),
			coPos.x() + SECTOR_LENGTH_IN_PIXELS, coPos.y() + SECTOR_LENGTH_IN_PIXELS,
			color, false);

		// Write sector value
		Broodwar->drawText(
			CoordinateType::Map,
			coPos.x(), coPos.y(),
			"Resource Value: %.03f", sector->getResourceValue());
		Broodwar->drawText(
			CoordinateType::Map,
			coPos.x(), coPos.y() + 10,
			"Enemy Value: %.03f", sector->getEnemyValue());
	}

	//draw custom minimap outline
	Broodwar->drawBox(
		CoordinateType::Screen, 
		miniMapOffsetX, miniMapOffsetY,
		MINIMAP_OFFSET_X + MINISECTOR_WIDTH, MINIMAP_OFFSET_Y + MINISECTOR_WIDTH, 
		Colors::Red, false);
}

void Adjuntant::displayTaskInfo(int &row, Agent* agent)
{
	if (agent != NULL) row++;

	TaskList tasks = createSortedTaskList();
	for(TaskList::iterator i = tasks.begin(); i != tasks.end(); i++)
	{
		Task* task = *i;

		if (task->isType(ASSAULT_SECTOR_TASK)) continue;
		if (task->isType(PROTECT_UNIT_TASK)) continue;
		if (task->isType(ATTACK_UNIT_TASK)) continue;
		if (task->isType(RECOLLECT_TASK)) continue;

		task->displayInfo(row, agent);

		// Don't display more than 20 rows
		if (row >= 20) break;
	}
}

Agent* Adjuntant::displayAgentInfo(int &row)
{
	AgentMap* agents = _env->getAgents();
	Agent* selectedAgent = NULL;

	for(AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;
		Unit* unit = agent->getUnit();

		if (selectedAgent == NULL && unit->isSelected()) selectedAgent = agent;

		agent->displayInfo(row);
	}

	return selectedAgent;
}

void Adjuntant::displayInfo()
{
	int row = 0;

	displayPlannedBuildings();

	Agent* selectedAgent = displayAgentInfo(row);

	displayTaskInfo(row, selectedAgent);

	displayMapSectors();
}

void Adjuntant::assignTasksToAgents(TaskList* tasks)
{
	AgentMap* agents = _env->getAgents();

	for(TaskList::iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		(*i)->assignNeededAgents(agents);
	}
}

void Adjuntant::evaluateStatus(TaskList* tasks)
{
	int currentFrame = Broodwar->getFrameCount();

	for(TaskList::iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = (*i);

		//int duration = currentFrame - task->getCreationFrame();
		//
		//if (duration%task->getExecValidationFreq() == 0) task->evaluateStatus();
		task->evaluateStatus();
	}
}	

void Adjuntant::stepAgents()
{
	AgentMap* agents = _env->getAgents();

	for(AgentMap::iterator i = agents->begin(); i != agents->end(); i++)
	{
		Agent* agent = (*i).second;

		agent->step();
	}
}

void Adjuntant::generateNecessaryTasks()
{
	Timer* timer;

	timer = _timerCollector->StartTimer("Generate TrainWorkerTask");
	TrainWorkerTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate RecollectMinTask");
	RecollectMinTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate RecollectGasTask");
	RecollectGasTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildTask");
	BuildTask::generateUnfinishedBuildTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate RecollectMinTask");
	BuildRefineryTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildSupplyTask");
	BuildSupplyTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildBarracksTask");
	BuildBarracksTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate TrainMarineTask");
	TrainMarineTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate TrainMedicTask");
	TrainMedicTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate TrainSiegeTankTask");
	TrainSiegeTankTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate TrainWraithTask");
	TrainWraithTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate ExploreSectorTask");
	ExploreSectorTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate AssaultSectorTask");
	AssaultSectorTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate AttackUnitTask");
	AttackUnitTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate ProtectUnitTask");
	ProtectUnitTask::generateNecessaryTasks(_env);
	timer->check();
	
	timer = _timerCollector->StartTimer("Generate MoveCommandTask");
	MoveCommandTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildCommandTask");
	BuildCommandTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildEngineeringTask");
	BuildEngineeringTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildAcademyTask");
	BuildAcademyTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildFactoryTask");
	BuildFactoryTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildStarportTask");
	BuildStarportTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate BuildScienceFacilityTask");
	BuildScienceFacilityTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate AddonBuildComSatTask");
	AddonBuildComSatTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate AddonBuildMachineShopTask");
	AddonBuildMachineShopTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate AddonBuildControlTowerTask");
	AddonBuildControlTowerTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate ScanTargetTask");
	ScanTargetTask::generateNecessaryTasks(_env);
	timer->check();

	timer = _timerCollector->StartTimer("Generate ResearchTask");
	ResearchTask::generateNecessaryTasks(_env);
	timer->check();
}

void Adjuntant::removeFinishedTasks()
{
	TaskList finishedTasks;

	TaskSet* tasks = _env->getTasks();

	for(TaskSet::const_iterator i = tasks->begin(); i != tasks->end(); i++)
	{
		Task* task = *i;

		if ((task->getStatus() == FINISHED) ||
			(task->getStatus() == CANCELLED))
		{
			finishedTasks.push_back(task);
		}
	}

	for(TaskList::iterator i = finishedTasks.begin(); i != finishedTasks.end(); i++)
	{
		//Broodwar->printf("Removing task.");
		Task* task = *i;
		_env->removeTask(task);
	}
}

void Adjuntant::process()
{
	Timer* timer;

	if (Broodwar->getFrameCount()%ANALYSIS_FREQ == 0)
	{
		timer = _timerCollector->StartTimer("removeFinishedTasks");
		removeFinishedTasks();
		timer->check();

		timer = _timerCollector->StartTimer("generateNecessaryTasks");
		generateNecessaryTasks();
		timer->check();

		timer = _timerCollector->StartTimer("prioritizeTasks");
		TaskList tasks = prioritizeTasks();
		timer->check();

		timer = _timerCollector->StartTimer("assignTasksToAgents");
		assignTasksToAgents(&tasks);
		timer->check();
		
		timer = _timerCollector->StartTimer("evaluateStatus");
		evaluateStatus(&tasks);
		timer->check();
	}

	timer = _timerCollector->StartTimer("stepAgents");
	stepAgents();
	timer->check();
}
