#include "Commander.h"
#include "AgentManager.h"
#include "ExplorationManager.h"
#include "SpottedObject.h"
#include "SquadFileReader.h"
#include "BuildPlanner.h"
#include "WorkerAgent.h"
#include <algorithm>

Commander* Commander::instance = NULL;

Commander::Commander() {
	currentID = 1;	
	currentState = DEFEND;

	ownDeadScore = 0;
	enemyDeadScore = 0;

	lastCallFrame = Broodwar->getFrameCount();

	SquadFileReader sfr = SquadFileReader();
	squads = sfr.readSquadList();
	sortSquadList();
}

Commander::~Commander() {
	for (int i = 0; i < (int)squads.size(); i++) {
		delete squads.at(i);
	}
	delete instance;
}

Commander* Commander::getInstance() {
	if (instance == NULL) {
		instance = new Commander();
	}
	return instance;
}

bool Commander::shallEngage() {
	TilePosition closeEnemy = getClosestEnemyBuilding(Broodwar->self()->getStartLocation());
	if (closeEnemy.x() == -1) {
		//No enemy sighted. Dont launch attack.
		return false;
	}

	for (int i = 0; i < (int)squads.size(); i++) {
		if (squads.at(i)->isRequired() && !squads.at(i)->isActive()) {
			return false;
		}
	}
	return true;
}

void Commander::updateGoals() {
	TilePosition defSpot = findChokePoint();
	
	if (defSpot.x() != -1) {
		for (int i = 0; i < (int)squads.size(); i++) {
			Squad* sq = squads.at(i);
			if (sq->isDefensive() || (sq->isOffensive() && !sq->isActive()) || currentState == DEFEND) {
				squads.at(i)->setGoal(defSpot);
			}
		}
	}
}

void Commander::computeActions() {
	//Dont call too often
	int cFrame = Broodwar->getFrameCount();
	if (cFrame - lastCallFrame < 45) {
		return;
	}
	lastCallFrame = cFrame;

	if (currentState == DEFEND) {
		if (shallEngage()) {
			Broodwar->printf("Launch attack");
			forceAttack();
		}
	}

	if (currentState == DEFEND) {
		//Check if we need to attack/kite enemy workers in the base
		checkWorkersAttack(AgentManager::getInstance()->getClosestBase(Broodwar->self()->getStartLocation()));

		for (int i = 0; i < (int)squads.size(); i++) {
			if (!squads.at(i)->hasGoal() && !squads.at(i)->isExplorer()) {
				TilePosition defSpot = findChokePoint();
				if (defSpot.x() != -1) {
					squads.at(i)->setGoal(defSpot);
				}
			}
		}
	}

	if (currentState == ATTACK) {
		TilePosition closeEnemy = getClosestEnemyBuilding(Broodwar->self()->getStartLocation());
		TilePosition offSquad = findOffensiveSquadPosition(closeEnemy);
		
		for (int i = 0; i < (int)squads.size(); i++) {
			if (squads.at(i)->isActive() && (squads.at(i)->isOffensive() || squads.at(i)->isSupport())) {
				if (closeEnemy.x() >= 0) {
					if (squads.at(i)->isGround()) {
						squads.at(i)->setGoal(closeEnemy);
					}
					else if (squads.at(i)->isAir()) {
						squads.at(i)->setGoal(offSquad);
					}
				}
			}
			else {
				if (!squads.at(i)->isExplorer()) {
					TilePosition defSpot = findChokePoint();
					if (defSpot.x() != -1) {
						squads.at(i)->setGoal(defSpot);
					}
				}
			}
		}
	}
	
	//Check if there are obstacles we can remove. Needed for some
	//strange maps.
	if (Broodwar->getFrameCount() % 150 == 0) {
		checkRemovableObstacles();
	}

	//Compute Squad actions.
	for(int i = 0; i < (int)squads.size(); i++) {
		squads.at(i)->computeActions();
	}

	//Check if any own buildings is under attack, if so try to assist them
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		if (agent->isAlive() && agent->isBuilding() && agent->isUnderAttack()) {
			assistBuilding(agent);
		}
		if (agent->isAlive() && agent->isWorker() && agent->isUnderAttack()) {
			assistWorker(agent);
		}
	}

	//Attack if we have filled all supply spots
	if (currentState == DEFEND) {
		int supplyUsed = Broodwar->self()->supplyUsed() / 2;
		if (supplyUsed >= 198) {
			forceAttack();
		}
	}

	//Terran only: Check for repairs and finish unfinished buildings
	if (BuildPlanner::isTerran()) {
		//Check if there are unfinished buildings we need
		//to complete.
		checkUnfinishedBuildings();

		//Check if there are units we need to repair.
		checkRepairUnits();
	}

	//Check for units not belonging to a squad
	checkNoSquadUnits();
}

void Commander::checkNoSquadUnits() {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		
		bool notAssigned = true;
		if (!agent->isAlive()) notAssigned = false;
		if (agent->getUnitType().isWorker()) notAssigned = false;
		if (!agent->getUnit()->exists()) notAssigned = false;
		if (agent->isOfType(UnitTypes::Zerg_Overlord)) notAssigned = false;
		if (agent->getUnitType().isBuilding()) notAssigned = false;
		if (agent->getUnitType().isAddon()) notAssigned = false;
		if (agent->getSquadID() != -1) notAssigned = false;

		if (notAssigned) {
			assignUnit(agent);
		}
	}
}

void Commander::assignUnit(BaseAgent* agent) {
	//Broodwar->printf("%s (%s) is not assigned to a squad", agent->getUnitType().getName().c_str(), agent->getTypeName().c_str());

	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (sq->needUnit(agent->getUnitType())) {
			sq->addMember(agent);
			//Broodwar->printf("%s is assigned to SQ %d", agent->getUnitType().getName().c_str(), sq->getID());
			return;
		}
	}
}

TilePosition Commander::getClosestEnemyBuilding(TilePosition start) {
	Unit* closestBuilding = NULL;
	SpottedObject* closestObj = NULL;
	double bestDist = -1;

	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++) {
		if ((*i)->getType().isBuilding()) {
			double cDist = start.getDistance((*i)->getTilePosition());
			
			if (!ExplorationManager::canReach(start, (*i)->getTilePosition())) {
				//cDist = -1;
			}

			if (bestDist < 0 || cDist < bestDist) {
				bestDist = cDist;
				closestBuilding = (*i);
			}
		}
	}

	if (closestBuilding != NULL) {
		//Broodwar->printf("[TSC]: Closest enemy building is %s", closestBuilding->getType().getName().c_str());
		return closestBuilding->getTilePosition();
	}

	if (ExplorationManager::getInstance()->isActive()) {
		bestDist = -1;
		vector<SpottedObject*> units = ExplorationManager::getInstance()->getSpottedBuildings();
		for (int i = 0; i < (int)units.size(); i++) {
			SpottedObject* obj = units.at(i);
			if (obj->getType().isBuilding() && obj->isActive()) {
				double cDist = obj->getDistance(start);
				if (bestDist < 0 || cDist < bestDist) {
					bestDist = cDist;
					closestObj = obj;
				}
			}
		}

		if (closestObj != NULL) {
			//Broodwar->printf("[TSC]: Closest enemy building is %s", closestBuilding->getType().getName().c_str());
			return closestObj->getTilePosition();
		}
	}
	
	return TilePosition(-1,-1);
}

void Commander::handleCloakedEnemy(TilePosition pos, Squad* squad) {
	if (BuildPlanner::isProtoss()) {
		for (int i = 0; i < (int)squads.size(); i++) {
			if (squads.at(i)->isSupport()) {
				vector<BaseAgent*> agents = squads.at(i)->getMembers();
				for (int a = 0; a < (int)agents.size(); a++) {
					BaseAgent* agent = agents.at(i);
					if (agent->isAlive() && agent->isOfType(UnitTypes::Protoss_Observer)) {
						squads.at(i)->setGoal(pos);
						return;
					}
				}
			}
		}

		//Didnt find an Observer, retreat to closest detector (if found).
		TilePosition dPos = AgentManager::getInstance()->getClosestDetector(pos);
		if (dPos.x() >= 0) {
			squad->setGoal(dPos);
		}
	}
	if (BuildPlanner::isTerran()) {
		vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
		if ((int)agents.size() > 0) {
			bool ok = agents.at(0)->doScannerSweep(pos);
			if (ok) {
				return;
			}
		}
		//Unable to sweep area, retreat to closest detector (if found).
		TilePosition dPos = AgentManager::getInstance()->getClosestDetector(pos);
		if (dPos.x() >= 0) {
			squad->setGoal(dPos);
		}
	}
	if (BuildPlanner::isZerg()) {
		vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
		for (int i = 0; i < (int)agents.size(); i++) {
			BaseAgent* agent = agents.at(i);
			if (agent->isAlive() && agent->isOfType(UnitTypes::Zerg_Overlord) && agent->getGoal().x() == -1) {
				agent->getUnit()->rightClick(Position(pos));
			}
		}
	}
}

Squad* Commander::getSquad(int id) {
	for (int i = 0; i < (int)squads.size(); i++) {
		if (squads.at(i)->getID() == id) {
			return squads.at(i);
		}
	}
	return NULL;
}

vector<Squad*> Commander::getSquads() {
	return squads;
}

int Commander::noOffensiveSquadsWithin(TilePosition center, int maxRange) {
	int no = 0;

	for (int i = 0; i < (int)squads.size(); i++) {
		if (squads.at(i)->isActive() && squads.at(i)->isGathered()) {
			TilePosition pos = squads.at(i)->getCenter();
			if (Position(center).getDistance(Position(pos)) <= maxRange) {
				no++;
			}
		}
	}

	return no;
}

void Commander::unitDestroyed(BaseAgent* agent) {
	int squadID = agent->getSquadID();
	if (squadID != -1) {
		Squad* squad = getSquad(squadID);
		if (squad != NULL) {
			squad->removeMember(agent);
		}
	}
}

void Commander::sortSquadList() {
	sort(squads.begin(), squads.end(), SortSquadList());
}

void Commander::unitCreated(BaseAgent* agent) {
	//Sort the squad list
	sortSquadList();

	for (int i = 0; i < (int)squads.size(); i++) {
		if (squads.at(i)->addMember(agent)) {
			break;
		}
	}

	//Check if we can merge to fill a squad
	//TODO: Currently disabled to see if it works better
	/*for (int i = 0; i < (int)squads.size(); i++) {
		if (!squads.at(i)->isFull()) {
			for (int j = 0; j < (int)squads.size(); j++) {
				if (squads.at(i)->canMerge(squads.at(j))) {
					Broodwar->printf("[SQ] Merged Squads %d with %d", squads.at(i)->getID(), squads.at(j)->getID());
				}
			}
		}
	}*/
}

bool Commander::checkWorkersAttack(BaseAgent *base) {
	int noAttack = 0;

	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++) {
		if ((*i)->exists() && (*i)->getType().isWorker()) {
			double dist = (*i)->getTilePosition().getDistance(base->getUnit()->getTilePosition());
			if (dist <= 12) {
				//Enemy unit discovered. Attack with some workers.
				vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
				for (int j = 0; j < (int)agents.size(); j++) {
					BaseAgent* agent = agents.at(j);
					if (agent->isAlive() && agent->isWorker() && noAttack < 1) {
						WorkerAgent* wAgent = (WorkerAgent*) agent;
						wAgent->setState(WorkerAgent::ATTACKING);
						agent->getUnit()->attack((*i));
						noAttack++;
					}
				}
			}
		}
	}

	if (noAttack > 0) {
		return true;
	}
	return false;
}

int Commander::enemyUnitsWithinRange(TilePosition start, int range) {
	int eCnt = 0;
	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++) {
		double dist = (*i)->getDistance(Position(start));
		if (dist <= range) {
			eCnt++;
		}
		
	}
	return eCnt;
}

void Commander::checkRemovableObstacles() {
	for(set<Unit*>::iterator m = Broodwar->getMinerals().begin(); m != Broodwar->getMinerals().end(); m++) {
		if ((*m)->getResources() <= 20) {
			//Found a mineral that we can remove.
			BaseAgent* baseAgent = AgentManager::getInstance()->getClosestBase((*m)->getTilePosition());
			if (baseAgent != NULL) {
				double cDist = baseAgent->getUnit()->getDistance((*m));
				if (cDist < 1000) {
					//It is close to a base, remove it.

					//Step 1: Check if someone already is working on it
					bool assign = true;
					vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
					for (int i = 0; i < (int)agents.size(); i++) {
						BaseAgent* agent = agents.at(i);
						if (agent->isWorker()) {
							Unit* u = agent->getUnit();
							if (u->isGatheringMinerals()) {
								Unit* t = u->getTarget();
								if (t != NULL && t->getID() == u->getID()) {
									//Someone already working on it. Dont bother.
									assign = false;
								}
							}
						}
					}

					if (assign) {
						BaseAgent* worker = AgentManager::getInstance()->findClosestFreeWorker((*m)->getTilePosition());
						if (worker != NULL) {
							worker->getUnit()->rightClick((*m));
						}
					}
				}
			}
		}
	}
}

TilePosition Commander::findUnfortifiedChokePoint() {
	double bestDist = 0;
	Chokepoint* bestChoke = NULL;

	for(set<Region*>::const_iterator i=getRegions().begin();i!=getRegions().end();i++) {
		if (isOccupied((*i))) {
			for(set<Chokepoint*>::const_iterator c=(*i)->getChokepoints().begin();c!=(*i)->getChokepoints().end();c++) {
				if (isEdgeChokepoint((*c))) {
					if (!chokePointFortified(TilePosition((*c)->getCenter()))) {
						double cDist = Broodwar->self()->getStartLocation().getDistance(TilePosition((*c)->getCenter()));
						if (cDist > bestDist) {
							bestDist = cDist;
							bestChoke = (*c);
						}
					}
				}
			}
		}
	}

	TilePosition buildPos = TilePosition(-1, -1);
	if (bestChoke != NULL) {
		buildPos = TilePosition(bestChoke->getCenter());
	}
	return buildPos;
}

bool Commander::chokePointFortified(TilePosition center) {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		if (agents.at(i)->isAlive()) {
			if (agents.at(i)->isOfType(UnitTypes::Terran_Bunker) || agents.at(i)->isOfType(UnitTypes::Protoss_Photon_Cannon) || agents.at(i)->isOfType(UnitTypes::Zerg_Sunken_Colony) || agents.at(i)->isOfType(UnitTypes::Zerg_Creep_Colony) || agents.at(i)->isOfType(UnitTypes::Zerg_Spore_Colony)) {
				double dist = center.getDistance(agents.at(i)->getUnit()->getTilePosition());
				if (dist <= 20) {
					return true;
				}
			}
		}
	}
	return false;
}

double Commander::getChokepointPrio(TilePosition center) {
	TilePosition ePos = ExplorationManager::getInstance()->getClosestSpottedBuilding(center);

	if (ePos.x() >= 0) {
		double dist = ePos.getDistance(center);
		return 1000 - dist;
	}
	else {
		double dist = Broodwar->self()->getStartLocation().getDistance(center);
		return dist;
	}
}

TilePosition Commander::findChokePoint() {
	double bestPrio = -1;
	Chokepoint* bestChoke = NULL;

	for(set<Region*>::const_iterator i=getRegions().begin();i!=getRegions().end();i++) {
		if (isOccupied((*i))) {
			for(set<Chokepoint*>::const_iterator c=(*i)->getChokepoints().begin();c!=(*i)->getChokepoints().end();c++) {
				if (isEdgeChokepoint((*c))) {
					double cPrio = getChokepointPrio(TilePosition((*c)->getCenter()));
					if (cPrio > bestPrio) {
						bestPrio = cPrio;
						bestChoke = (*c);
					}
				}
			}
		}
	}

	TilePosition guardPos = Broodwar->self()->getStartLocation();
	if (bestChoke != NULL) {
		guardPos = findDefensePos(bestChoke);
	}

	return guardPos;
}

TilePosition Commander::findDefensePos(Chokepoint* choke) {
	TilePosition defPos = TilePosition(choke->getCenter());
	TilePosition chokePos = defPos;

	double size = choke->getWidth();
	if (size <= 32 * 3) {
		//Very narrow chokepoint, dont crowd it
		double bestDist = 1000;
		TilePosition basePos = Broodwar->self()->getStartLocation();

		int maxD = 8;
		int minD = 5;

		//We found a chokepoint. Now we need to find a good place to defend it.
		for (int cX = chokePos.x() - maxD; cX <= chokePos.x() + maxD; cX++) {
			for (int cY = chokePos.y() - maxD; cY <= chokePos.y() + maxD; cY++) {
				TilePosition cPos = TilePosition(cX, cY);
				if (ExplorationManager::canReach(basePos, cPos)) {
					double chokeDist = chokePos.getDistance(cPos);
					double baseDist = basePos.getDistance(cPos);

					if (chokeDist >= minD && chokeDist <= maxD) {
						if (baseDist < bestDist) {
							bestDist = baseDist;
							defPos = cPos;
						}
					}
				}
			}
		}
	}

	//Uncomment to make defenders crowd around defensive structures.
	/*UnitType defType;
	if (BuildPlanner::isZerg()) defType = UnitTypes::Zerg_Sunken_Colony;
	if (BuildPlanner::isProtoss()) defType = UnitTypes::Protoss_Photon_Cannon;
	if (BuildPlanner::isTerran()) defType = UnitTypes::Terran_Bunker;

	BaseAgent* turret = AgentManager::getInstance()->getClosestAgent(defPos, defType);
	if (turret != NULL) {
		TilePosition tPos = turret->getUnit()->getTilePosition();
		double dist = tPos.getDistance(defPos);
		if (dist <= 12) {
			defPos = tPos;
		}
	}*/

	return defPos;
}

double Commander::getDistToBase(Chokepoint* choke) {
	double bestDist = 10000;

	TilePosition center = TilePosition(choke->getCenter());
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		if (agent->isAlive() && agent->getUnitType().isResourceDepot()) {
			double cDist = center.getDistance(agent->getUnit()->getTilePosition());
			if (cDist < bestDist) {
				bestDist = cDist;
			}
		}
	}

	return bestDist;
}

bool Commander::isEdgeChokepoint(Chokepoint* choke) {
	pair<Region*,Region*> regions = choke->getRegions();
	if (isOccupied(regions.first) && isOccupied(regions.second)) {
		return false;
	}
	return true;
}

bool Commander::isOccupied(Region* region) {
	TilePosition center = TilePosition(region->getCenter());
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		if (agent->isAlive() && agent->isBuilding()) {
			double dist = agent->getUnit()->getTilePosition().getDistance(center);
			if (dist <= 10) {
				return true;
			}
		}
	}

	//Check expansion site
	TilePosition expansionSite = ExplorationManager::getInstance()->getExpansionSite();
	if (expansionSite.x() >= 0) {
		double dist = expansionSite.getDistance(center);
		if (dist <= 15) {
			return true;
		}
	}

	return false;
}

bool Commander::chokePointGuarded(TilePosition center) {
	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (sq->isDefensive() || sq->isOffensive()) {
			double dist = center.getDistance(sq->getCenter());
			if (dist <= 5) {
				return true;
			}
			dist = center.getDistance(sq->getGoal());
			if (dist <= 5) {
				return true;
			}
		}
	}

	return false;
}

bool Commander::isPowered(TilePosition pos) {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		if (agent->isOfType(UnitTypes::Protoss_Pylon)) {
			double dist = agent->getUnit()->getTilePosition().getDistance(pos);
			if (dist <= 4) {
				return true;
			}
		}
	}
	return false;
}

bool Commander::isBuildable(TilePosition pos) {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		if (agent->isOfType(UnitTypes::Protoss_Pylon)) {
			if (!agent->getUnit()->isBeingConstructed()) {
				double dist = agent->getUnit()->getTilePosition().getDistance(pos);
				if (dist <= 4) {
					return true;
				}
			}
		}
	}
	return false;
}

bool Commander::needUnit(UnitType type) {
	int prevPrio = 1000;

	for (int i = 0; i < (int)squads.size(); i++) {
		if (!squads.at(i)->isFull()) {
			if (squads.at(i)->getPriority() > prevPrio) {
				return false;
			}

			if (squads.at(i)->needUnit(type)) {
				return true;
			}
			
			prevPrio = squads.at(i)->getPriority();
		}
	}
	return false;
}

TilePosition Commander::findOffensiveSquadPosition(TilePosition closeEnemy) {
	TilePosition pos = TilePosition(-1, -1);

	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (pos.x() == -1) {
			if (sq->isActive() && sq->isOffensive() && sq->isGround() && currentState == ATTACK && sq->size() >= 2) {
				pos = sq->getCenter();
			}
		}
		if (sq->isAttacking() && sq->isOffensive()) {
			return closeEnemy;
		}
	}

	if (pos.x() == -1) {
		//No position found, wait at home base
		pos = Broodwar->self()->getStartLocation();
	}

	return pos;
}

void Commander::assistBuilding(BaseAgent* building) {
	//Find out who targets the building
	TilePosition defPos = building->getUnit()->getTilePosition();
	Unit* target = NULL;
	int bestScore = -1;

	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++) {
		if ((*i)->exists()) {
			double dist = (*i)->getDistance(Position(defPos));
			if (dist <= 13 * 32) {
				//Found a target
				if ((*i)->getType().destroyScore() > bestScore) {
					target = (*i);
					bestScore = (*i)->getType().destroyScore();
				}
			}
		}
	}

	if (target != NULL) {
		//Broodwar->printf("Assisting building: Targeting enemy %s", target->getType().getName().c_str());
		defPos = target->getTilePosition();
	}
	
	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (sq->isDefensive() || sq->isOffensive() || sq->isSupport() || (sq->isRush() && !sq->isActive())) {
			sq->setGoal(defPos);
		}
	}
}

void Commander::assistWorker(BaseAgent* worker) {
	if (worker->getSquadID() != -1) {
		//Worker is in a squad. Do nothing.
		return;
	}

	//Find out who targets the worker
	TilePosition defPos = worker->getUnit()->getTilePosition();
	Unit* target = NULL;
	int bestScore = -1;

	for(set<Unit*>::const_iterator i=Broodwar->enemy()->getUnits().begin();i!=Broodwar->enemy()->getUnits().end();i++) {
		if ((*i)->exists()) {
			double dist = (*i)->getDistance(Position(defPos));
			if (dist <= 13 * 32) {
				//Found a target
				if ((*i)->getType().destroyScore() > bestScore) {
					target = (*i);
					bestScore = (*i)->getType().destroyScore();
				}
			}
		}
	}

	if (target != NULL) {
		//Broodwar->printf("Assisting building: Targeting enemy %s", target->getType().getName().c_str());
		defPos = target->getTilePosition();
	}
	
	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (sq->isDefensive() || sq->isOffensive() || sq->isSupport() || (sq->isRush() && !sq->isActive())) {
			sq->setGoal(defPos);
		}
	}
}

void Commander::forceAttack() {
	TilePosition closeEnemy = getClosestEnemyBuilding(Broodwar->self()->getStartLocation());
	TilePosition offSquad = findOffensiveSquadPosition(closeEnemy);

	Broodwar->printf("Force attack at (%d,%d)", closeEnemy.x(), closeEnemy.y());

	if (closeEnemy.x() == -1) {
		return;
	}

	for (int i = 0; i < (int)squads.size(); i++) {
		if (squads.at(i)->isOffensive() || squads.at(i)->isSupport()) {
			TilePosition cGoal = TilePosition(-1, -1);
			
			if (squads.at(i)->isGround()) {
				//Ground squad: Move to closest base.
				cGoal = closeEnemy;
			}
			else if (squads.at(i)->isAir()) {
				//Air squad: Follow the ground squads.
				cGoal = offSquad;
			}

			if (cGoal.x() >= 0) {
				squads.at(i)->setGoal(cGoal);
				squads.at(i)->forceActive();
			}
		}
	}

	currentState = ATTACK;
}

bool Commander::checkRepairUnits() {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) 
	{
		BaseAgent* agent = agents.at(i);
		if (agent != NULL && agent->isAlive() && isImportantUnit(agent->getUnit()) && agent->isDamaged() && !agent->getUnit()->isBeingConstructed())
		{
			repair(agent);
		}
	}
	return false;
}

bool Commander::isImportantUnit(Unit* unit) {
	UnitType type = unit->getType();

	if (type.isBuilding()) return true;
	if (BaseAgent::isOfType(type, UnitTypes::Terran_Siege_Tank_Tank_Mode)) return true;
	if (BaseAgent::isOfType(type, UnitTypes::Terran_Siege_Tank_Siege_Mode)) return true;
	if (BaseAgent::isOfType(type, UnitTypes::Terran_Science_Vessel)) return true;
	if (BaseAgent::isOfType(type, UnitTypes::Terran_Battlecruiser)) return true;

	return false;
}

void Commander::repair(BaseAgent* agent) {
	
	//First we must check if someone is repairing this building
	if(AgentManager::getInstance()->isAnyAgentRepairingThisAgent(agent)) {
		return;
	}

	BaseAgent* repUnit = AgentManager::getInstance()->findClosestFreeWorker(agent->getUnit()->getTilePosition());
	if (repUnit != NULL) {
		//Broodwar->printf("Assigned SCV %d to repair %s (%d/%d)", repUnit->getUnitID(), agent->getUnitType().getName().c_str(), agent->getUnit()->getHitPoints(), agent->getUnitType().maxHitPoints());
		repUnit->assignToRepair(agent->getUnit());
	}
}

void Commander::finishBuild(BaseAgent* baseAgent) {
	//First we must check if someone is repairing this building
	if(AgentManager::getInstance()->isAnyAgentRepairingThisAgent(baseAgent))
		return;

	BaseAgent* repUnit = AgentManager::getInstance()->findClosestFreeWorker(baseAgent->getUnit()->getTilePosition());
	if (repUnit != NULL) {
		repUnit->assignToFinishBuild(baseAgent->getUnit());
	}
}

bool Commander::checkUnfinishedBuildings() {
	vector<BaseAgent*> agents = AgentManager::getInstance()->getAgents();
	for (int i = 0; i < (int)agents.size(); i++) {
		BaseAgent* agent = agents.at(i);
		// Double check so we don't get a dead agent. This could cause null pointers.
		if (agent != NULL && agent->isAlive() && agent->isBuilding() && agent->isDamaged())
		{
			Unit* builder = agent->getUnit()->getBuildUnit();
			if (builder == NULL || !builder->isConstructing()) {
				finishBuild(agent);
			}
		}
	}
	return false;
}

void Commander::printInfo() {
	Broodwar->drawTextScreen(295,0,"Attacking squads:");
	int no = 0;
	for (int i = 0; i < (int)squads.size(); i++) {
		Squad* sq = squads.at(i);
		if (sq->isRequired()) {
			Broodwar->drawTextScreen(295,no*16+16, "SQ %d: (%d/%d)", sq->getID(), sq->getSize(), sq->getTotalUnits());
			no++;
		}
	}

	Broodwar->drawTextScreen(440,32, "Own Deadscore: %d", ownDeadScore);
	Broodwar->drawTextScreen(440,48, "Enemy Deadscore: %d", enemyDeadScore);
}

void Commander::addBunkerSquad() {
	Squad* bSquad = new Squad(100 + AgentManager::getInstance()->countNoUnits(UnitTypes::Terran_Bunker), Squad::BUNKER, Squad::GROUND, "BunkerSquad", 9);
	bSquad->addSetup(UnitTypes::Terran_Marine, 4);
	squads.push_back(bSquad);
}
